//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2011, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_AST_ATTRINFO_H
#define AAP_AST_ATTRINFO_H

#include <string>
#include <vector>
#include <map>

#include <llvm/Support/CommandLine.h>
#include "sourceobj.h"

namespace aap {

  /// Creates storage for function attributes.
  class AttrInfo {
  public:
    AttrInfo (const std::string, const std::string FileName = "");
    void SetAnalyzed (bool);
    bool IsAnalyzed (void);
    void SetDefault (void);

    /// Sets GNU function attribute pure.
    void SetPure (bool);
    /// Is function GNU function attribute pure.
    bool IsPure  (void);
    /// Sets GNU function attribute const.
    void SetConst(bool);
    /// Is function GNU function attribute const.
    bool IsConst (void);

    /// Adds dependency to other function's attribute.
    void AddDependence (AttrInfo*);
    /// Get dependent function's attribute
    AttrInfo * GetDependence (unsigned int);
    /// Size of dependent attributes.
    unsigned int DependencesSize (void);
    /// Erases function attribute dependency.
    void EraseDependence (unsigned int);
    /// Clears all dependencies.
    void ClearDependences (void);
    /// Search for dependency to a function attribute.
    bool SearchDependence (AttrInfo*);

    /// Adds SourceObjRef of the function declaration.
    void AddObjRef (SourceObjRef);
    /// Gets SourceObjRef of the function declaration.
    SourceObjRef GetObjRef (unsigned int);
    /// Number of declarations.
    unsigned int ObjRefVectorSize (void);
    /// Clear the SourceObjRef vector.
    void ClearObjRefVector (void);

    /// Handles dependency of the function attributes.
    int HandleDependence (void);
    void SetDepth (int);
    int GetDepth (void);

    void AddTime (unsigned int);

    static AttrInfo* GetOrInsertAttrInfo (const std::string,
					  const std::string FileName = "");
    static AttrInfo* SearchAttr (const std::string,
				 const std::string FileName = "");
    static void InsertSourceObj (SourceObjRef, const std::string,
				 const std::string FileName = "");
    static void SetTotalTime (unsigned int);
    /// Process all function attributes and resolve dependencies.
    static void HandleAttributesDependences (void);

    static llvm::cl::opt<double> ColdTimePercentage;
    static llvm::cl::opt<double> HotTimePercentage;
    ~AttrInfo (void);
  private:
    bool pure;            // GNU function attribute pure
    bool AttrConst;       // GNU function attribute const
    bool analyzed;
    int  depth;
    bool setdefault;
    unsigned int Time;
    std::string FuncName;
    std::string FileName;
    std::vector<SourceObjRef> obj;
    std::vector<AttrInfo*> Dependences;
    static unsigned int TotalTime;
    static std::map<std::string,AttrInfo*> FunctionAttributes;
  };

} // namespace aap

#endif // AAP_AST_ATTRINFO_H

