//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <iostream>
#include <sysexits.h>

#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/XMLString.hpp>

#include "buildoptionfactory.h"
#include "buildoptions.h"
#include "utils/options.h"

using namespace std;
using namespace xercesc;
using namespace aap;

static llvm::sys::Path defaultPath("build-options.xml");

llvm::cl::opt<llvm::sys::Path, false, llvm::cl::parser<std::string> >
BuildOptionFactory::file("build-file",
                         llvm::cl::init(defaultPath),
                         llvm::cl::ValueRequired,
                         llvm::cl::value_desc("path"),
                         llvm::cl::desc("Path to build-options file."));

static llvm::cl::alias
file_short("b",
           llvm::cl::ValueRequired,
           llvm::cl::desc("alias of -build-file"),
           llvm::cl::aliasopt(BuildOptionFactory::file));

static XMLCh kEmpty[] = {'\0'};
static XMLCh kName[] = {'n','a','m','e','\0'};
static XMLCh kPath[] = {'p','a','t','h','\0'};
static XMLCh kValue[] = {'v','a','l','u','e','\0'};

/// The return value is a mapping from file names to an objecting containing the
/// build options for that file. Parsing is handled by creating a SAX parser,
/// which then calls back to the startElement and endElement methods. The format
/// of the file can be found in the BuildOptions documentation.
map<string,BuildOptions>
BuildOptionFactory::parse (const llvm::sys::Path& path)
{
    if (!path.exists()) {
        if (path == defaultPath && !file.getNumOccurrences()) {
            if (Options::verbose())
                cerr << "default build options file unavailable\n";
            return options_;
        } else {
            cerr << "invalid build file \"" << path.str() << "\"\n";
            exit(EX_NOINPUT);
        }
    }

    try {
        XMLPlatformUtils::Initialize();
        SAX2XMLReader* reader = XMLReaderFactory::createXMLReader();
        reader->setContentHandler(this);
        reader->setErrorHandler(this);
        reader->parse (path.str().c_str());
        delete reader;
        XMLPlatformUtils::Terminate();
    } catch (...) {
        cout << "Unexpected Exception" << endl;
    }

    return options_;
}

/// Finalized the set of build options associated with a file when the <file>
/// element's close tag is encountered by inserting the file-name, build options
/// pair into the options mapping. Likewise finalized the set of global options.
void
BuildOptionFactory::endElement(const XMLCh*, // uri
                               const XMLCh* localname,
                               const XMLCh*) // qname
{
    char* name = XMLString::transcode (localname);
    if (0 == (strcasecmp (name,"file"))) {
        options_.insert (make_pair (current_file_,*current_options_));
        if (current_options_ != &BuildOptions::global_)
            delete current_options_;
        current_options_ = NULL;
    } else if (0 == (strcasecmp (name, "globaloptions"))) {
        current_options_ = NULL;
    }
    XMLString::release (&name);
}

void
BuildOptionFactory::startElement(const XMLCh*, // uri,
                                 const XMLCh* localname,
                                 const XMLCh*, // qname
                                 const xercesc::Attributes& attrs)
{
    char* name = XMLString::transcode (localname);

    if (0 == (strcasecmp (name, "file"))) {
        // For a new file, create a build options object.

        const XMLCh* xml_path = attrs.getValue (kEmpty, kPath);
        char* file = XMLString::transcode (xml_path);
        current_file_ = file;
        current_options_ = new BuildOptions();
        XMLString::release (&file);

    } else if (0 == (strcasecmp (name, "option"))) {
        // Add each object to the current file's set of options.
        const XMLCh* xml_name  = attrs.getValue (kEmpty, kName);
        const XMLCh* xml_value = attrs.getValue (kEmpty, kValue);

        char* name  = XMLString::transcode (xml_name);
        char* value = XMLString::transcode (xml_value);

        if (current_options_ == NULL) {
            cout << "build option without associated file" << endl;
            exit (19);
        }

        current_options_->addOption (name,value);

        XMLString::release (&name);
        XMLString::release (&value);

    } else if (0 == strcasecmp (name, "globaloptions")) {
        current_options_ = &BuildOptions::global_;

    } else if (0 == strcasecmp (name, "buildoptions")) {
        // We don't need to actually do anything for the umbrella element.
    } else {
        cerr << "error: unknown tag " << name << endl;
        exit (1);
    }
 
    XMLString::release (&name);    
}

void
BuildOptionFactory::warning(const xercesc::SAXParseException& excep)
{
    cout << "warning: " << XMLString::transcode(excep.getMessage()) << endl;
}

void
BuildOptionFactory::error(const xercesc::SAXParseException& excep)
{
    cout << "error: " << XMLString::transcode(excep.getMessage()) << endl;
}

void
BuildOptionFactory::fatalError(const xercesc::SAXParseException& excep)
{
    cout << "error: " << XMLString::transcode(excep.getMessage()) << endl;
    exit (1);
}
