//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <limits.h>
#include <iostream>
#include <sstream>
#include <vector>

#include "utils/options.h"
#include "Community.h"
#include "graphutils.h"
#include "Limbo.h"

using namespace aap;

Agraph_t* Community::partition (Agraph_t *Graph)
{
    double MQ,Q;
    unsigned int Lines;  
    Agraph_t *Subgraph;
    std::vector<Agraph_t*> Subgraphs;
    RootGraph = Graph;
    if (Options::verbose()) {
        std::cout << "Graph Lines: " << GraphUtils::NumLines(Graph)
                  << ", Nodes: "<< agnnodes(Graph) << std::endl;
    }
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        Subgraphs.push_back(Subgraph);
    for (unsigned int i=0; i<Subgraphs.size(); i++) {
        Subgraph = Subgraphs[i];
        Lines = (unsigned int) GraphUtils::NumLines(Subgraph);
        if ( ((unsigned) agnnodes (Subgraph) > Options::rpuMaxFunctions())  ||
             (Lines > Options::rpuMaxLines() && agnnodes(Subgraph) > 1) )
            SubgraphPartition(Subgraph);
    }
    Subgraphs.clear();

    if ((Options::keep_graph()) && 
        (Options::partitioner() == Options::COMMUNITY))
        GraphUtils::UpdateEdgeStyles(Graph);
    GraphUtils::MoveEdgesToSubgraphs(Graph);
    if (Options::verbose()) {
        MQ = GraphUtils::CalculateMQ(Graph);
        Q = GraphUtils::CalculateQ(Graph);
        std::cout << "Modulation Quality: " << MQ << std::endl;
        std::cout << "Modularity: " << Q << std::endl;
    }

    if (Options::partitioner() == Options::COMMUNLIMBO) {
        Limbo Combine;
        Combine.CombinePartitions(Graph);
        if (Options::keep_graph())
            GraphUtils::UpdateEdgeStyles(Graph);
        GraphUtils::MoveEdgesToSubgraphs(Graph);
        if (Options::verbose()) {
            MQ = GraphUtils::CalculateMQ(Graph);
            Q = GraphUtils::CalculateQ(Graph);
            std::cout << "After Limbo Modulation Quality: " << MQ << std::endl;
            std::cout << "After Limbo Modularity: " << Q << std::endl;
        }
    }
    return Graph;
}

void Community::SubgraphPartition (Agraph_t *Graph)
{
    unsigned int Lines;
    bool Continue;
    Agraph_t *Subgraph;
    Agnode_t *Head, *Tail;
    Agedge_t *MaxEdge;

    Continue = true;
    InitializeVectors(Graph);
    while (Continue) {
        MaxEdge = SubgraphShortestPaths(Graph);
        Head = aghead(MaxEdge);
        Tail = agtail(MaxEdge);
        agdeledge(Graph,MaxEdge);
        Subgraph = GraphUtils::NewSubGraph(RootGraph,Options::rpuPrefix());
        GraphUtils::MoveConnected(Head,Graph,Subgraph);
        if (agsubnode(Subgraph,Tail,FALSE) == NULL)
            Continue = false;
        else {
            agdelsubg(RootGraph,Graph);
            Graph = Subgraph;
        }
    }
    FreeVectors();

    Lines = (unsigned int) GraphUtils::NumLines(Graph);
    if ( ((unsigned) agnnodes (Graph) > Options::rpuMaxFunctions())  ||
         (Lines > Options::rpuMaxLines() && agnnodes(Graph) > 1) )
        SubgraphPartition(Graph);
    else
        if (Options::verbose()) {
            std::cout << agnameof(Graph) << " Subgraph    Lines: "
                      << GraphUtils::NumLines(Graph) << "   Nodes: "
                      << agnnodes(Graph) << std::endl;
        }
            
    Lines = (unsigned int) GraphUtils::NumLines(Subgraph);
    if ( ((unsigned) agnnodes (Subgraph) > Options::rpuMaxFunctions())  ||
         (Lines > Options::rpuMaxLines() && agnnodes(Subgraph) > 1) )
        SubgraphPartition(Subgraph);
    else
        if (Options::verbose()) {
            std::cout << agnameof(Subgraph) << " Subgraph    Lines: "
                      << GraphUtils::NumLines(Subgraph) << "   Nodes: "
                      << agnnodes(Subgraph) << std::endl;
        }
}

void Community::InitializeVectors (Agraph_t *Graph)
{
    int Index;
    Agnode_t *Node;

    NodeNumber = agnnodes(Graph);
    Nodes = new Agnode_t* [NodeNumber];
    Index = 0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node)) {
        Nodes[Index] = Node;
        NodesMap.insert(std::pair<Agnode_t*,int>(Node,Index));
        Index++;
    }
}
void Community::FreeVectors (void)
{
    delete [] Nodes;
    Nodes = NULL;
    NodesMap.clear();
}

// Return Value is the edge which is taken highest
Agedge_t* Community::SubgraphShortestPaths (Agraph_t *Graph)
{
    int  Max, Current, Weight;
    Agnode_t *Node;
    Agedge_t *Edge, *MaxEdge;
    Agedgeinfo_t *EdgeInfo;

    // Shortest paths from each node
    for (Current=0; Current<NodeNumber; Current++)
        SingleShortestPaths(Graph,Current);

    // Determine Edge with maximum path flows
    MaxEdge = NULL;
    Max = 0;
    Weight = INT_MAX;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node))
        for (Edge = agfstedge(Graph,Node); Edge != NULL;
             Edge = agnxtedge(Graph,Edge,Node)) {
            EdgeInfo = GraphUtils::getInfo(Edge);
            if ( (EdgeInfo->NumTaken > Max)      ||
                 ((EdgeInfo->NumTaken == Max) &&
                  (EdgeInfo->weight < Weight)  )  ) {
                Max = EdgeInfo->NumTaken;
                MaxEdge = Edge;
                Weight = EdgeInfo->weight;
            }
        }
    return MaxEdge;
}

void Community::SingleShortestPaths (Agraph_t *Graph, int Current)
{
    bool Continue;
    int i, Loc;
    Agnode_t *Cur;

    // Allocation and initialization
    int Distance[NodeNumber];
    int PrevNode[NodeNumber];
    bool NodeProc[NodeNumber];
    Agedge_t* PrevEdge[NodeNumber];

    for (i=0; i<NodeNumber; i++) {
        Distance[i] = INT_MAX;
        NodeProc[i] = true;
    }

    // Determine paths with Dijkstra's algorithm
    Distance[Current] = 0;
    PrevEdge[Current] = NULL;
    PrevNode[Current] = 0;
    Continue = true;
    while (Continue) {
        int CurDist = INT_MAX;
        for (i=0; i<NodeNumber; i++)
            if (NodeProc[i]) if (Distance[i] < CurDist) {
                    CurDist = Distance[i];
                    Loc = i;
                }
        Agnode_t* Src = Nodes[Loc];
        NodeProc[Loc] = false;
        for (Agedge_t* Edge = agfstedge(Graph,Src); Edge != NULL;
             Edge = agnxtedge(Graph,Edge,Src)) {
            Agedgeinfo_t *EdgeInfo = GraphUtils::getInfo(Edge);
            Agnode_t* Dst = GraphUtils::DestinedNode(Src,Edge);
            int NxtLoc = NodesMap[Dst];
            if (NodeProc[NxtLoc])
                if (Distance[NxtLoc] >
                    (Distance[Loc] + EdgeInfo->weight)) {
                    PrevNode[NxtLoc] = Loc;
                    PrevEdge[NxtLoc] = Edge;
                    Distance[NxtLoc] = Distance[Loc] + EdgeInfo->weight;
                }
        }

        Loc = 0;
        Continue = false;
        while ((Continue==false) && (Loc<NodeNumber)) {
            Continue = NodeProc[Loc];
            Loc++;
        }

    }

    // Count how many times an edge is taken
    for (i=0; i<NodeNumber; i++) {
        if (i == Current) continue;
        Agnode_t* Dst = Nodes[i];
        Loc = PrevNode[i];
        Cur = Nodes[Loc];
        Agedgeinfo_t *EdgeInfo = GraphUtils::getInfo(PrevEdge[i]);
        EdgeInfo->NumTaken++;
        while (Cur != Nodes[Current]) {
            EdgeInfo = GraphUtils::getInfo(PrevEdge[Loc]);
            EdgeInfo->NumTaken++;
            Loc = PrevNode[Loc];
            Cur = Nodes[Loc];
        }
    }
}

Community::Community (void)
{
    RootGraph  = NULL;
    NodeNumber = 0;
}

