//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <iostream>
#include <sstream>
#include <vector>
#include <cmath>

#include "graph/Limbo.h"
#include "utils/options.h"

#define LOG102 3.321928095 // 1/log10(2)

using namespace aap;

Agraph_t* Limbo::partition (Agraph_t *Graph)
{
    double MQ,Q;
    unsigned int Lines;
    Agraph_t *Subgraph, *NewGraph;
    std::vector<Agraph_t*> Subgraphs;
    RootGraph = Graph;
    if (Options::verbose()) {
        std::cout << "Graph Lines: " << GraphUtils::NumLines(Graph)
                  << ", Nodes: "<< agnnodes(Graph) << std::endl;
    }
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        Subgraphs.push_back(Subgraph);
    for (unsigned int i=0; i<Subgraphs.size(); i++) {
        Subgraph = Subgraphs[i];
        Lines = (unsigned int) GraphUtils::NumLines(Subgraph);
        if ( ((unsigned) agnnodes (Subgraph) > Options::rpuMaxFunctions())  ||
             (Lines > Options::rpuMaxLines() && agnnodes(Subgraph) > 1) ) {
            InitializeMatrices(Subgraph);
            GraphPartition();
            FreeMatrices();
        }
    }
    Subgraphs.clear();
    if (Options::keep_graph())
        GraphUtils::UpdateEdgeStyles(Graph);
    GraphUtils::MoveEdgesToSubgraphs(Graph);
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        Subgraphs.push_back(Subgraph);
    for (unsigned int i=0; i<Subgraphs.size(); i++) {
        Subgraph = Subgraphs[i];
        NewGraph = GraphUtils::MoveSubgraph(Subgraph, Options::rpuPrefix());
        if (Options::verbose())
            std::cout << agnameof(NewGraph) << " Subgraph    Lines: "
                      << (unsigned int) GraphUtils::NumLines(NewGraph)
                      << "   Nodes: "
                      << agnnodes(NewGraph) << std::endl;
    }
    Subgraphs.clear();
    if (Options::verbose()) {
        MQ = GraphUtils::CalculateMQ(Graph);
        Q = GraphUtils::CalculateQ(Graph);
        std::cout << "Modulation Quality: " << MQ << std::endl;
        std::cout << "Modularity: " << Q << std::endl;
    }
    return Graph;
}

void Limbo::GraphPartition (void)
{
    unsigned int SumNodes, SumLines;
    double MinLoss, CurLoss;
    int i,j,Cluster1, Cluster2, UseNumber;
    Agraph_t *Subgraph1,*Subgraph2, *NewGraph;

    UseNumber = ClusterNumber-1;
    while (UseNumber) {
        // File two clusters with least information loss
        MinLoss = 100.0;
        Cluster1 = Cluster2 = 0;
        for (i=0; i<ClusterNumber; i++)
            if (Use[i])
                for (j=i+1; j<ClusterNumber; j++)
                    if (Use[j] && Neighbor[i][j]) {
                        CurLoss = InformationLoss(i,j);
                        if (CurLoss < MinLoss) {
                            MinLoss  = CurLoss;
                            Cluster1 = i;
                            Cluster2 = j;
                        }
                    }

        if (MinLoss == 100.0) break;
        UseNumber--;

            // Combine clusters with least information loss
            // if maximum number of nodes not exceeded
            Subgraph1 = SubGraphMap[Cluster1];
            Subgraph2 = SubGraphMap[Cluster2];
            SumNodes = (unsigned int) agnnodes(Subgraph1) +
                       (unsigned int) agnnodes(Subgraph2);
            SumLines = (unsigned int) GraphUtils::NumLines(Subgraph1) +
                       (unsigned int) GraphUtils::NumLines(Subgraph2);
            if ( (SumNodes <= Options::rpuMaxFunctions()) &&
                 (SumLines <= Options::rpuMaxLines()) ) {
                NewGraph = GraphUtils::CombineSubgraphs(Subgraph1,Subgraph2,
                                                        Options::rpuPrefix());
                SubGraphMap.erase(Cluster2);
                SubGraphMap.erase(Cluster1);
                SubGraphMap.insert(std::pair<int,Agraph_t*>(Cluster1,
                                                            NewGraph));
                CombineClusters(Cluster1,Cluster2);
                Use[Cluster2] = false;
            }
            else {
                if (agnnodes(Subgraph1) > agnnodes(Subgraph2)) {
                    Use[Cluster1] = false;
                    SubGraphMap.erase(Cluster1);
                }
                else {
                    Use[Cluster2] = false;
                    SubGraphMap.erase(Cluster2);
                }
            }
    }
}

void Limbo::InitializeMatrices (Agraph_t *Graph)
{
    int i,j,Index;
    double Sum;
    std::map<Agnode_t*,int> NodeMap;
    Agedge_t *Edge, *Nedg;
    Agedgeinfo_t *EdgeInfo;
    Agnode_t *Node, *Next;
    Agraph_t *Subgraph;
    double **AdjMat;

    // Allocation and initialization
    NodeNumber = agnnodes(Graph);
    ClusterNumber = NodeNumber;
    AdjMat      = new double* [NodeNumber];
    AdjMat[0]   = new double  [NodeNumber*NodeNumber];
    Neighbor    = new bool* [NodeNumber];
    Neighbor[0] = new bool  [NodeNumber*NodeNumber];
    for (i=0; i<NodeNumber; i++) {
        AdjMat[i]   = AdjMat[0]+i*NodeNumber;
        Neighbor[i] = Neighbor[0]+i*NodeNumber;
        for (j=0; j<NodeNumber; j++) {
            AdjMat[i][j] = 0.0;
            Neighbor[i][j] = false;
        }
    }
    Index = 0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node)) {
        NodeMap.insert(std::pair<Agnode_t*,int>(Node,Index));
        Index++;
    }

    // Create Adjacency Matrix
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node)) {
        Edge = agfstedge(Graph,Node);
        while (Edge != NULL) {
            Next = GraphUtils::DestinedNode(Node,Edge);
            EdgeInfo = GraphUtils::getInfo(Edge);
            i = NodeMap[Node];
            j = NodeMap[Next];
            AdjMat[i][j] = (double) EdgeInfo->weight;
            AdjMat[j][i] = (double) EdgeInfo->weight;
            if (EdgeInfo->weight)  Neighbor[i][j] = true;
            Nedg = agnxtedge(Graph,Edge,Node);
            agdeledge(Graph,Edge);
            Edge = Nedg;
        }
    }

    // Create subgraphs with single nodes
    Node = agfstnode(Graph);
    while (Node != NULL) {
        Next = agnxtnode(Graph,Node);
        Index = NodeMap[Node];
        Subgraph = GraphUtils::NewSubGraph(RootGraph,Options::rpuPrefix());
        GraphUtils::moveNode(Node,Graph,Subgraph);
        SubGraphMap.insert(std::pair<int,Agraph_t*>(Index,Subgraph));
        Node = Next;
    }
    agdelsubg(RootGraph,Graph);
    NodeMap.clear();

    // Create Normalized matrix A\B
    NrmMat      = new double* [NodeNumber];
    NrmMat[0]   = new double  [NodeNumber*NodeNumber];
    Use         = new bool    [NodeNumber];
    PMF         = new double  [NodeNumber];
    for (i=0; i<NodeNumber; i++) {
        PMF[i] = 1.0 / ((double)NodeNumber);
        Use[i] = true;
        NrmMat[i] = NrmMat[0] + i*NodeNumber;
        Sum = 0.0;
        for (j=0; j<NodeNumber; j++) Sum += AdjMat[i][j];
        if (Sum == 0.0)
            for (j=0; j<NodeNumber; j++) NrmMat[i][j] = 0.0;
        else
            for (j=0; j<NodeNumber; j++) NrmMat[i][j] = AdjMat[i][j] / Sum;
    }
    delete [] AdjMat[0];
    delete [] AdjMat;
}

double Limbo::InformationLoss (int Cluster1, int Cluster2)
{
    int i;
    double NewPMF;
    double DKL1, DKL2, DJS;
    double Avg;

    // DKL1, DKL2: Kullback-Leibler (KL) divergence for each cluster
    // DJS: Jensen-Shannon (JS) divergence
    NewPMF = PMF[Cluster1]+PMF[Cluster2];
    DKL1 = 0.0;
    DKL2 = 0.0;
    for (i=0; i<NodeNumber; i++) {
        Avg  = PMF[Cluster1]*NrmMat[Cluster1][i];
        Avg += PMF[Cluster2]*NrmMat[Cluster2][i];
        Avg /= NewPMF;
        if (Avg != 0.0) {
            if (NrmMat[Cluster1][i] != 0.0)
                DKL1 += NrmMat[Cluster1][i] *
                    log10(NrmMat[Cluster1][i]/Avg)*LOG102;
            if (NrmMat[Cluster2][i] != 0.0)
                DKL2 += NrmMat[Cluster2][i] *
                    log10(NrmMat[Cluster2][i]/Avg)*LOG102;
        }
    }
    DJS  = (PMF[Cluster1]/NewPMF)*DKL1;
    DJS += (PMF[Cluster2]/NewPMF)*DKL2;

    return NewPMF*DJS;
}

void Limbo::CombineClusters (int Dst, int Src)
{
    int i;
    double NewPMF, Value;
    NewPMF = PMF[Dst] + PMF[Src];
    for (i=0; i<NodeNumber; i++) {
        Value  = PMF[Dst]*NrmMat[Dst][i];
        Value += PMF[Src]*NrmMat[Src][i];
        NrmMat[Dst][i] = Value / NewPMF;
        NrmMat[Src][i] = 0.0;
    }
    PMF[Dst] = NewPMF;
    for (i=0; i<ClusterNumber; i++) {
        Neighbor[Dst][i] = Neighbor[Dst][i] || Neighbor[Src][i];
        Neighbor[i][Dst] = Neighbor[i][Dst] || Neighbor[i][Src];
    }
}

void Limbo::FreeMatrices (void)
{
    delete [] NrmMat[0];
    delete [] NrmMat;
    delete [] Neighbor[0];
    delete [] Neighbor;
    delete [] Use;
    delete [] PMF;
    SubGraphMap.clear();
}

Limbo::Limbo (void)
{
    RootGraph = NULL;
    NodeNumber = 0;
    ClusterNumber = 0;
    NrmMat = NULL;
    Use = NULL;
    PMF = NULL;
}

void Limbo::CombinePartitions (Agraph_t *Graph)
{
    int i,j,Index;
    double Sum;
    std::map<Agnode_t*,int> NodeMap;
    std::vector<Agraph_t*> Subgraphs;
    std::map<Agnode_t*,int> NodeSub;
    Agedge_t *Edge;
    Agedgeinfo_t *EdgeInfo;
    Agnode_t *Node, *Next;
    Agraph_t *Subgraph, *NewGraph;
    double **AdjMat;

    RootGraph = Graph;
    // Allocation and initialization
    NodeNumber = agnnodes(Graph);
    Index = 0;
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph)) {
        SubGraphMap.insert(std::pair<int,Agraph_t*>(Index,Subgraph));
        Subgraphs.push_back(Subgraph);
        Index++;
    }
    ClusterNumber = Index;
    AdjMat      = new double* [ClusterNumber];
    AdjMat[0]   = new double  [ClusterNumber*NodeNumber];
    Neighbor    = new bool* [ClusterNumber];
    Neighbor[0] = new bool  [ClusterNumber*NodeNumber];
    for (i=0; i<ClusterNumber; i++) {
        AdjMat[i] = AdjMat[0]+i*NodeNumber;
        Neighbor[i] = Neighbor[0]+i*NodeNumber;
        for (j=0; j<NodeNumber; j++)
            AdjMat[i][j] = 0.0;
        for (j=0; j<ClusterNumber; j++)
            Neighbor[i][j] = false;
    }
    Index = 0;
    for (i=0; i<ClusterNumber; i++) {
        Subgraph = Subgraphs[i];
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node)) {
            NodeMap.insert(std::pair<Agnode_t*,int>(Node,Index));
            NodeSub.insert(std::pair<Agnode_t*,int>(Node,i));
            Index++;
        }
    }

    // Create Adjacency Matrix
    for (i=0; i<ClusterNumber; i++) {
        Subgraph = Subgraphs[i];
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node))
            for (Edge = agfstedge(Graph,Node); Edge != NULL;
                 Edge = agnxtedge(Graph,Edge,Node)) {
                Next = GraphUtils::DestinedNode(Node,Edge);
                EdgeInfo = GraphUtils::getInfo(Edge);
                j = NodeMap[Next];
                AdjMat[i][j] += (double) EdgeInfo->weight;
                Index = NodeSub[Next];
                if (EdgeInfo->weight) Neighbor[i][Index] = true;
            }
    }
    NodeMap.clear();

    // Create Normalized matrix A\B
    NrmMat      = new double* [ClusterNumber];
    NrmMat[0]   = new double  [ClusterNumber*NodeNumber];
    Use         = new bool    [ClusterNumber];
    PMF         = new double  [ClusterNumber];
    for (i=0; i<ClusterNumber; i++) {
        PMF[i]  = ((double) agnnodes(Subgraphs[i]));
        PMF[i] /= ((double) NodeNumber);
        Use[i] = true;
        NrmMat[i] = NrmMat[0] + i*NodeNumber;
        Sum = 0.0;
        for (j=0; j<NodeNumber; j++) Sum += AdjMat[i][j];
        if (Sum == 0.0)
            for (j=0; j<NodeNumber; j++) NrmMat[i][j] = 0.0;
        else
            for (j=0; j<NodeNumber; j++) NrmMat[i][j] = AdjMat[i][j] / Sum;
    }
    Subgraphs.clear();
    delete [] AdjMat[0];
    delete [] AdjMat;

    // Combine Clusters together
    GraphPartition();
    FreeMatrices();

    GraphUtils::UpdateEdgeStyles(Graph);
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        Subgraphs.push_back(Subgraph);
    for (unsigned int i=0; i<Subgraphs.size(); i++) {
        Subgraph = Subgraphs[i];
        NewGraph = GraphUtils::MoveSubgraph(Subgraph, Options::rpuPrefix());
        if (Options::verbose())
            std::cout << agnameof(NewGraph) << " Subgraph    Lines: "
                      << (unsigned int) GraphUtils::NumLines(NewGraph)
                      << "   Nodes: "
                      << agnnodes(NewGraph) << std::endl;
    }
    Subgraphs.clear();
}
