//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <iostream>
#include <sstream>
#include <vector>

#include "utils/options.h"
#include "graph/RandomWalks.h"
#include "graph/graphutils.h"
#include "graph/Limbo.h"

using namespace aap;

Agraph_t* RandomWalks::partition (Agraph_t *Graph)
{
    double MQ,Q;
    unsigned int Lines;  
    Agraph_t *Subgraph;
    std::vector<Agraph_t*> Subgraphs;
    RootGraph = Graph;
    if (Options::verbose()) {
        std::cout << "Graph Lines: " << GraphUtils::NumLines(Graph)
                  << ", Nodes: "<< agnnodes(Graph) << std::endl;
    }
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        Subgraphs.push_back(Subgraph);
    for (unsigned int i=0; i<Subgraphs.size(); i++) {
        Subgraph = Subgraphs[i];
        Lines = (unsigned int) GraphUtils::NumLines(Subgraph);
        if ( ((unsigned) agnnodes (Subgraph) > Options::rpuMaxFunctions())  ||
             (Lines > Options::rpuMaxLines() && agnnodes(Subgraph) > 1) )
            SubgraphPartition(Subgraph);
    }
    Subgraphs.clear();

    if ((Options::keep_graph()) &&
        (Options::partitioner() == Options::RANDOMWALKS))
        GraphUtils::UpdateEdgeStyles(Graph);
    GraphUtils::MoveEdgesToSubgraphs(Graph);
    if (Options::verbose()) {
        MQ = GraphUtils::CalculateMQ(Graph);
        Q = GraphUtils::CalculateQ(Graph);
        std::cout << "Modulation Quality: " << MQ << std::endl;
        std::cout << "Modularity: " << Q << std::endl;
    }

    if (Options::partitioner() == Options::RANDOMLIMBO) {
        Limbo Combine;
        Combine.CombinePartitions(Graph);
        if (Options::keep_graph())
            GraphUtils::UpdateEdgeStyles(Graph);
        GraphUtils::MoveEdgesToSubgraphs(Graph);
        if (Options::verbose()) {
            MQ = GraphUtils::CalculateMQ(Graph);
            Q = GraphUtils::CalculateQ(Graph);
            std::cout << "After Limbo Modulation Quality: " << MQ << std::endl;
            std::cout << "After Limbo Modularity: " << Q << std::endl;
        }
    }
    return Graph;
}

void RandomWalks::SubgraphPartition (Agraph_t *Graph)
{
    int Src, Dst;
    unsigned int Lines;
    bool Continue;
    Agraph_t *Subgraph;
    Agnode_t *Head, *Tail;
    Agedge_t *MaxEdge;

    Continue = true;
    InitializeMatrices(Graph);
    while (Continue) {
        SubgraphRandomWalks(&Src,&Dst);
        MaxEdge = EdgeMap[Src*NodeNumber+Dst];
        Head = aghead(MaxEdge);
        Tail = agtail(MaxEdge);
        agdeledge(Graph,MaxEdge);
        Subgraph = GraphUtils::NewSubGraph(RootGraph,Options::rpuPrefix());
        GraphUtils::MoveConnected(Head,Graph,Subgraph);
        if (agsubnode(Subgraph,Tail,FALSE) == NULL)
            Continue = false;
        else {
            agdelsubg(RootGraph,Graph);
            Graph = Subgraph;
            AdjMat[Src*NodeNumber+Dst] = 0.0;
            AdjMat[Dst*NodeNumber+Src] = 0.0;
        }
    }
    FreeMatrices();

    Lines = (unsigned int) GraphUtils::NumLines(Graph);
    if ( ((unsigned) agnnodes (Graph) > Options::rpuMaxFunctions())  ||
         (Lines > Options::rpuMaxLines() && agnnodes(Graph) > 1) )
        SubgraphPartition(Graph);
    else
        if (Options::verbose()) {
            std::cout << agnameof(Graph) << " Subgraph    Lines: "
                      << GraphUtils::NumLines(Graph) << "   Nodes: "
                      << agnnodes(Graph) << std::endl;
        }
            
    Lines = (unsigned int) GraphUtils::NumLines(Subgraph);
    if ( ((unsigned) agnnodes (Subgraph) > Options::rpuMaxFunctions())  ||
         (Lines > Options::rpuMaxLines() && agnnodes(Subgraph) > 1) )
        SubgraphPartition(Subgraph);
    else
        if (Options::verbose()) {
            std::cout << agnameof(Subgraph) << " Subgraph    Lines: "
                      << GraphUtils::NumLines(Subgraph) << "   Nodes: "
                      << agnnodes(Subgraph) << std::endl;
        }
}

void RandomWalks::InitializeMatrices (Agraph_t *Graph)
{
    int Index,i,j;
    double Weight;
    Agnode_t *Node,*Next;
    Agedge_t *Edge;
    Agedgeinfo_t *EdgeInfo;
    std::map<Agnode_t*,int> NodesMap;

    // Allocation and initialization
    NodeNumber = agnnodes(Graph);
    AdjMat = new double [NodeNumber*NodeNumber];
    for (i=0; i<NodeNumber; i++)
        for (j=0; j<NodeNumber; j++)
            AdjMat[i*NodeNumber+j] = 0.0;

    // Map Nodes to Integer
    Index = 0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node)) {
        NodesMap.insert(std::pair<Agnode_t*,int>(Node,Index));
        Index++;
    }

    // Map Integers to Edges and Create Adjacency Matrix
    MaxWeight = 0.0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node))
        for (Edge = agfstedge(Graph,Node); Edge != NULL;
             Edge = agnxtedge(Graph,Edge,Node)) {
            EdgeInfo = GraphUtils::getInfo(Edge);
            Weight = ((double)EdgeInfo->weight);
            if (Weight > MaxWeight) MaxWeight = Weight + 1.0;
            Next = GraphUtils::DestinedNode(Node,Edge);
            i = NodesMap[Node];
            j = NodesMap[Next];
            AdjMat[i*NodeNumber+j] = Weight;
            AdjMat[j*NodeNumber+i] = Weight;
            EdgeMap.insert(std::pair<int,Agedge_t*>(i*NodeNumber+j,Edge));
        }
    NodesMap.clear();
    
    for (i=0; i<NodeNumber; i++)
        for (j=0; j<NodeNumber; j++)
            if (AdjMat[i*NodeNumber+j] > 0.0)
                AdjMat[i*NodeNumber+j] = MaxWeight - AdjMat[i*NodeNumber+j];
}

void RandomWalks::FreeMatrices (void)
{
    delete [] AdjMat;
    EdgeMap.clear();
}

// Return Value is the edge which is taken highest
void RandomWalks::SubgraphRandomWalks (int *Src, int *Dst)
{
    int i,j,Current;
    double *Matrix, *Source, *Result, *Betweenness, Diff, Max;

    // Allocate Matrices and Vectors
    Matrix      = new double [NodeNumber*NodeNumber];
    Betweenness = new double [NodeNumber*NodeNumber];
    Source      = new double [NodeNumber];
    Result      = new double [NodeNumber];
    for (i=0; i<NodeNumber; i++)
        for (j=0; j<NodeNumber; j++)
            Betweenness[i*NodeNumber+j] = 0.0;
    
    // Random Walks from each node
    for (Current=0; Current<(NodeNumber-1); Current++)  {
        CalculateResult(Matrix,Source,Result,Current);
        for (i=0; i<NodeNumber; i++) {
            for (j=i+1; j<NodeNumber; j++) {
                Diff = Result[i] - Result[j];
                if (Diff < 0.0) Diff = -Diff;
                Betweenness[i*NodeNumber+j] += AdjMat[i*NodeNumber+j]*Diff;
            }
        }
    }

    // Determine Edge with maximum random walks
    Max = 0.0;
    *Src = -1;
    *Dst = -1;
    for (i=0; i<NodeNumber; i++)
        for (j=i+1; j<NodeNumber; j++) {
            if (Betweenness[i*NodeNumber+j] > Max) {
                Max = Betweenness[i*NodeNumber+j];
                *Src = i;
                *Dst = j;
            }
        }

    // Free allocated matrices and arrays
    delete [] Matrix;
    delete [] Betweenness;
    delete [] Source;
    delete [] Result;
}

// Solves AX=B linear equations for random walks.
// Source is the B vector where current is set 1 and others to 0
void RandomWalks::CalculateResult (double *Matrix, double *Source,
                                   double *Result, int Current)
{
    int Row,Col,Pivot,Size;
    double Factor,Sum;

    Size = NodeNumber-1;
    for (Row=0; Row<Size; Row++) {
        Sum = 0.0;
        for (Col=0; Col<NodeNumber; Col++) {
            if (AdjMat[Row*NodeNumber+Col] == 0.0)
                Matrix[Row*NodeNumber+Col] = 0.0;
            else
                Matrix[Row*NodeNumber+Col] = -AdjMat[Row*NodeNumber+Col];
            Sum += AdjMat[Row*NodeNumber+Col];
        }
        Matrix[Row*NodeNumber+Row] = Sum;
        Source[Row] = 0.0;
        Result[Row] = 0.0;
    }
    Source[Current] = MaxWeight;
    Source[Row] = 0.0;
    Result[Row] = 0.0;

    for (Pivot=0; Pivot<Size; Pivot++) {
        for (Row=Pivot+1; Row<Size; Row++) {
            Factor = Matrix[Row*NodeNumber  +Pivot] /
                Matrix[Pivot*NodeNumber+Pivot];
            Matrix[Row*NodeNumber+Pivot] = 0.0;
            for (Col=Pivot+1; Col<Size; Col++)
                Matrix[Row*NodeNumber+Col] -=
                    Factor*Matrix[Pivot*NodeNumber+Col];
            Source[Row] -= Factor*Source[Pivot];
        }
    }

    for (Row=Size-1; Row>=0; Row--) {
        Sum = 0.0;
        for (Col=Row+1; Col<Size; Col++) {
            Sum += Matrix[Row*NodeNumber+Col]*Result[Col];
        }
        Result[Row] = (Source[Row]-Sum)/Matrix[Row*NodeNumber+Row];
    }
}

RandomWalks::RandomWalks (void)
{
    RootGraph  = NULL;
    NodeNumber = 0;
    AdjMat = NULL;
}


