//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "graph/graphutils.h"
#include "utils/options.h"

#include <assert.h>

#include <cstdio>
#include <iostream>
#include <sstream>

using namespace std;
using namespace aap;

typedef vector<Agedge_t*> edge_v;
typedef vector<Agnode_t*> NodeVector;

static char kEdgeInfo[]  = "AapEdgeInfo";
static char kGraphInfo[] = "AapGraphInfo";
static char kNodeInfo[]  = "AapNodeInfo";

static char kEmpty[]     = "";
static char kFile[]      = "file";
static char kLabel[]     = "label";
static char kLine[]      = "line";
static char kLineCount[] = "linecount";
static char kRetcnt[]    = "retcnt";
static char kTime[]      = "time";
static char kZero[]      = "0";

static char StrStyle[]  = "style";
static char StrSolid[]  = "solid";
static char StrDashed[] = "dashed";
static char StrDotted[] = "dotted";

bool
GraphUtils::edge_compare (Agedge_t *e1, Agedge_t *e2)
{
    Agedgeinfo_t *e1info, *e2info;
    e1info = getInfo (e1);
    e2info = getInfo (e2);
    return (e1info->weight < e2info->weight);
}

int
GraphUtils::NumLines (Agraph_t* graph)
{
    int num = 0;

    for (Agnode_t* node = agfstnode(graph);
         node != NULL;
         node = agnxtnode(graph, node))
    {
        Agnodeinfo_t* nodeInfo;
        nodeInfo = GraphUtils::getInfo (node);
        ProcedureRef proc = nodeInfo->procedure;
        if (proc->function()) {
            const SourceLocation& loc = proc->function()->source();
            num += loc.lastLine() - loc.first_line();
        }
    }

    return num;
}

Agedge_t*
GraphUtils::find_min_edge (Agraph_t* graph)
{
    Agnode_t* node;
    Agedge_t* edge;
    Agedge_t* min_e = NULL;
    edge_v::iterator it;
    long min_w = LONG_MAX;
    Agedgeinfo_t* edgeInfo;

    for (node = agfstnode (graph);
         node != NULL;
         node = agnxtnode (graph, node))
    {
        if (agisdirected(graph))
            for (edge = agfstout (graph, node);
                 edge != NULL;
                 edge = agnxtout (graph, edge))
            {
                edgeInfo = getInfo (edge);
                if (min_w > edgeInfo->weight && Validator::isValidCut (edge))
                {
                    min_e = edge;
                    min_w = edgeInfo->weight;
                }
            }
        else
            for (edge = agfstedge (graph, node);
                 edge != NULL;
                 edge = agnxtedge (graph, edge, node))  {
                edgeInfo = getInfo(edge);
                if (min_w > edgeInfo->weight)   {
                    min_e = edge;
                    min_w = edgeInfo->weight;
                }
            }
    }
    return min_e;
}

void
GraphUtils::moveNode (Agnode_t* node, Agraph_t* from, Agraph_t* to)
{
    Agnodeinfo_t* nodeInfo = getInfo (node);
    agsubnode (to, node, TRUE);
    if (from != agroot(to)) agdelnode (from, node);
    nodeInfo->subg = to;
}

// move the tree rooted at root to from oldg to newg
void
GraphUtils::move_tree (Agnode_t* root, Agraph_t* oldg, Agraph_t* newg)
{
    edge_v edges;
    Agedge_t* edge;

    // identify edges from the root to other nodes in the old sub-graph
    for (edge = agfstout (oldg, root); edge; edge = agnxtout (oldg, edge))
        edges.push_back (edge);

    // move the root to the new sub-graph
    moveNode (root, oldg, newg);

    // move each subtree to the new sub-graph
    edge_v::iterator it, end = edges.end();
    for (it = edges.begin(); it != end; ++it) {
        edge = *it;
        agsubedge (newg, edge, TRUE);
        agdeledge (oldg, edge);
        move_tree (aghead (edge), oldg, newg);
    }
}

void
GraphUtils::remove_tree (Agraph_t* graph, Agnode_t* root)
{
    Agedge_t* edge;
    for (edge = agfstout(graph, root); edge; edge = agnxtout (graph, edge)) {
        remove_tree(graph, aghead (edge));
        agdeledge(graph, edge);
    }
    agdelnode(graph, root);
}

long
GraphUtils::tree_weight (Agraph_t* graph, Agnode_t* root)
{
    long size = 0;
    Agedge_t* edge;
    Agedgeinfo_t* edgeInfo;
    for (edge = agfstout(graph, root); edge; edge = agnxtout(graph, edge)) {
        edgeInfo = (Agedgeinfo_t*) aggetrec (edge, kEdgeInfo, FALSE);
        size += edgeInfo->weight;
        size += tree_weight(graph, aghead (edge));
    }

    return size;
}

int
GraphUtils::tree_time(Agraph_t* graph, Agnode_t* root)
{
    Agnodeinfo_t* nodeInfo = GraphUtils::getInfo (root);
    int time = nodeInfo->time;
    for (Agedge_t* edge = agfstout(graph, root);
         edge != NULL;
         edge = agnxtout(graph, edge))
    {
        time += tree_time(graph, aghead (edge));
    }

    return time;
}

void
GraphUtils::dump_graph (Agraph_t* graph, const string& file)
{
    FILE* ofile = fopen (file.c_str(), "w");
    agwrite (graph, ofile);
    fclose (ofile);
}

Agraphinfo_t*
GraphUtils::getInfo (Agraph_t* graph)
{
    return (Agraphinfo_t*) aggetrec (graph, kGraphInfo, FALSE);
}

Agnodeinfo_t*
GraphUtils::getInfo (Agnode_t* node)
{
    return (Agnodeinfo_t*) aggetrec (node, kNodeInfo, FALSE);
}

Agedgeinfo_t*
GraphUtils::getInfo (Agedge_t* edge)
{
    return (Agedgeinfo_t*) aggetrec (edge, kEdgeInfo, FALSE);
}

Agedge_t*
GraphUtils::createEdge (Agraph_t* graph, Agnode_t* from, Agnode_t* to)
{
    Agedge_t* edge = agedge (graph, from, to, NULL, TRUE);
    agbindrec (edge, kEdgeInfo, sizeof (Agedgeinfo_t), FALSE);
    return edge;
}

Agnode_t*
GraphUtils::createNode (Agraph_t* graph, char* label)
{
    Agnode_t* node = agnode (graph, label, TRUE);
    initializeNode (node, graph);
    return node;
}

void
GraphUtils::initializeGraph (Agraph_t* graph)
{
    agattr (graph, AGRAPH, kLabel, kEmpty);

    agattr (graph, AGEDGE, kLabel, kEmpty);

    agattr (graph, AGNODE, kLabel,     kEmpty);
    agattr (graph, AGNODE, kRetcnt,    kZero);
    agattr (graph, AGNODE, kTime,      kZero);
    agattr (graph, AGNODE, kFile,      kEmpty);
    agattr (graph, AGNODE, kLine,      kEmpty);
    agattr (graph, AGNODE, kLineCount, kEmpty);

    agbindrec (graph, kGraphInfo, sizeof (Agraphinfo_t), FALSE);
}

void GraphUtils::initializeNode (Agnode_t* node, Agraph_t* subgraph)
{
    Agnodeinfo_t* NodeInfo = (Agnodeinfo_t*)
        agbindrec (node, kNodeInfo, sizeof (Agnodeinfo_t),FALSE);
    NodeInfo->subg = subgraph;
    NodeInfo->retcnt = 0;
    NodeInfo->time = 0;
    NodeInfo->staticSite = 0;
    NodeInfo->Depth = 0;
}

Agnode_t*
GraphUtils::treeRoot (Agraph_t* subgraph)
{
    // get the "first" node and move up to it's highest ancestor
    Agnode_t* node = agfstnode (subgraph);
    Agedge_t* inedge;
    if (agisundirected (subgraph)) return node;
    while ((inedge = agfstin (subgraph, node)) != NULL)
        node = agtail (inedge);
    return node;
}

void
GraphUtils::setLineCounts (Agraph_t* graph)
{
    Agnode_t* node = agfstnode(graph);
    while (node) {
        // set the line count as an attribute (for Jack)
        SourceObjRef function;
        Agnodeinfo_t* info = GraphUtils::getInfo(node);
        function = info->procedure->function();

        char lineCountStr[100];
        if (function != NULL) {
            const SourceLocation& loc = function->source();
            int lineCount = loc.lastLine() - loc.first_line() + 1;
            snprintf(lineCountStr, 100, "%d", lineCount);
        } else {
            lineCountStr[0] = '0';
            lineCountStr[1] = '\0';
        }
        agset(node, kLineCount, lineCountStr);

        node = agnxtnode(graph, node);
    }
}

Agnode_t* GraphUtils::DestinedNode (Agnode_t* Source, Agedge_t* Edge)
{
  Agnode_t *Destination;
  if (aghead(Edge) == Source)
    Destination = agtail(Edge);
  else
    Destination = aghead(Edge);
  return Destination;
}

// Create Connected Subgraph
void GraphUtils::MoveConnected (Agnode_t* root, Agraph_t* oldg, Agraph_t* newg)
{
  edge_v HeadEdges,TailEdges;
  edge_v::iterator iter;
  Agedge_t* edge;
  // identify edges from the root to other nodes in the old sub-graph
  for (edge = agfstedge (oldg, root); edge != NULL;
       edge = agnxtedge (oldg, edge, root))
    if (aghead(edge) == root)
      TailEdges.push_back(edge);
    else
      HeadEdges.push_back(edge);
  // move the root to the new sub-graph
  moveNode (root, oldg, newg);
  // move each subtree to the new sub-graph
  for (iter = HeadEdges.begin(); iter != HeadEdges.end(); ++iter)
    {
      edge = *iter;
      if (agsubnode(newg,aghead(edge),FALSE) == NULL)
	MoveConnected (aghead(edge), oldg, newg);
      agsubedge (newg, edge, TRUE);
    }
  for (iter = TailEdges.begin(); iter != TailEdges.end(); ++iter)
    {
      edge = *iter;
      if (agsubnode(newg,agtail(edge),FALSE) == NULL)
	MoveConnected (agtail(edge), oldg, newg);
      agsubedge (newg, edge, TRUE);
    }
}

void GraphUtils::MoveUndTree (Agnode_t* Root, Agraph_t* Old, Agraph_t* New)
{
  edge_v HeadEdges,TailEdges;
  edge_v::iterator iter;
  Agnodeinfo_t *NodeInfo, *RootInfo;
  Agnode_t *Node;
  Agedge_t *edge;
  // identify edges from the root to other nodes in the old sub-graph 
  for (edge = agfstedge (Old, Root); edge != NULL;
       edge = agnxtedge (Old, edge, Root))
    if (aghead(edge) == Root)
      TailEdges.push_back(edge);
    else
      HeadEdges.push_back(edge);
  // move the root to the new sub-graph
  RootInfo = GraphUtils::getInfo(Root);
  moveNode (Root, Old, New);

  // move each subtree to the new sub-graph

  for (iter = HeadEdges.begin(); iter != HeadEdges.end(); ++iter)
    {
      edge = *iter;
      Node = aghead(edge);
      NodeInfo = GraphUtils::getInfo(Node);
      if (NodeInfo->Depth > RootInfo->Depth)
	{
	  if (agsubnode(New,Node,FALSE) == NULL)
	    MoveUndTree (Node, Old, New);
	  agsubedge(New, edge, TRUE);
	}
    }
  for (iter = TailEdges.begin(); iter != TailEdges.end(); ++iter)
    {
      edge = *iter;
      Node = agtail(edge);
      NodeInfo = GraphUtils::getInfo(Node);
      if (NodeInfo->Depth > RootInfo->Depth)
        {
          if (agsubnode(New,Node,FALSE) == NULL)
            MoveUndTree (Node, Old, New);
          agsubedge(New, edge, TRUE);
        }
    }
}

void GraphUtils::SetDepth (Agnode_t* Node, unsigned int Depth)
{
  Agnodeinfo_t *NodeInfo = GraphUtils::getInfo(Node);
  NodeInfo->Depth = Depth;
}
unsigned int GraphUtils::GetDepth (Agnode_t* Node)
{
  Agnodeinfo_t *NodeInfo = GraphUtils::getInfo(Node);
  return NodeInfo->Depth;
}

void GraphUtils::CreateDepth (Agnode_t* Root, Agraph_t* Graph)
{
  Agnodeinfo_t *NodeInfo = GraphUtils::getInfo(Root);
  Agedge_t *Edge;  Agnode_t *Node;
  NodeVector Nodes;
  NodeVector::iterator iter;
  for (Edge = agfstedge(Graph,Root); Edge != NULL;
       Edge = agnxtedge(Graph,Edge,Root))
    {
      Node = DestinedNode(Root,Edge);
      if (GetDepth(Node) == 0)
	{
	  SetDepth(Node, NodeInfo->Depth+1);
	  Nodes.push_back(Node);
	}
    }
  for (iter = Nodes.begin(); iter != Nodes.end(); iter++)
    {
      Node = *iter;
      CreateDepth(Node,Graph);
    }
  Nodes.clear();
}

// Returns a new subgraph with a unique name. Creates a name, checks for an 
// existing subgraph, and if none exits creates one with that name.
Agraph_t* GraphUtils::NewSubGraph (Agraph_t* RootGraph, const string& prefix)
{
    assert (RootGraph == agroot (RootGraph));
    unsigned rpu_uid = 0;
    Agraph_t* NewGraph  = NULL;
    while (!NewGraph)  {
        std::ostringstream oss;
        oss << prefix << rpu_uid++;
        char* label = strdup (oss.str().c_str());
        NewGraph = agsubg (RootGraph, label, FALSE);
        if (NewGraph) {
            NewGraph = NULL;
        } else {
            NewGraph = agsubg (RootGraph, label, TRUE);
            agset(NewGraph, kLabel, label);
        }
        free (label);
    }
    return NewGraph;
}

void GraphUtils::DeleteGraph (Agraph_t* Graph)
{
  Agnode_t* Node;
  Agnodeinfo_t* NodeInfo;
  for (Node = agfstnode(Graph); Node != NULL;
       Node = agnxtnode(Graph, Node))
    {
      NodeInfo = GraphUtils::getInfo(Node);
      NodeInfo->procedure = NULL;
    }
  agclose(Graph);
}

// Modify the style of graph edges
// Solid:  Intra-Subgraph edges
// Dotted: Inter-Subgraph edges (Connecting nodes of different subgraphs)
// Dashed: Edges connecting nodes of same subgraph,
//         but are not included in the subgraph (cut edges)
void GraphUtils::UpdateEdgeStyles (Agraph_t *Graph)
{
    Agraph_t *Subgraph;
    Agnode_t *Node, *Next;
    Agedge_t *Edge;
    agattr(Graph,AGEDGE,StrStyle,StrSolid);
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node))
            for (Edge = agfstedge(Graph,Node); Edge != NULL;
                 Edge = agnxtedge(Graph,Edge,Node))
                if (agsubedge(Subgraph,Edge,FALSE) == NULL) {
                    Next = GraphUtils::DestinedNode(Node,Edge);
                    if (agsubnode(Subgraph,Next,FALSE) != NULL) {
                        agset((void*)Edge,StrStyle,StrDashed);
                    }
                    else
                        agset((void*)Edge,StrStyle,StrDotted);
                }
}

// Moves cut edges which connect two nodes of the same subgraph back
// to their corresponding subgraphs
void GraphUtils::MoveEdgesToSubgraphs (Agraph_t *Graph)
{
    Agraph_t *Subgraph;
    Agnode_t *Node, *Next;
    Agedge_t *Edge;
    // Include cut edges connecting nodes of the same
    // subgraph back to the subgraphs
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph))
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node))
            for (Edge = agfstedge(Graph,Node); Edge != NULL;
                 Edge = agnxtedge(Graph,Edge,Node))
                if (agsubedge(Subgraph,Edge,FALSE) == NULL) {
                    Next = GraphUtils::DestinedNode(Node,Edge);
                    if (agsubnode(Subgraph,Next,FALSE) != NULL)
                        agsubedge(Subgraph,Edge,TRUE);
                }
}

// Calculate Modulation quality. The measure is proposed in paper:
// "On the Automatic Modularization of Software Systems Using the Bunch Tool",
// IEEE Transactions on Software Engineering.
double GraphUtils::CalculateMQ (Agraph_t *Graph)
{
    double MQ, CF, mu, eps;
    Agraph_t *Subgraph;
    Agnode_t *Node;
    Agedge_t *Edge;
    Agedgeinfo_t *EdgeInfo;
    MQ = 0.0;
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph)) {
        mu  = 0.0;
        eps = 0.0;
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node))
            for (Edge = agfstedge(Graph,Node); Edge != NULL;
                 Edge = agnxtedge(Graph,Edge,Node)) {
                EdgeInfo = GraphUtils::getInfo(Edge);
                if (agsubedge(Subgraph,Edge,FALSE) == NULL)
                    eps += ((double) EdgeInfo->weight);
                else
                    mu  += ((double) EdgeInfo->weight);
            }
        if (mu == 0.0) CF = 0.0;
        else
            CF = 2*mu / (2*mu+eps);
        MQ += CF;
    }
    return MQ;
}

// Calculate Modularity. The measure is proposed in paper:
// "Finding and evaluating community structure in networks",
// Physical Review E 69.
double GraphUtils::CalculateQ (Agraph_t *Graph)
{
    double Q, Sum, InEdges, AllEdges, SumSq;
    Agraph_t *Subgraph;
    Agnode_t *Node;
    Agedge_t *Edge;
    Agedgeinfo_t *EdgeInfo;

    Sum = 0.0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node))
        for (Edge = agfstedge(Graph,Node); Edge != NULL;
             Edge = agnxtedge(Graph,Edge,Node)) {
            EdgeInfo = GraphUtils::getInfo(Edge);
            Sum += (double) EdgeInfo->weight;
        }
    if (Sum == 0.0) return 0.0;

    SumSq = Sum*Sum;
    Q=0.0;
    for (Subgraph = agfstsubg(Graph); Subgraph != NULL;
         Subgraph = agnxtsubg(Subgraph)) {
        InEdges  = 0.0;
        AllEdges = 0.0;
        for (Node = agfstnode(Subgraph); Node != NULL;
             Node = agnxtnode(Subgraph,Node)) {
            for (Edge = agfstedge(Graph,Node); Edge != NULL;
                 Edge = agnxtedge(Graph,Edge,Node)) {
                EdgeInfo = GraphUtils::getInfo(Edge);
                AllEdges += (double) EdgeInfo->weight;
            }
            for (Edge = agfstedge(Subgraph,Node); Edge != NULL;
                 Edge = agnxtedge(Subgraph,Edge,Node)) {
                EdgeInfo = GraphUtils::getInfo(Edge);
                InEdges += (double) EdgeInfo->weight;
            }
        }
        Q += InEdges/Sum - (AllEdges*AllEdges)/SumSq;
    }
    return Q;
}

// Combine Subgraphs 1 and 2
Agraph_t* GraphUtils::CombineSubgraphs (Agraph_t *Subgraph1,
                                        Agraph_t *Subgraph2,
                                        const string& prefix)
{
    Agraph_t *Root, *CombinedSubgraph;
    unsigned int i;
    Agnode_t *Node;
    Agedge_t *Edge;
    NodeVector Nodes;

    Root = agroot(Subgraph1);
    for (Node = agfstnode(Subgraph1); Node != NULL;
         Node = agnxtnode(Subgraph1,Node))
        Nodes.push_back(Node);
    for (Node = agfstnode(Subgraph2); Node != NULL;
         Node = agnxtnode(Subgraph2,Node))
        Nodes.push_back(Node);
    agdelsubg(Root,Subgraph1);
    agdelsubg(Root,Subgraph2);

    // Move Nodes and Edges to new Subgraph
    CombinedSubgraph = NewSubGraph(Root,prefix);
    for (i=0; i<Nodes.size(); i++)
        moveNode(Nodes[i],Root,CombinedSubgraph);
    Nodes.clear();
    for (Node = agfstnode(CombinedSubgraph); Node != NULL;
         Node = agnxtnode(CombinedSubgraph,Node))
        for (Edge = agfstedge(Root,Node); Edge != NULL;
             Edge = agnxtedge(Root,Edge,Node))
            if (agsubnode(CombinedSubgraph,
                          DestinedNode(Node,Edge),FALSE) != NULL)
                agsubedge(CombinedSubgraph,Edge,TRUE);

    return CombinedSubgraph;
}

// Create a new subgraph and move Subgraph to new one
Agraph_t* GraphUtils::MoveSubgraph (Agraph_t *Subgraph, const string& prefix)
{
    Agraph_t *Root, *NewSubgraph;
    unsigned int i;
    Agnode_t *Node;
    Agedge_t *Edge;
    NodeVector Nodes;
    edge_v Edges;

    Root = agroot(Subgraph);
    for (Node = agfstnode(Subgraph); Node != NULL;
         Node = agnxtnode(Subgraph,Node)) {
        Nodes.push_back(Node);
        for (Edge = agfstedge(Subgraph,Node); Edge != NULL;
             Edge = agnxtedge(Subgraph,Edge,Node))
            Edges.push_back(Edge);
    }
    agdelsubg(Root,Subgraph);

    // Move Nodes and Edges to new Subgraph
    NewSubgraph = NewSubGraph(Root,prefix);
    for (i=0; i<Nodes.size(); i++)
        moveNode(Nodes[i],Root,NewSubgraph);
    Nodes.clear();
    for (i=0; i<Edges.size(); i++)
        agsubedge(NewSubgraph,Edges[i],TRUE);
    Edges.clear();

    return NewSubgraph;
}

long GraphUtils::NodeWeight (Agraph_t* Graph, Agnode_t* Node)
{
    long TotalWeight;
    Agedge_t *Edge;
    Agedgeinfo_t *EdgeInfo;
    TotalWeight = 0;
    for (Edge = agfstedge(Graph,Node); Edge != NULL;
         Edge = agnxtedge(Graph,Edge,Node)) {
        EdgeInfo = getInfo(Edge);
        TotalWeight += EdgeInfo->weight;
    }
    return TotalWeight;
}

long GraphUtils::GraphWeight (Agraph_t* Graph)
{
    long TotalWeight;
    Agnode_t *Node;
    TotalWeight = 0;
    for (Node = agfstnode(Graph); Node != NULL;
         Node = agnxtnode(Graph,Node))
        TotalWeight += NodeWeight(Graph,Node);
    return TotalWeight;
}
