//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "graph/mincut.h"

#include <assert.h>

#include <cstdlib>
#include <iostream>
#include <sstream>
#include <vector>

#include "sourcelocation.h"
#include "graph/graph.h"
#include "graph/graphutils.h"
#include "utils/options.h"

using namespace std;
using namespace aap;

static char kLabel[] = "label";

// return a new graph into which some subset of the nodes and edges from the
// given graph has been moved. those objects no longer appear in the input
// graph (the provided graph should be a subgraph of another Agraph_t object)
static Agraph_t*
cut_graph (Agraph_t* graph, const string& prefix)
{
    Agedge_t* cut;
    // make sure we're not cutting the root graph
    assert (graph != agroot (graph));

    cut = GraphUtils::find_min_edge (graph);

    if (cut == NULL) {
        cerr << "Aborting: unable to find an edge to cut\n";
        abort();
    }

    Agraph_t* new_graph = GraphUtils::NewSubGraph(agroot(graph), prefix);

    // get one of the endpoints from the edges and use it as the root
    Agnode_t *Head, *Tail, *root;
    Agnodeinfo_t *HeadInfo, *TailInfo;
    Head = aghead(cut);
    Tail = agtail(cut);
    if (agisdirected(graph)) root = Head;
    else {
        HeadInfo = GraphUtils::getInfo(Head);
        TailInfo = GraphUtils::getInfo(Tail);
        if (HeadInfo->Depth < TailInfo->Depth)
            root = Tail;
        else
            root = Head;
    }

    agdeledge (graph, cut);
    if (agisdirected (graph))
        GraphUtils::move_tree (root, graph, new_graph);
    else
        GraphUtils::MoveUndTree (root, graph, new_graph);
    
    return new_graph;
}

static void
min_cuts (Agraph_t* graph,
          const string& prefix,
          unsigned thresholdNodes,
          unsigned thresholdLines)
{
    Agraph_t *subgraph, *sibling;
    vector<Agraph_t*> tocut;
    unsigned Lines;

    if (Options::verbose()) {
        cout << "Graph lines: " << GraphUtils::NumLines(graph)
             << ", nodes: " << agnnodes(graph) << endl;
    }

    subgraph = agfstsubg (graph);
    while (subgraph) {
        Lines = (unsigned) GraphUtils::NumLines(subgraph);
        if ((unsigned) agnnodes (subgraph) > thresholdNodes
            || (Lines > thresholdLines && agnnodes(subgraph) > 1))
        {
            tocut.push_back (subgraph);
        }
        subgraph = agnxtsubg (subgraph);
    }

    while ( !tocut.empty())
    {
        subgraph = tocut.back();

        sibling = cut_graph (subgraph, prefix);

        Lines = (unsigned) GraphUtils::NumLines(subgraph);
        if ((unsigned) agnnodes (subgraph) <= thresholdNodes    &&
            (Lines <= thresholdLines || agnnodes(subgraph) == 1) ) {
            if (Options::verbose())
                cout << agnameof (subgraph) << " subgraph lines: " 
                     << Lines << ",   nodes: " 
                     << agnnodes (subgraph) << endl;
            tocut.pop_back();
        }

        Lines = (unsigned) GraphUtils::NumLines(sibling);
        if ((unsigned) agnnodes (sibling) > thresholdNodes   ||
            (Lines > thresholdLines && agnnodes(sibling) > 1) ) {
            tocut.push_back (sibling);
        }
        else
        {
            if (Options::verbose())
                cout << agnameof (sibling) << " sibling lines:  " 
                     << Lines << ",   nodes: " 
                     << agnnodes (sibling) << endl;
        }
    }
}

Agraph_t*
MinCut::partition (Agraph_t* graph)
{
    double MQ, Q;
    unsigned thresholdNodes = Options::rpuMaxFunctions();
    unsigned thresholdLines = Options::rpuMaxLines();

    min_cuts (graph, Options::rpuPrefix(), thresholdNodes, thresholdLines);
    if (Options::keep_graph())
        GraphUtils::UpdateEdgeStyles(graph);
    GraphUtils::MoveEdgesToSubgraphs(graph);
    if (Options::verbose()) {
        MQ = GraphUtils::CalculateMQ(graph);
        Q = GraphUtils::CalculateQ(graph);
        std::cout << "Modulation Quality: " << MQ << std::endl;
        std::cout << "Modularity: " << Q << std::endl;
    }

    return graph;
}
