//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>

#include <cstdlib>
#include <iostream>
#include <sstream>

#include "graph/graphutils.h"
#include "graph/prune.h"
#include "utils/options.h"

using namespace std;
using namespace aap;
using namespace GraphUtils;

static char kLabel[] = "label";
static int numPruned = 0;

static void
addTimeToLabel (Agnode_t* node, Agnodeinfo_t* nodeInfo)
{
    ostringstream oss;
    oss << agget(node, kLabel) << ":" << nodeInfo->time;
    char* label = strdup(oss.str().c_str());
    agset(node, kLabel, label);
    free (label);
}

static Agnode_t*
parentNode (Agraph_t* graph, Agnode_t* node)
{
    Agedge_t* inedge = agfstin (graph, node);
    if (inedge == NULL) return NULL;
    return agtail (inedge);
}

Agraph_t*
Prune::partition (Agraph_t* graph)
{
    // make sure we're deleting from the root graph
    graph = agroot (graph);

    Agraphinfo_t* graphInfo = GraphUtils::getInfo (graph);
    double threshold = Options::pruneThreshold() * graphInfo->total_time;

    Agnode_t* node = agfstnode (graph);

    while (node) {
        Agnodeinfo_t* nodeInfo = GraphUtils::getInfo (node);
        if (0) addTimeToLabel (node, nodeInfo);

        // Get the next node before pruning the current one from the tree
        Agnode_t* next = agnxtnode (graph, node);
        while (nodeInfo->time < threshold && (agfstout (graph, node) == NULL)) {
            // Check the parent too in case we've already passed it by
            Agnode_t* parent = parentNode (graph, node);
            agdelnode (graph, node);
            ++numPruned;
            if (!parent) break;
            node = parent;
            nodeInfo = GraphUtils::getInfo (node);
            // Avoid checking the next node twice (since it may be removed)
            if (node == next) next = agnxtnode (graph, node);
        }
        node = next;
    }

    if (Options::verbose())
        cout << "Pruned " << numPruned << " nodes" << endl;

    return graph;
}
