//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <limits.h>
#include <stdint.h>
#include <sys/wait.h>

#include <algorithm>
#include <iostream>
#include <sstream>
#include <vector>

#include "graph/graph.h"
#include "graph/graphutils.h"
#include "graph/recombine.h"
#include "utils/options.h"
#include "sourcefile.h"
#include "sourcelocation.h"

using namespace std;
using namespace aap;

static char kLabel[] = "label";

typedef std::vector<Agedge_t*> edge_v;

// Find all edges that are between subgraphs, and combine the
// two subgraphs it connects, if they are small enough
static void
recombine (Agraph_t* graph, unsigned thresholdNodes, unsigned thresholdLines)
{
    // build a set of edges leading into each subgraph:
    edge_v combine;
    Agraph_t* subgraph = agfstsubg (graph);
    while (subgraph) {
        Agnode_t* root = GraphUtils::treeRoot (subgraph);
        Agedge_t* edge = agfstin (graph, root);
        if (edge) combine.push_back (edge);
        subgraph = agnxtsubg (subgraph);
    }

    // Recombine edges in descending order of weights
    // STL sort orders lowest element first, so we need to iterate backwards
    sort(combine.begin(), combine.end(), GraphUtils::edge_compare);
    edge_v::reverse_iterator erit, edgeEnd;
    edgeEnd = combine.rend();
    for (erit = combine.rbegin(); erit != edgeEnd; ++erit) {
        Agedge_t* edge = *erit;

        Agnodeinfo_t *headInfo = GraphUtils::getInfo (aghead (edge));
        Agnodeinfo_t *tailInfo = GraphUtils::getInfo (agtail (edge));
        
        Agraph_t* head_graph = headInfo->subg;
        Agraph_t* tail_graph = tailInfo->subg;

        unsigned totalNodes = agnnodes(head_graph) + agnnodes(tail_graph);
        unsigned totalLines = GraphUtils::NumLines(head_graph)
            + GraphUtils::NumLines(tail_graph);
        if ( totalNodes <= thresholdNodes && totalLines <= thresholdLines) {
            if (Options::verbose()) {
                cout << "Joining " << agget(head_graph, kLabel)
                     << " to " << agget(tail_graph, kLabel) << "\n";
            }
            GraphUtils::move_tree (aghead (edge), head_graph, tail_graph);
            // We might like to remove the subgraph at this point, but doing so
            // leads to an error when agclose() is called.
        }
    }
}

Agraph_t*
Recombine::partition (Agraph_t* graph)
{
    unsigned thresholdNodes = Options::rpuMaxFunctions();
    unsigned thresholdLines = Options::rpuMaxLines();

    GraphUtils::dump_graph (graph, Options::outPath() + "/aap.pre-recombine.dot");

    recombine (graph, thresholdNodes, thresholdLines);

    return graph;
}
