//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <iostream>
#include <fstream>
#include <sstream>
#include "SloDatabase.h"
#include "graph/graphutils.h"
#include "sourcefile.h"
#include "utils/options.h"

using namespace aap;

static char StrReuse[]  = "Reuse Graph";
static char StrFile[]   = "file";
static char StrLabel[]  = "label";

static llvm::sys::Path defaultPath("./");
llvm::cl::opt<llvm::sys::Path, false, llvm::cl::parser<std::string> >
SloDatabase::profile("dbslo",
                     llvm::cl::desc("SLO database directory (default: ./)"),
                     llvm::cl::init(defaultPath));

static llvm::sys::Path defaultSourcePath_("./");
llvm::cl::opt<llvm::sys::Path, false, llvm::cl::parser<std::string> >
SloDatabase::sourceLocation(
    "s", llvm::cl::init(defaultSourcePath_),
    llvm::cl::desc("Source location for SLO profiled code"));

static llvm::cl::alias sourceloc_long(
    "slo-source", llvm::cl::desc("alias of -s"),
    llvm::cl::aliasopt(SloDatabase::sourceLocation));


/// Source Code Information gathered from SLO database
class aap::SloFile
{
public:
  SloFile (std::string&);
  ~SloFile (void);
  std::string GetName (void);
  int ProcessFile (void);
  SloProcedure* GetAccessProcedure (int);
  void Print (std::ofstream);
private:
  int ProcessFunctionInfo  (void);
  int ProcessMemAccessInfo (void);
  std::string Name;
  std::vector<SloProcedure*> Procedures;
  std::map<int,SloProcedure*> MemoryAccess;
};

/// Memory Reuse among two functions
class aap::SloReuse
{
public:
  SloReuse (SloProcedure*, SloProcedure*, int);
  SloProcedure *GetFirstProcedure (void);
  SloProcedure *GetSecndProcedure (void);
  int GetCount (void);
  void IncreaseCount (int);
  void Print (void);
private:
  SloProcedure *First;
  SloProcedure *Secnd;
  int Count;
};

SloFile::SloFile (std::string& Name) { this->Name = Name; }
SloFile::~SloFile (void)
{
  MemoryAccess.clear();
  Procedures.clear();
  MemoryAccess.clear();
}

int SloFile::ProcessFunctionInfo (void)
{
  int MinBB, MaxBB;
  char TempString[256], c, del;
  SloProcedure *Function;
  std::ifstream ReadFile;
  std::string Name;
  Name = this->Name + ".function_info";
  ReadFile.open(Name.c_str(), std::ifstream::in);
  while (ReadFile.good())
    {
      while ((((char)ReadFile.get()) != '(') && ReadFile.good());
      if ((ReadFile.good()) == false) break;
      while (((char)ReadFile.get()) != '"');
      ReadFile.get(TempString, 128, '"');
      Name = TempString;
      Function = new SloProcedure(Name,this->Name);
      while (((char)ReadFile.get()) != '(');
      MinBB = 0;
      while ((del=(char)ReadFile.get()) != '=')
	{
	  MinBB *= 10;
	  MinBB += (int) (del - '0');
	}
      if (MinBB == 0)
	{
	  while (((char)ReadFile.get()) != '(');
	  while ((del=(char)ReadFile.get()) != '=')
	    {
	      MinBB *= 10;
	      MinBB += (int) (del - '0');
	    }
	}
      MaxBB = MinBB;
      while (((c = ReadFile.get()) != '\n') && (ReadFile.good()))
	{
	  if (c == '(')
	    {
	      MaxBB = 0;
	      while ((del=(char)ReadFile.get()) != '=')
		{
		  MaxBB *= 10;
		  MaxBB += (int) (del - '0');
		}
	    }
	}
      Function->SetMinMaxBB(MinBB, MaxBB);
      Procedures.push_back(Function);
      Function = NULL;
    }
  ReadFile.close();
  return 0;
}

int SloFile::ProcessMemAccessInfo (void)
{
  int RefAccess, RefBB;
  unsigned int i;
  char c;
  bool Stop;
  SloProcedure *Function;
  std::ifstream ReadFile;
  std::string Name;
  Name = this->Name + ".memaccess_info";
  ReadFile.open(Name.c_str(), std::ifstream::in);
  while (ReadFile.good())
    {
      while ((((char)ReadFile.get()) != '(') && ReadFile.good());
      if ((ReadFile.good()) == false) break;
      ReadFile >> RefAccess;
      ReadFile.get();
      c = (char) ReadFile.get();
      while (c == '\n')
	{
	  while (((char)ReadFile.get()) != ')');
	  while (((char)ReadFile.get()) != ')');
	  c = (char) ReadFile.get();
	}
      RefBB = 0;
      while ((c = (char)ReadFile.get()) != ')')
	{
	  RefBB *= 10;
	  RefBB += (int) (c - '0');
	}
      Function = NULL;
      Stop = false;
      i = 0;
      while ((Stop==false) && (i < Procedures.size()))
	{
	  if ((RefBB >= (Procedures[i]->GetMinBB())) &&
	      (RefBB <= (Procedures[i]->GetMaxBB())) )
	    {
	      Function = Procedures[i];
	      Stop = true;
	    }
	  i++;
	}
      MemoryAccess.insert(std::pair<int,SloProcedure*>(RefAccess,Function));
    }
  ReadFile.close();
  return 0;
}

int SloFile::ProcessFile (void)
{
  ProcessFunctionInfo();
  ProcessMemAccessInfo();
  return 0;
}

SloProcedure* SloFile::GetAccessProcedure (int RefNum)
{
  std::map<int,SloProcedure*>::iterator iter;
  iter=MemoryAccess.find(RefNum);
  if (iter != MemoryAccess.end())  return iter->second;
  return NULL;
}

std::string SloFile::GetName (void) { return Name; } 

void SloFile::Print (std::ofstream PF)
{
  PF << "===================================" << std::endl;
  PF << "Source File: " << Name << std::endl;
  for (unsigned int i=0; i<Procedures.size(); i++)
    Procedures[i]->Print();
  PF << "-----------------------------------" << std::endl;
  std::map<int,SloProcedure*>::iterator iter;
  for (iter = MemoryAccess.begin(); iter != MemoryAccess.end(); iter++)
    PF << "Reference: " << iter->first << " => "
       << "Function: "  << iter->second->name() << std::endl;
}

SloReuse::SloReuse (SloProcedure *First, SloProcedure *Secnd, int Count)
{
  this->First = First;
  this->Secnd = Secnd;
  this->Count = Count;
}
SloProcedure* SloReuse::GetFirstProcedure (void)   {  return First;  }
SloProcedure* SloReuse::GetSecndProcedure (void)   {  return Secnd;  }
int  SloReuse::GetCount (void)   {  return Count;  }
void SloReuse::IncreaseCount (int Count)    {  this->Count += Count;  }
void SloReuse::Print (void)
{
  std::cout << "< " << First->name() << " , " << Secnd->name()
	    << " , " << Count << " >" << std::endl;
}



SloDatabase::SloDatabase (const llvm::sys::Path& PathToBRD)
{
  this->MemoryDatabase = PathToBRD;
  this->MemoryDatabase.appendComponent("BRD");
}

SloDatabase::~SloDatabase (void)
{
  std::map<int,SloFile*>::iterator iter;
  for (iter=FileMap.begin(); iter!=FileMap.end(); iter++)
    delete iter->second;
  for (unsigned int i=0; i<MemoryReuse.size(); i++)
    delete MemoryReuse[i];
  FileMap.clear();
  MemoryReuse.clear();
  SubGraphID = 0;
}

int SloDatabase::ProcessBRD (const llvm::sys::Path& SourcePath)
{
  std::ifstream BRDFile;
  std::string Name, SourceLoc;
  int Value, RefCount;
  unsigned int CurPos;
  char TempString[256], c;
  SloFile *CurFile;
  bool Stop;
  SloReuse *CurReuse;
  SloProcedure *First, *Secnd;
  BRDFile.open(MemoryDatabase.c_str(), std::ifstream::in);
  if ((BRDFile.good()) == false) return -1;

  while (((char)BRDFile.get()) != 'f');
  while (((char)BRDFile.get()) != ' ');

  // Determine File Offset Value
  Value = 0;
  while ((c=(char)BRDFile.get()) != ')')
    {
      Value *= 10;
      Value += (int) (c - '0');
    }
  Offset = Value;

  // Determine and create project files
  SourceLoc = SourcePath.str();
  c = (*(SourceLoc.rbegin()));
  if (c != '/')
      SourceLoc.append("/");
  while ((c=(char)BRDFile.get()) != '"');
  while (c == '"')
    {
      BRDFile.get(TempString, 255, '"');
      Name = SourceLoc;
      Name += TempString;
      CurFile = new SloFile (Name);
      BRDFile.get();
      BRDFile.get();
      Value = 0;
      while ((c=(char)BRDFile.get()) != ')')
        {
          Value *= 10;
          Value += (int) (c - '0');
        }
      CurFile->ProcessFile();
      FileMap.insert(std::pair<int,SloFile*>(Value,CurFile));
      CurFile = NULL;
      while (((char)BRDFile.get()) != '(');
      c = (char)BRDFile.get();
    }

  // Memory Reuses
  while (((char)BRDFile.get()) != 'u');
  while (((char)BRDFile.get()) != '\n');
  c = (char) BRDFile.get();
  while (c == ' ')
    {
      while (((char)BRDFile.get()) != 'c');
      while (((char)BRDFile.get()) != '(');
      Value = 0;
      while ((c = (char)BRDFile.get()) != ')')
	{
          Value *= 10;
          Value += (int) (c - '0');
	}
      RefCount = Value;
      while (((char)BRDFile.get()) != 'd');
      while (((char)BRDFile.get()) != ' ');
      BRDFile >> Value;
      First = FileMap[(Value/Offset)]->GetAccessProcedure(Value%Offset);
      while (((char)BRDFile.get()) != ' ');
      while (((char)BRDFile.get()) != ' ');
      Value = 0;
      while ((c = (char)BRDFile.get()) != ')')
        {
          Value *= 10;
          Value += (int) (c - '0');
        }
      Secnd = FileMap[(Value/Offset)]->GetAccessProcedure(Value%Offset);
      if ((First != NULL) && (Secnd != NULL) && (First != Secnd))
	{
	  Stop = false;
	  CurPos = 0;
	  while ((Stop == false) && (CurPos < MemoryReuse.size()))
	    {
	      CurReuse = MemoryReuse[CurPos];
	      if ( ( (First == (CurReuse->GetFirstProcedure())) &&
		     (Secnd == (CurReuse->GetSecndProcedure())) )  ||
		   ( (Secnd == (CurReuse->GetFirstProcedure())) &&
		     (First == (CurReuse->GetSecndProcedure())) )  )
		{
		  Stop = true;
		  CurReuse->IncreaseCount(RefCount);
		}
	      else
		CurPos++;
	    }
	  if (Stop == false)
	    {
	      CurReuse = new SloReuse(First,Secnd,RefCount);
	      MemoryReuse.push_back(CurReuse);
	    }
	}
      c = (char) BRDFile.get();
      while (c != ')')
	{
	  while (((char)BRDFile.get()) != ')');
	  c = (char) BRDFile.get();
	}
      while (((char)BRDFile.get()) != '\n');
      c = (char) BRDFile.get();
    }
  BRDFile.close();
  return 0;
}

void SloDatabase::PrintBRD (void)
{
  std::map<int,SloFile*>::iterator iter;
  std::cout << "File Offset: " << Offset << std::endl;
  for (iter = FileMap.begin(); iter != FileMap.end(); iter++)
    std::cout << iter->first << ") " << iter->second->GetName() << std::endl;
  for (unsigned int i=0; i<MemoryReuse.size(); i++)
    MemoryReuse[i]->Print();
}

Agraph_t* SloDatabase::GetGraph (void) { return SloGraph; }

static Agnode_t*
getOrCreateProcedureNode(std::map<SloProcedure*,Agnode_t*>& ProcNode,
                         SloProcedure *proc,
                         Agraph_t* SubGraph)
{
    std::string NodeName;
    char *name, *fileName;
    std::map<SloProcedure*,Agnode_t*>::iterator iter = ProcNode.find(proc);
    if (iter != ProcNode.end())
        return iter->second;

    NodeName = proc->name();
    NodeName.append("|");
    NodeName.append(proc->file());
    name = strdup(NodeName.c_str());
    Agnode_t* node = GraphUtils::createNode(SubGraph, name);
    Agnodeinfo_t* info = GraphUtils::getInfo (node);
    info->procedure = (Procedure*)proc;
    free(name);
    name = strdup(proc->name().c_str());
    agset(node,StrLabel, name);
    fileName = strdup(proc->file().c_str());
    agset(node, StrFile, fileName);
    ProcNode.insert(std::pair<SloProcedure*,Agnode_t*>(proc,node));
    free(name);
    free(fileName);
    return node;
}

void SloDatabase::CreateGraph (void)
{
  Agraph_t *SubGraph;
  Agnode_t *FirstNode, *SecndNode;
  SloReuse *Reuse;
  Agedge_t *Edge;
  Agedgeinfo_t *EdgeInfo;
  std::map<SloProcedure*,Agnode_t*> ProcNode;
  std::map<SloProcedure*,Agnode_t*>::iterator iter;

  SloGraph = agopen(StrReuse, Agstrictundirected, &AgDefaultDisc);
  GraphUtils::initializeGraph (SloGraph);
  char* name = strdup("temp");
  SubGraph = agsubg(SloGraph, name, TRUE);
  free(name);
  for (unsigned int i=0; i<MemoryReuse.size(); i++)
  {
      Reuse = MemoryReuse[i];
      FirstNode = getOrCreateProcedureNode(ProcNode,
                                           Reuse->GetFirstProcedure(),
                                           SubGraph);
      SecndNode = getOrCreateProcedureNode(ProcNode,
                                           Reuse->GetSecndProcedure(),
                                           SubGraph);

      Edge = GraphUtils::createEdge(SubGraph, FirstNode, SecndNode);
      std::ostringstream Stream;
      char *Label;
      Stream << Reuse->GetCount();
      Label = strdup(Stream.str().c_str());
      agset(Edge, StrLabel, Label);
      EdgeInfo = GraphUtils::getInfo(Edge);
      EdgeInfo->reuse  = (long) Reuse->GetCount();
      EdgeInfo->weight = EdgeInfo->reuse;
      EdgeInfo->calls  = 0;
      EdgeInfo->NumTaken = 0;
  }
  ProcNode.clear();
}

void SloDatabase::CreateInitialSubGraphs (void)
{
  char *Name, *Label;
  Agraph_t* OrgSubGraph;
  Agraph_t* Subgraph;
  Agnode_t* Node;
  OrgSubGraph = agfstsubg(SloGraph);
  SubGraphID = 0;
  Node = agfstnode(OrgSubGraph);
  while (Node)
    {
      std::ostringstream Stream;
      Stream << Options::rpuPrefix() << SubGraphID++;
      Name  = strdup(Stream.str().c_str());
      Label = strdup(Stream.str().c_str());
      Subgraph = agsubg (SloGraph, Name, TRUE);
      agset(Subgraph, StrLabel, Label);
      GraphUtils::MoveConnected(Node, OrgSubGraph, Subgraph);
      Node = agfstnode(OrgSubGraph);
    }
  agdelsubg(SloGraph,OrgSubGraph);
  for (Subgraph = agfstsubg(SloGraph); Subgraph != NULL;
       Subgraph = agnxtsubg(Subgraph))
    {
      Node = agfstnode(Subgraph);
      GraphUtils::SetDepth(Node,1);
      GraphUtils::CreateDepth(Node,Subgraph);
    }
}

void SloDatabase::IncludeHpcInfo (Agraph_t* HpcGraph)
{
    Agraph_t *SloSubgraph, *HpcSubgraph;
    Agnode_t *Src, *Dst, *HpcNode, *HpcHead, *HpcTail;
    Agedge_t *Edge, *HpcEdge;
    Agnodeinfo_t *SrcInfo, *DstInfo, *HpcHeadInfo, *HpcTailInfo;
    Agedgeinfo_t *EdgeInfo, *HpcEdgeInfo;
    std::string SloFile, HpcFile, SrcName, DstName;
    SloProcedure *Function;

    if (HpcGraph == NULL) return;
    if (SloGraph == NULL) return;

    SloSubgraph = agfstsubg(SloGraph);
    HpcSubgraph = agfstsubg(HpcGraph);
    
    // Add features from HPC graph to the SLO graph.
    for (Src = agfstnode(SloSubgraph); Src != NULL;
         Src = agnxtnode(SloSubgraph,Src))
        for (Edge = agfstedge(SloSubgraph,Src); Edge != NULL;
             Edge = agnxtedge(SloSubgraph,Edge,Src)) {
            Dst = GraphUtils::DestinedNode(Src,Edge);
            SrcInfo = GraphUtils::getInfo(Src);
            DstInfo = GraphUtils::getInfo(Dst);
            SloFile = SrcInfo->procedure->file();
            SrcName = SrcInfo->procedure->name();
            DstName = DstInfo->procedure->name();
            for (HpcTail = agfstnode(HpcSubgraph); HpcTail != NULL;
                 HpcTail = agnxtnode(HpcSubgraph,HpcTail)) {
                HpcTailInfo = GraphUtils::getInfo(HpcTail);
                HpcFile = HpcTailInfo->procedure->file();
                if (HpcFile.find(SloFile) != std::string::npos)
                    if (SrcName.compare(HpcTailInfo->procedure->name())
                        == 0) {
                        for (HpcEdge  = agfstout(HpcSubgraph,HpcTail);
                             HpcEdge != NULL;
                             HpcEdge  = agnxtout(HpcSubgraph,HpcEdge)) {
                            HpcHead = aghead(HpcEdge); 
                            HpcHeadInfo = GraphUtils::getInfo(HpcHead);
                            if (DstName.compare(HpcHeadInfo->procedure
                                                ->name()) == 0) {
                                EdgeInfo = GraphUtils::getInfo(Edge);
                                HpcEdgeInfo = GraphUtils::getInfo(HpcEdge);
                                // Add call count to the SLO edge info.
                                // There could multiple nodes and edges
                                // between same functions.
                                EdgeInfo->NumTaken += HpcEdgeInfo->weight;
                                HpcEdgeInfo->NumTaken++;
                            }
                        }
                    }
            }
        }

    for (Src = agfstnode(SloSubgraph); Src != NULL;
         Src = agnxtnode(SloSubgraph,Src))
        for (Edge = agfstedge(SloSubgraph,Src); Edge != NULL;
             Edge = agnxtedge(SloSubgraph,Edge,Src)) {
            std::ostringstream Stream;
            char *Label;
            EdgeInfo = GraphUtils::getInfo(Edge);
            if (EdgeInfo->NumTaken) {
                // Multiply the edge weight with number of calls+1.
                EdgeInfo->weight = EdgeInfo->weight*(EdgeInfo->NumTaken+1);
                Stream << EdgeInfo->weight;
                Label = strdup(Stream.str().c_str());
                agset(Edge, StrLabel, Label);
                EdgeInfo->calls = EdgeInfo->NumTaken;
                EdgeInfo->NumTaken = 0;
            }
        }

    // Include addition nodes and/or edges gathered from HPC graph.
    for (HpcTail = agfstnode(HpcSubgraph); HpcTail != NULL;
         HpcTail = agnxtnode(HpcSubgraph,HpcTail))
        for (HpcEdge = agfstout(HpcSubgraph,HpcTail); HpcEdge != NULL;
             HpcEdge = agnxtout(HpcSubgraph,HpcEdge)) {
            HpcEdgeInfo = GraphUtils::getInfo(HpcEdge);
            if (HpcEdgeInfo->NumTaken == 0) {
                HpcHead = aghead(HpcEdge);
                HpcTailInfo = GraphUtils::getInfo(HpcTail);
                HpcHeadInfo = GraphUtils::getInfo(HpcHead);
                std::string NodeName;
                char *Name;
                NodeName = HpcTailInfo->procedure->name();
                NodeName.append("|");
                HpcFile = HpcTailInfo->procedure->file();
                SloFile = HpcFile;
                HpcFile = "./src";
                SloFile.erase(0,HpcFile.size());
                NodeName.append(SloFile);
                Name = strdup(NodeName.c_str());
                Src = agnode(SloSubgraph,Name,false);
                // No function in SLO graph. Create it!.
                if (Src == NULL) {
                    Src = GraphUtils::createNode(SloSubgraph,Name);
                    free(Name);
                    Name = strdup(HpcTailInfo->procedure->name().c_str());
                    Function = new SloProcedure(Name,SloFile);
                    SrcInfo = GraphUtils::getInfo(Src);
                    SrcInfo->procedure = (Procedure*)Function;
                    agset(Src,StrLabel,Name);
                    free(Name);
                    Name = strdup(SloFile.c_str());
                    agset(Src,StrFile,Name);
                    Function = NULL;
                }
                free(Name);
                NodeName = HpcHeadInfo->procedure->name().c_str();
                NodeName.append("|");
                HpcFile = HpcHeadInfo->procedure->file();
                SloFile = HpcFile;
                HpcFile = "./src";
                SloFile.erase(0,HpcFile.size());
                NodeName.append(SloFile);
                Name = strdup(NodeName.c_str());
                Dst = agnode(SloSubgraph,Name,false);
                // No function in SLO graph. Create it!.
                if (Dst == NULL) {
                    Dst = GraphUtils::createNode(SloSubgraph,Name);
                    free(Name);
                    Name = strdup(HpcHeadInfo->procedure->name().c_str());
                    Function = new SloProcedure(Name,SloFile);
                    DstInfo = GraphUtils::getInfo(Dst);
                    DstInfo->procedure = (Procedure*)Function;
                    agset(Dst,StrLabel,Name);
                    free(Name);
                    Name = strdup(SloFile.c_str());
                    agset(Dst,StrFile,Name);
                    Function = NULL;
                }
                free(Name);
                std::ostringstream Stream;
                // SLO graph edge is available between both nodes.
                // Add HPC graph edge info to the available SLO graph edge.
                if ((Edge = agedge(SloSubgraph,Src,Dst,NULL,false))) {
                    EdgeInfo = GraphUtils::getInfo(Edge);
                    EdgeInfo->weight += HpcEdgeInfo->weight;
                    EdgeInfo->calls = EdgeInfo->weight;
                    Stream << EdgeInfo->weight;
                    Name = strdup(Stream.str().c_str());
                    agset(Edge, StrLabel, Name);
                    free(Name);
                }
                // No edge between both nodes.
                // Create an edge add feautures from HPC graph edge.
                else {
                    Edge = GraphUtils::createEdge(SloSubgraph,Src,Dst);
                    EdgeInfo = GraphUtils::getInfo(Edge);
                    EdgeInfo->weight = HpcEdgeInfo->weight;
                    EdgeInfo->calls  = HpcEdgeInfo->weight;
                    EdgeInfo->reuse  = 0;
                    EdgeInfo->NumTaken = 0;
                    Stream << EdgeInfo->weight;
                    Name = strdup(Stream.str().c_str());
                    agset(Edge, StrLabel, Name);
                    free(Name);
                }
            }
        }
}
