//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <cstdio>
#include <iostream>
#include <sstream>
#include <string>

#include <sysexits.h>

#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/util/XMLString.hpp>

#include "config/llvm.h"

#include "sourcefile.h"
#include "graph/graphutils.h"
#include "hpcdatabase.h"
#include "hpcprocedure.h"
#include "utils/options.h"

using namespace std;
using namespace xercesc;
using namespace aap;

// if we want to print the graph, we need to include character attributes on
// the graph elements. this flag indicates this preference
static const bool iolabels = true;

static const XMLCh kF[] = {'f','\0'};
static const XMLCh kI[] = {'i','\0'};
static const XMLCh kL[] = {'l','\0'};
static const XMLCh kN[] = {'n','\0'};
static const XMLCh kS[] = {'s','\0'};
static const XMLCh kV[] = {'v','\0'};

static const string kUnknownFile = "~unknown-file~";
static const string kUnknownProc = "~unknown-proc~";

static char kCall[]   = "call";
static char kFile[]   = "file";
static char kLabel[]  = "label";
static char kLine[]   = "line";
static char kRetcnt[] = "retcnt";
static char kTime[]   = "time";

static string
append_to_cwd (const string& str)
{
    llvm::sys::Path path = llvm::sys::Path::GetCurrentDirectory();
    path.appendComponent(str);
    return path.str();
}

static string
get_absolute_path (const string& str)
{
    if (str.at (0) == '/') return str;
    else return append_to_cwd (str);
}

static int convert_string(char* s)
{
    char *endptr = NULL;
    int num = strtol(s, &endptr, 10);

    if (endptr != NULL) {
        float f;
        sscanf(s, "%e", &f);
        num = (int)f;
    }

    return num;
}

/// Default path to the HPC Toolkit profile database.
static llvm::sys::Path defaultPath("hpctoolkit-database/");

llvm::cl::opt<llvm::sys::Path, false, llvm::cl::parser<std::string> >
HpcDatabase::profile(llvm::cl::Positional,
                     llvm::cl::desc("[HPC database]"),
                     llvm::cl::init(defaultPath));

static llvm::cl::alias
DBpathOpt_("hpctk",
           llvm::cl::ValueRequired,
           llvm::cl::desc("HPC database directory"),
           llvm::cl::aliasopt(HpcDatabase::profile));

Agraph_t*
HpcDatabase::readGraph (const std::string&)
{
    FILE* ingraph = fopen (Options::inputGraph().c_str(), "r");
    if (ingraph == NULL) {
        cerr << "failed to open graph input "
             << Options::inputGraph() << endl;
        exit(EX_NOINPUT);
    }
    Agraph_t* graph = agread (ingraph, &AgDefaultDisc);
    GraphUtils::initializeGraph (graph);
    Agraph_t* subgraph = agfstsubg (graph);
    assert (subgraph && "A graph should always contain an RPU subgraph");
    while (subgraph) {
        Agnode_t* node = agfstnode (subgraph);
        Agnodeinfo_t* nodeInfo;
        // iterate over the nodes and edges and set the internal attributes
        while (node != NULL) {
            char* file  = agget (node, kFile);
            char* label = agget (node, kLabel);
            GraphUtils::initializeNode (node, subgraph);
            nodeInfo = GraphUtils::getInfo (node);
            nodeInfo->procedure = new HpcProcedure(label,file);
            node = agnxtnode (subgraph, node);
        }
        subgraph = agnxtsubg (subgraph);
    }
    return graph;
}

/// Strictly speaking, the representation is not constructed until some action
/// (such as graph generation) is taken.
HpcDatabase::HpcDatabase(const llvm::sys::Path& path)
{
  path_ = path;
  path_.makeAbsolute();
  defaultPath.makeAbsolute();
  stackUid = 0;
  subgUid_ = 0;
  HpcProcedure* proc = new HpcProcedure ("","");
  proc->setShouldOutput (false);
  unknownProc_ = ProcedureRef (proc);
  // we don't need to retain unknownProc_. The allocation does so, and it will
  // be released in the deconstructor
  nameLookup_[kUnknownProc] = unknownProc_;
  callStack_.push (0);
}

Agraph_t*
HpcDatabase::generateGraph (void)
{
  llvm::sys::Path input = path_;
  input.appendComponent("experiment.xml");

  if (!input.exists()) {
    // Silently ignore a missing default profile.
    if (profile.getNumOccurrences() == 0 &&
        DBpathOpt_.getNumOccurrences() == 0 &&
        path_ == defaultPath)
    {
      return NULL;
    }
    cerr << "Fatal error: HPC profile " << input.str() << " not found\n";
    exit(EX_NOINPUT);
  }

  graph_ = agopen (kCall, Agstrictdirected, &AgDefaultDisc);
  GraphUtils::initializeGraph (graph_);

  try {
    XMLPlatformUtils::Initialize();
    SAX2XMLReader* reader = XMLReaderFactory::createXMLReader();
    reader->setContentHandler(this);
    reader->setErrorHandler(this);
    reader->parse (input.str().c_str());
    delete reader;
    XMLPlatformUtils::Terminate();
  } catch (...) {
    cout << "Unexpected Exception" << endl;
  }

  return graph_;
}

std::string*
HpcDatabase::pathToFile (const string& path) const
{
  string* result = new string();

  int pos = path.find ("./src", 0);
  if (0==pos)
  {
    (*result) = path.substr (5);
  } else {
    (*result) = path_.str();
    result->append("/");
    result->append(path);
  }

  return result;
}

void
HpcDatabase::popStacks(void)
{
    assert (!procStack_.empty());
    Agnode_t* function = procStack_.top();
    Agnodeinfo_t* nodeInfo = GraphUtils::getInfo (function);
    procStack_.pop();
    // if stack top was an output function, pop if from the output stack too
    HpcProcedure* HpcProc
      = llvm::dyn_cast<HpcProcedure>(nodeInfo->procedure);
    if (HpcProc && HpcProc->shouldOutput()) {
        assert (!outStack_.empty());
        outStack_.pop();
        // if we've gotten to a non-output function, then we've reached the
        // root of the output subgraph (so pop the sub graph stack)
        if (!procStack_.empty()) {
            Agnodeinfo_t* topInfo = GraphUtils::getInfo (procStack_.top());
            HpcProcedure* topProc =
              llvm::dyn_cast<HpcProcedure>(topInfo->procedure);
            if (!(topProc && topProc->shouldOutput())) {
                assert (!subgStack_.empty());
                subgStack_.pop();
            }
        }
    } else {
        agdelete (graph_, function);
    }
}

void 
HpcDatabase::endElement(const XMLCh*,// uri, 
                      const XMLCh* localname, 
                      const XMLCh*)// qname)
{
  char* name = XMLString::transcode (localname);
  if (0 == strcasecmp (name, "pf"))
      popStacks();
  else if (0 == strcasecmp (name, "c")) {
      assert (!callStack_.empty());
      callStack_.pop();
  }
  XMLString::release (&name);
}

static Agnode_t*
lookupTwin (Agnode_t* parent, unsigned site, ProcedureRef proc)
{
    // look to see if there is a call from parent to a like named function at
    // the same call site

    Agedge_t* edge;
    Agraph_t* graph = agraphof (parent);
    edge = agfstout (graph, parent);
    while (edge) {
        Agnodeinfo_t* nodeInfo = GraphUtils::getInfo (aghead (edge));
        // the edge head is the child (where the arrowhead would be)
        if (nodeInfo->procedure == proc && nodeInfo->staticSite == site)
            return aghead (edge);

        edge = agnxtout (graph, edge);
    }

    return NULL;
}

// Creates a call (graph edge) to the given function (node) using the top of
// the stack (only of functions that should be output) as the caller.
void
HpcDatabase::createCall(Agnode_t* function)
{
    Agedge_t* edge;
    Agraph_t* node_graph;       // where to put the node
    Agraph_t* edge_graph;       // where to put the edge
    Agnodeinfo_t* functionInfo = GraphUtils::getInfo (function);
    HpcProcedure* functionProc
      = llvm::dyn_cast<HpcProcedure>(functionInfo->procedure);
    if (functionProc && functionProc->shouldOutput())
    {
        if (agfstin (graph_, function) == NULL) 
        {
            if (procStack_.empty()){
                addSubgraph();
                edge_graph = graph_; // the edge belongs to the master graph
            } else {
                Agnodeinfo_t* topInfo = GraphUtils::getInfo (procStack_.top());
                HpcProcedure* topProc =
                  llvm::dyn_cast<HpcProcedure>(topInfo->procedure);
                if (!(topProc && topProc->shouldOutput())) {
                    addSubgraph();
                    edge_graph = graph_; // the edge belongs to the master graph
                } else {
                    edge_graph = subgStack_.top();
                }
            }
            node_graph = subgStack_.top();
            function = agsubnode (node_graph, function, TRUE);
            GraphUtils::getInfo (function)->subg = node_graph;
            if (!outStack_.empty())
                edge = GraphUtils::createEdge (edge_graph,
                                               outStack_.top(),
                                               function);
        }
        outStack_.push (function);
    }
    procStack_.push (function);
}

// Record the mapping from procedure id to name as in:
// <Procedure i = "4543" n="MiscUtils::atox(char*)"/>
void  
HpcDatabase::addProcedure (const xercesc::Attributes& attrs)
{

  const XMLCh* i    = attrs.getValue (kI);
  const XMLCh* n    = attrs.getValue (kN);
  char*  name = XMLString::transcode (n);
  unsigned int id;

  XMLString::textToBin (i,id);

  procedures_[id] = name;

  XMLString::release (&name);
}

void
HpcDatabase::addSubgraph (void)
{
    ostringstream oss;
    oss << Options::rpuPrefix() << subgUid_++;
    char* name  = strdup (oss.str().c_str());
    char* label = strdup (oss.str().c_str());
    Agraph_t* subgraph = agsubg (graph_, name, TRUE);
    agset(subgraph, kLabel, label);
    subgStack_.push (subgraph);
    free (name);
    free (label);
}

static char*
new_label(int id)
{
  std::ostringstream oss;
  oss << id;
  return strdup (oss.str().c_str());
}


// Return an existing procedure or create a new one.
ProcedureRef
HpcDatabase::lookupProcedure(string& name, string& file_name)
{
  HpcProcedure* proc;
  if (nameLookup_.count(name)) {
    proc = llvm::dyn_cast<HpcProcedure>(nameLookup_[name]);
    assert(proc && "found a non-hpc procedure in the hpc database");
    proc->incrementReferences();
  } else {
    proc = new HpcProcedure(name, file_name);
    if (string(kUnknownFile).compare (file_name) == 0)
      proc->setShouldOutput (false);
    nameLookup_[name] = ProcedureRef(proc);
  }
  return nameLookup_[name];
}

// When a procedure call is found we create an edge from the parent to the
// child. Because we only create graph nodes once a callee is seen, the child
// graph node may be created as well.
void  
HpcDatabase::procedureCall (const xercesc::Attributes& attrs)
{
  // <PF s="5780" l="107" lm="2" f="5202" n="5780">
  const XMLCh *l;
  unsigned int id, file_id, line;
  string name;
  string file_name;
  unsigned staticSite;
  Agnode_t* function = NULL;
  ProcedureRef proc;

  l = attrs.getValue (kL);

  XMLString::textToBin (attrs.getValue (kS), id);
  XMLString::textToBin (l, line);
  XMLString::textToBin (attrs.getValue (kF), file_id);

  name = procedures_[id];

  file_name = files[file_id];

  staticSite = callStack_.top();

  proc = lookupProcedure (name, file_name);

  if (!procStack_.empty())
      function = lookupTwin (procStack_.top(), staticSite, proc);

  if (function == NULL) {
      char* label = new_label (stackUid++);
      function = GraphUtils::createNode (graph_, label);
      Agnodeinfo_t* info = GraphUtils::getInfo (function);
      info->procedure = &(*proc);
      info->staticSite = staticSite;
      if (iolabels) {
          char* cL = XMLString::transcode (l);
          char* cName = strdup (name.c_str());
          char* cFileName = strdup (file_name.c_str());
          agset (function, kLabel, cName);
          agset (function, kFile,  cFileName);
          agset (function, kLine,  cL);
          XMLString::release (&cL);
          free (cFileName);
          free (cName);
      }
      free (label);
  }
  createCall (function);
}

void  
HpcDatabase::metricNode (const xercesc::Attributes& attrs)
{
  // <M n = "0" v="11996"/>
  assert (!procStack_.empty());

  Agnode_t*    function = procStack_.top();
  const XMLCh* n        = attrs.getValue (kN);
  unsigned int id;
  char*        v        = XMLString::transcode (attrs.getValue (kV));
  unsigned int value    = convert_string (v);

  Agnodeinfo_t* functionInfo;
  Agraphinfo_t* graphInfo;

  XMLString::textToBin (n, id);
  
  if (wallclockIndexes_.count(id) != 0) {
      graphInfo = GraphUtils::getInfo (graph_);
      if (graphInfo) graphInfo->total_time += value;
      functionInfo = GraphUtils::getInfo (function);
      functionInfo->time += value;
      if (iolabels) {
          ostringstream oss;
          oss << functionInfo->time;
          agset (function, kTime, strdup (oss.str().c_str()));
      }
  } else if (retCountIndexes_.count(id) != 0) {
      functionInfo = GraphUtils::getInfo (function);
      functionInfo->retcnt += value;
      if (iolabels) {
          ostringstream oss;
          oss << functionInfo->retcnt;
          agset (function, kRetcnt, strdup (oss.str().c_str()));
      }
      HpcProcedure* functionProc
        = llvm::dyn_cast<HpcProcedure>(functionInfo->procedure);
      if (functionProc) {
        functionProc->retcnt_++;
        functionProc->wallclock_ = functionInfo->time;
      }
  }

  XMLString::release (&v);
}

void  
HpcDatabase::addFile (const xercesc::Attributes& attrs)
{
  // <File i="18" n="./Joe_/UgpWithCvCompFlow.cpp"/>

  const XMLCh* i    = attrs.getValue (kI);
  const XMLCh* n    = attrs.getValue (kN);
  char*        name = XMLString::transcode (n);
  unsigned int id;

  XMLString::textToBin (i,id);

  files[id] = name;

  XMLString::release (&name);
}

void
HpcDatabase::registerMetric (const xercesc::Attributes& attrs)
{
    const XMLCh* i = attrs.getValue (kI);
    const XMLCh* n = attrs.getValue (kN);

    char* cId   = XMLString::transcode(i);
    char* cName = XMLString::transcode (n);

    int id = convert_string (cId);
    string name (cName);

    while (name != "") {
        if (isalpha(name[0]))
            break;
        else
            name = name.substr(name.find_first_of(".")+1);
    }

    if (name.substr(0, 9).compare("WALLCLOCK") == 0) {
        wallclockIndexes_.insert(id);
    } else if (name.substr(0, 6).compare("RETCNT") == 0) {
        retCountIndexes_.insert(id);
    }
    XMLString::release (&cId);
    XMLString::release (&cName);
}

static void
pushCallSite (const xercesc::Attributes& attrs, stack<unsigned>& callStack)
{
    unsigned int static_id;
    const XMLCh* s = attrs.getValue (kS);
    XMLString::textToBin (s, static_id);
    callStack.push (static_id);
}

void 
HpcDatabase::startElement(const XMLCh*,// uri, 
                        const XMLCh* localname, 
                        const XMLCh*,// qname, 
                        const xercesc::Attributes& attrs)
{
  char* name = XMLString::transcode (localname);

  if (0 == (strcasecmp (name, "c")))
      pushCallSite (attrs, callStack_);
  else if (0 == (strcasecmp (name,"pf")))
    procedureCall (attrs);
  else if (0 == (strcasecmp (name,"m")))
    metricNode (attrs);
  else if (0 == (strcasecmp (name,"file")))
    addFile (attrs);
  else if (0 == (strcasecmp (name,"procedure")))
    addProcedure (attrs);
  else if (0 == (strcasecmp (name, "metric")))
      registerMetric (attrs);

  XMLString::release (&name);
}

void 
HpcDatabase::warning(const xercesc::SAXParseException& excep)
{
    cerr << "warning: " << XMLString::transcode(excep.getMessage()) << endl;
}

void 
HpcDatabase::error(const xercesc::SAXParseException& excep)
{
    cerr << "error: " << XMLString::transcode(excep.getMessage()) << endl;
}

void
HpcDatabase::fatalError(const xercesc::SAXParseException& excep)
{
    cerr << "error: " << XMLString::transcode(excep.getMessage()) << endl;
    exit(EX_DATAERR);
}

void HpcDatabase::assignWeights  (Agraph_t* graph)
{
    Agnode_t* node = agfstnode (graph);
    Agedgeinfo_t* edgeInfo;
    Agnode_t* head;
    Agnodeinfo_t* headInfo;
    int count = 0;

    while (node) {
        Agedge_t* edge = agfstout (graph, node);
        while (edge) {
            ++count;
            head = aghead (edge);

            edgeInfo = GraphUtils::getInfo (edge);
            headInfo = GraphUtils::getInfo (head);

            edgeInfo->calls = (long) headInfo->retcnt;
            edgeInfo->weight = edgeInfo->calls;
            edgeInfo->NumTaken = 0;
            edgeInfo->reuse = 0;
            ostringstream oss;
            oss << edgeInfo->weight;
            char* label = strdup (oss.str().c_str());
            agset (edge, kLabel, label);
            free (label);
            edge = agnxtout (graph, edge);
        }
        node = agnxtnode (graph, node);
    }

    assert (count == agnedges (graph));
}

Agraph_t* HpcDatabase::IncludeSLOInfo (Agraph_t* HpcGraph,
                                       Agraph_t* SloGraph)
{
    Agraph_t *SloSubgraph, *HpcSubgraph;
    Agnode_t *SloSrc, *SloDst, *Head, *Tail;
    Agnodeinfo_t *HeadInfo, *TailInfo;
    Agedge_t *SloEdge, *Edge;
    Agedgeinfo_t *EdgeInfo, *SloEdgeInfo;
    string NodeName, FileName;
    char *Name;

    if (SloGraph == NULL) return HpcGraph;
    if (HpcGraph == NULL) return NULL;

    HpcSubgraph = agfstsubg(HpcGraph);
    SloSubgraph = agfstsubg(SloGraph);

    // Add features from SLO graph to the HPC graph.
    for (Tail = agfstnode(HpcSubgraph); Tail != NULL;
         Tail = agnxtnode(HpcSubgraph,Tail)) {
        TailInfo = GraphUtils::getInfo(Tail);
        NodeName = "./src";
        FileName = TailInfo->procedure->file();
        FileName.erase(0,NodeName.size());
        NodeName = TailInfo->procedure->name();
        NodeName.append("|");
        NodeName.append(FileName);
        Name = strdup(NodeName.c_str());
        SloSrc = NULL;
        SloSrc = agnode(SloSubgraph,Name,false);
        free(Name);
        // There is a SLO graph node for the function in HPC graph node
        if (SloSrc)
            for (Edge = agfstedge(HpcSubgraph,Tail); Edge != NULL;
                 Edge = agnxtedge(HpcSubgraph,Edge,Tail)) {
                Head = aghead(Edge);
                HeadInfo = GraphUtils::getInfo(Head);
                NodeName = "./src";
                FileName = HeadInfo->procedure->file();
                FileName.erase(0,NodeName.size());
                NodeName = HeadInfo->procedure->name();
                NodeName.append("|");
                NodeName.append(FileName);
                Name = strdup(NodeName.c_str());
                SloDst = NULL;
                SloDst = agnode(SloSubgraph,Name,false);
                free(Name);
                // There is a SLO node for the function in HPC node
                if (SloDst) {
                    EdgeInfo = GraphUtils::getInfo(Edge);
                    SloEdge = NULL;
                    SloEdge = agedge(SloSubgraph,SloSrc,SloDst,NULL,false);
                    // There is an edge between both SLO nodes.
                    // Multiply the HPC graph edge weight with
                    // SLO graph edge weight.
                    if (SloEdge) {
                        std::ostringstream Stream;
                        SloEdgeInfo = GraphUtils::getInfo(SloEdge);
                        EdgeInfo->weight *= (SloEdgeInfo->weight+1);
                        EdgeInfo->reuse = SloEdgeInfo->weight;
                        Stream << EdgeInfo->weight;
                        Name = strdup(Stream.str().c_str());
                        agset(Edge,kLabel,Name);
                        free(Name);
                    }
                }
            }
    }
    return HpcGraph;
}
