//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_PROFILING_HPCDATABSE_H
#define AAP_PROFILING_HPCDATABSE_H

#include <map>
#include <stack>
#include <string>
#include <set>

#include <xercesc/sax2/DefaultHandler.hpp>

#include <llvm/Support/CommandLine.h>
#include "config/llvm.h"

#include "graph/graph.h"
#include "profiling/hpcprocedure.h"

namespace aap {

/// Parses an HPC profile database to create a call tree representation. 
class HpcDatabase : public xercesc::DefaultHandler
{
public:
    /// Path to the HPC Toolkit profile directory.
    static llvm::cl::opt<llvm::sys::Path, false,
                         llvm::cl::parser<std::string> > profile;
    /// Constructs a representation of the HPC profile at the given path.
    HpcDatabase (const llvm::sys::Path& path=profile);

    /// Builds the graph representation of the profile. 
    Agraph_t*           generateGraph   (void);
    std::string*        pathToFile      (const std::string&) const;
    /// Reads an existing graph and associates profile information with it. 
    static Agraph_t*    readGraph       (const std::string&);
    /// Assign return counts to edge weights
    static void         assignWeights  (Agraph_t*);
    /// Include data reuse info to the edge weights.
    static Agraph_t*    IncludeSLOInfo (Agraph_t*, Agraph_t*);

private:
    void addSubgraph (void);
    void popStacks   (void);

    ProcedureRef lookupProcedure(std::string& name, std::string& fileName);

    // handlers for HPC nodes
    void addProcedure   (const xercesc::Attributes& attrs);
    void procedureCall  (const xercesc::Attributes& attrs);
    void loopCall       (const xercesc::Attributes& attrs);
    void metricNode     (const xercesc::Attributes& attrs);
    void addFile        (const xercesc::Attributes& attrs);
    void registerMetric (const xercesc::Attributes& attrs);
    void createCall     (Agnode_t*);

    // sax ContentHandler methods
    void endElement(const XMLCh*, const XMLCh*, const XMLCh*);
    void startElement(const XMLCh*, const XMLCh*, 
                      const XMLCh*, const xercesc::Attributes&);

    // sax ErrorHandler methods
    void warning(const xercesc::SAXParseException&);
    void error(const xercesc::SAXParseException&);
    void fatalError(const xercesc::SAXParseException&);

    std::map<int,std::string> procedures_;
    std::map<int,std::string> files;
    std::stack<Agnode_t*> procStack_; // call stack
    std::stack<Agnode_t*> outStack_;  // call stack minus unavailable functions
    std::stack<Agraph_t*> subgStack_;
    std::stack<unsigned>  callStack_; // stack of call sites (vs procedures)
    llvm::sys::Path path_;            // path to the database
    Agraph_t* graph_;
    unsigned stackUid;          // unique identifier for all stack elements
    unsigned subgUid_;          // unique identifier for subgraphs
    std::set<int> wallclockIndexes_;
    std::set<int> retCountIndexes_;
    ProcedureRef  unknownProc_;

    std::map<std::string,ProcedureRef> nameLookup_;
};

} // namespace app

#endif

// Local Variables:
// mode:c++
// End:
