//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_PROFILING_HPCPROCEDURE_H
#define AAP_PROFILING_HPCPROCEDURE_H

#include <set>

#include "llvm/ADT/IntrusiveRefCntPtr.h"

#include "procedure.h"
#include "sourceobj.h"

namespace aap {

/// Represents a procedure in the HPC profile.

/// Multiple nodes in the resulting call tree may refer to the same procedure,
/// which is important to know later to ensure that only one output procedure
/// exports the original name.  Procedures that HPC toolkit could not identify,
/// or that are located in files that are unknown should not be reported (but
/// may be recorded while the profile is being parsed).
class HpcProcedure : //public llvm::RefCountedBase<HpcProcedure>,
                     public Procedure
{
public:
    HpcProcedure (const std::string& name, const std::string& file);

    /// Returns the name of the procedure. 
    const std::string& name     (void) const;
    /// Returns the file HPCtk believes contains the procedure.
    const std::string& file     (void) const;
    /// Returns the SourceObject representing the procedure.
    const SourceObjRef function (void) const;

    /// Returns true if this procedure should be represented in the call-graph.
    bool shouldOutput (void) const;

    /// Increments the number of procedures referring to the same function.
    void incrementReferences (void);
    /// Sets the SourceObject representing the procedure.
    void setFunction         (const SourceObjRef);
    /// Returns true if the procedure should be represented (e.g. in graphs).
    void setShouldOutput     (bool);

    /// Support for llvm::dyn_cast et al.
    static bool classof(const HpcProcedure*);
    static bool classof(const Procedure*);

protected:
    virtual Procedure* getRepresentativeProcedure() const;

private:
    HpcProcedure& operator= (const HpcProcedure &);

    std::string name_;          // procedure name
    std::string file_;          // source file path
    int ref_count_;             // number of instances of this procedure
    SourceObjRef func_;
    bool shouldOutput_;         // whether this procedure should be reported
    unsigned retcnt_;           // HPCTK RETCNT metric
    float wallclock_;           // HPCTK WALLCLOCK metric

    friend class HpcDatabase;
};

// typedef llvm::IntrusiveRefCntPtr<HpcProcedure> HpcProcedureRef;
// typedef std::set<HpcProcedureRef>     HpcProcedureSet;

} // namespace aap

#endif
