//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <fstream>
#include <iostream>
#include <sstream>

#include <assert.h>

#include "profiling/hpcdatabase.h"
#include "rpu/rpu.h"
#include "rpu/rpuexporter.h"
#include "utils/options.h"
#include "sourcefile.h"
#include "sourceobj.h"

using namespace aap;
using namespace std;

// this object does not own the RPUs it contains
map<string, RPU*>         RPU::globalObjects_;

map<string,set<string> >  RPU::exportedObjects_;
multiset<string>          RPU::staticCollisions_;
multiset<string>          RPU::crossRpuObjects_;

static const string kFunction = "function ";

static bool so_cmp (SourceObjRef lhs, SourceObjRef rhs)
{
    return (*lhs) < (*rhs);
}

/// If a SourceFile is given (the default is NULL) then the RPU represents the
/// file directly. No parsing is done, and extraction will simply involve
/// pre-processing the file.
RPU::RPU (SourceFile* file)
    : path_      (NULL),
      outputted_ (false),
      file_      (file),
      needed_types_ (so_cmp)
{
    if (file_) name_ = file->name();
    buildOptions_ = new BuildOptions();
}

RPU::~RPU (void)
{
    if (buildOptions_) delete buildOptions_;
    if (path_) delete path_;
}

const string&
RPU::path(void)
{
    if (path_ == NULL) {
        ostringstream oss;
        oss << "rpu." << name_;
        if (file_) {
            file_->keepPreprocFile();
        } else {
            oss << ".c";
        }
        path_ = new string (oss.str());
    }
    return *path_;
}

const string&
RPU::name (void) const
{
    return name_;
}

bool
RPU::outputted (void)
{
    return outputted_;
}


const BuildOptions&
RPU::buildOptions (void) const
{
    if (file_) return file_->buildOptions();
    return *buildOptions_;
}

bool
RPU::requireDefinition (const string& type_name, const SourceFile& file)
{
    SourceObjRef type = file.lookupDef (type_name);

    // it may be okay for the type to be unknown if e.g. we only use pointers
    // to a typedef of a struct. the typedef is needed, but not the struct
    if (type == SourceObj::UnknownType ){
        type = file.lookupDef (kFunction + type_name);
        if (type == SourceObj::UnknownType) return true;
    }

    const set<string>& deps = type->dependencies();

    bool isFullyExplored = fullyExplored_.count (type_name) > 0;
    bool needed = (needed_types_.count (type) == 0);
    bool needRecheck = false;
    bool unknownLocation = type->source().isUnknownLocation();

    if (needed) registerNeededObj (type);

    // It's quite easy to get an infinite loop here. Be sure to only check a
    // given name once for a single recursive call.
    if (!isFullyExplored || needed || unknownLocation) {
        fullyExplored_.insert (type_name); // (set here to prevent cycles)
        set<string>::iterator req_it;
        for (req_it = deps.begin(); req_it != deps.end(); req_it ++)
            needRecheck |= requireDefinition ((*req_it), file);
        if (needRecheck) fullyExplored_.erase (type_name);
    }

    return needRecheck;
}

// needed_types_ may hold more than one representation of the same source
// objects, but only one from a preprocessed source file. during the export
// process only one (the earliest) will be included
void
RPU::registerNeededObj (SourceObjRef obj)
{
    needed_types_.insert (obj);
    neededTypesMap_.insert (make_pair (obj->name(), obj));
    if ((obj->isVariable() && !obj->isExtern())
        || (obj->isFunction() && obj->isDefinition()))
    {
        const string& globalName = obj->globalName();
        if (globalObjects_.count (globalName) != 0) {
            if (entryPoints_.count (obj->name())) {
                globalObjects_.erase (globalName);
                globalObjects_[globalName] = this;
            }
        } else {
            globalObjects_[globalName] = this;
        }
    }
    registerExportedObject (obj);
}


void
RPU::registerExportedObject (const SourceObjRef objRef)
{
    if (objRef == SourceObj::UnknownType) return;

    SourceObj& obj = *objRef;
    if (&(obj.file()) == NULL) return;
    if (!(obj.isVariable() || (obj.isFunction() && obj.isDefinition()))) return;

    const string& name = obj.name();
    const string& file = obj.file().name();

    exportedObjects_[name].insert (file);

    // Register variables that are used in multiple RPUs. This is essential for
    // potentially making the variable static.
    if (obj.isVariable()) crossRpuObjects_.insert (obj.name());
}

bool
RPU::nameCollides (const string& name)
{
    if (exportedObjects_.count (name) == 0) return false;
    return (exportedObjects_[name].size() > 1);
}

static void
printFindError (const SourceFile& source, const string& name)
{
        cerr << "warning: failed to find " << name 
             << " in " << source.name() << endl;
        if (Options::verbose())
            cerr << "\t" << source.path() << endl;
}

void
RPU::registerDependencies(void)
{
    map<string, const SourceObjRef>::iterator proc_it;

    for (proc_it = procedures_.begin();
         proc_it != procedures_.end();
         proc_it++)
    {
        const SourceObjRef proc = proc_it->second;
        if (!proc) continue;

        const string& file = proc->file().name();
        if (0 == (file.compare ("~unknown-file~"))) continue;

        if (proc != SourceObj::UnknownType) {
            requireDefinition (proc->name(), proc->file());
        } else {
            printFindError (proc->file(), proc->name());
        }
    }
}

void
RPU::exportToFile (void)
{
    RpuExporter exporter (*this);
    exporter.exportToFile();
}
