//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_RPU_H
#define AAP_RPU_H

#include <map>
#include <set>
#include <string>

#include "sourceobj.h"

namespace aap {

class BuildOptions;
class HpcDatabase;
class RpuExporter;
class SourceFile;
class SourceObj;

/// Refactored Program Unit

/// A set of functions that should be grouped into a single translation unit
/// (e.g. a file).
class RPU {

public:
    ~RPU (void);

    /// Returns the path to the output file.
    const std::string&  path (void);
    /// Returns the short name of the RPU.
    const std::string&  name (void) const;
    /// Returns the set of options needed to build the resulting RPU file. 
    const BuildOptions& buildOptions (void) const;

    void registerDependencies(void); ///< Finalize the set of required SourceObj
    void exportToFile (void);        ///< Extract the needed definitions
    bool outputted (void);           ///< Returns true if file output is done

    friend class RpuExporter;
    friend class RpuFactory;
    friend class RpuGraphUtil;

private:
    /// RPUs are built using the RpuFactory class
    RPU(SourceFile* = NULL);

    // Indicate the RPU requires the source object with the given name, which
    // is found in the given file
    bool requireDefinition      (const std::string&, const SourceFile&);

    void registerExportedObject (const SourceObjRef);
    void registerNeededObj      (SourceObjRef);

    static bool nameCollides (const std::string&);

    std::string*        path_;         // path to the output file
    BuildOptions*       buildOptions_; 
    bool                outputted_;    // true after RPU output to file
    std::string         name_;

    /// For dummy RPUs that represent a SourceFile directly.
    SourceFile*         file_;

    // Functions which must have exported symbols (i.e. not 'static')
    std::set<std::string> entryPoints_;

    // Names of objects for which all possible dependencies are known
    std::set<std::string> fullyExplored_;

    /// Objects we need to extract. This includes both explicitly required
    /// procedures and objects they implicitly require.
    std::set<SourceObjRef,bool(*)(SourceObjRef,SourceObjRef)> needed_types_;
    std::map<std::string, SourceObjRef> neededTypesMap_;

    /// Procedures explicitly assigned into this RPU.
    std::map<std::string, const SourceObjRef> procedures_;

    // mapping of caller/callee pair to an RPU name
    std::map<std::string, std::string> callTargets_;

    /// Mapping of function names and their new names. This information is only
    /// consumed if a final graph is generated.
    std::map<std::string, std::string> mangledNames_;

    // mapping of obj/file pair to the RPU that "owns" it
    static std::map<std::string, RPU*>      globalObjects_;

    // mapping from source object name to a set of file names from which that
    // object has been (/ will be) exported.
    static std::map<std::string,
                    std::set<std::string> >  exportedObjects_;

    static std::multiset<std::string>       staticCollisions_;

    /// Names of objects used in multiple RPUs 
    static std::multiset<std::string> crossRpuObjects_;
};

} // namespace aap

#endif
