//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <libgen.h>

#include "analysis/arraypadding.h"
#include "graph/graph.h"
#include "rpu/rpuexporter.h"
#include "utils/options.h"
#include "sourceextractor.h"
#include "sourcefile.h"
#include "sourcefileparser.h"
#include "sourceobj.h"

#include <iostream>

using namespace aap;
using namespace std;

static const string kFunction = "function ";

RpuExporter::RpuExporter (RPU& rpu)
    : output_(rpu)
{
    rpu_ = &rpu;
}

void
RpuExporter::exportObj (const SourceObjRef& obj)
{
    SourceExtractor se (obj, output_);
    se.extract ();
}

void
RpuExporter::addCallReplacements (const SourceObjRef& obj,
                                  map<string,string>& replace)
{
    // check all the dependencies to see if they need to be mangled
    const set<string>& deps = obj->dependencies();
    set<string>::const_iterator dep_it;
    for (dep_it = deps.begin(); dep_it != deps.end(); dep_it++) {
        // if the dependency is on a static function in another RPU the
        // function's name will have been mangled

        // assume it's a function (if not, we don't care about the object)
        string name = kFunction + (*dep_it);
        // skip the case of a function depending on itself
        if (name.compare (obj->name()) == 0) continue;
        if (&obj->file() == NULL) continue;
        // assume it's static and build the globally unique name (we can use
        // the file name of the dependent object since the statics would be in
        // the same file)
        string globalName = name + "/" + obj->file().name();
        SourceFile& origin = SourceFile::findFunction (globalName);

        // if there is no known instance of such a static function, skip it
        if (&origin == NULL) continue;

        SourceObjRef dep_obj = origin.lookupDef (name);
        if (dep_obj->isStatic()
            && ((rpu_->procedures_.count (name) == 0)
                || (RPU::nameCollides (name))))
        {
            string key = (*dep_it) + "/" + dep_obj->baseName();
            string mangled;
            if (rpu_->callTargets_.count (key)) {
                mangled = (*rpu_->callTargets_.find (key)).second;
            } else {
                mangled = mangledTargetName (dep_obj);
            }
            replace.insert (make_pair (*dep_it, mangled));
        }
    }
}

void
RpuExporter::addRefReplacements (const SourceObjRef& obj,
                                 map<string,string>& replace)
{
    // check all the dependencies to see if they need to be mangled
    const set<string>& deps = obj->dependencies();
    set<string>::const_iterator dep_it;
    for (dep_it = deps.begin(); dep_it != deps.end(); dep_it++) {
        const string& name = (*dep_it);
        int count = 0;
        count += RPU::globalObjects_.count (name);
        count += RPU::staticCollisions_.count (name);
        if (count > 1) {
            assert (rpu_->neededTypesMap_.count (name));
            SourceObjRef dep = rpu_->neededTypesMap_[name];
            string mangledName = mangledVarName (dep);
            replace.insert (make_pair (name, mangledName));
        }
    }
}

void
RpuExporter::exportFunction (const SourceObjRef& obj)
{
    RPU* owner = NULL;
    string globalName = obj->globalName();
    if (!obj->isDefinition()) globalName = kFunction + globalName;
    if (RPU::globalObjects_.count (globalName))
        owner = (*(RPU::globalObjects_).find (globalName)).second;

    if (owner == rpu_) exportFunctionDirect (obj);
    else if (obj->isDefinition()) exportFunctionIndirect (obj);
    else exportPrototypeIndirect (obj);
}

void
RpuExporter::exportFunctionDirect (const SourceObjRef& obj)
{
    if (ArrayPadding::paddingActive()) {
        if (Options::injectPragmas(Options::kArrayPaddingPragmas)) {
            set<PragmaInfo> pis = ArrayPadding::paddingPragmasForFunction(obj->globalName());
            for (set<PragmaInfo>::const_iterator i = pis.begin(), e = pis.end(); i != e; ++i) {
                obj->addPragma(*i);
            }
        }
    }

    bool redirected = obj->isStatic();
    SourceExtractor se (obj, output_);
    map<string,string> replace;
    addCallReplacements (obj, replace);
    addRefReplacements (obj, replace);
    if (replace.size()) se.addReplacements (replace);
    se.extractNonExtern();
    if (redirected) {
        string mangled = mangledName (obj);
        se.extractRedirection (mangled);
        rpu_->mangledNames_.insert (make_pair (obj->name(), mangled));
        Options::summary() << "\t" << mangled
                           << " [wraps " << obj->baseName() << "]\n";
    }

    if (obj->isDefinition()) {
        Options::summary() << "\t" << obj->baseName();
        if (redirected) Options::summary() << " [static]";
        Options::summary() << "\n";
    }
}

void
RpuExporter::exportFunctionIndirect (const SourceObjRef& obj)
{
    bool redirected = false;

    SourceExtractor se (obj, output_);

    redirected = rpu_->entryPoints_.count (obj->name()) > 0;

    map<string,string> replace;
    addCallReplacements (obj, replace);
    addRefReplacements (obj, replace);
    if (replace.size()) se.addReplacements (replace);
    se.extractStatic ();
    Options::summary() << "\t" << obj->baseName() << " [static]\n";

    string mangled = mangledName (obj);
    if (redirected) {
        se.extractRedirection (mangled);
        Options::summary() <<"\t" <<mangled <<" [wraps " <<obj->name() <<"]\n";
    }
    rpu_->mangledNames_.insert (make_pair (obj->name(), mangled));
}

void
RpuExporter::exportPrototypeIndirect (const SourceObjRef& obj)
{
    bool haveLocal, localShadow;
    SourceExtractor se (obj, output_);

    string functionName = kFunction + obj->name();
    string globalName   = functionName + "/" + obj->file().name();

    // if the prototype is for a static function, 
    if (obj->isStatic()) {
        haveLocal = (rpu_->procedures_.count (globalName) > 0);
    } else {
        if (rpu_->procedures_.count (functionName) > 0) {
            const SourceObjRef func = rpu_->procedures_.find (functionName)->second;
            haveLocal = (func != SourceObj::UnknownType && !func->isStatic());
        } else {
            haveLocal = false;
        }
    }

    localShadow = (rpu_->procedures_.count (functionName) > 0);

    if (haveLocal) {
        // we have the function locally, and we're not the owner -> static
        se.extractStatic ();
    } else {
        // if we have something local that shadows the object, or it's a static
        // function, we'll have to access it's wrapper function
        if (localShadow || obj->isStatic())
            se.addReplacement (obj->name(), mangledTargetName (obj));
        se.extractExtern ();
    }
}

void
RpuExporter::exportVariable (const SourceObjRef& obj)
{
    RPU* owner = NULL;
    string globalName = obj->globalName();
    if (RPU::globalObjects_.count (globalName))
        owner = (*(RPU::globalObjects_).find (globalName)).second;

    if (ArrayPadding::paddingActive())
    {
        ArrayPadding::associateArrayWithRPU (obj->baseName(), rpu_->name());
        if (ArrayPadding::shouldPadArray(obj->baseName()) &&
            Options::injectPragmas(Options::kArrayPaddingPragmas))
        {
            obj->addPragma(ArrayPadding::safePaddingPragma(obj->baseName(), 0));
        }
    }

    if (owner == rpu_) exportVariableDirect (obj);
    else exportVariableIndirect (obj);
}

void
RpuExporter::exportVariableDirect (const SourceObjRef& obj)
{
    SourceExtractor se (obj, output_);
    int collide = 0;
    const string& name = obj->name();

    collide += RPU::globalObjects_.count (name);
    collide += RPU::staticCollisions_.count (name);

    map<string,string> replace;
    addCallReplacements (obj, replace);
    se.addReplacements (replace);

    if (collide > 1)
        se.addReplacement (name, mangledVarName (obj));

    if (Options::promoteStatic() && RPU::crossRpuObjects_.count (name) == 1)
        se.extractStatic();
    else
        se.extractNonStatic ();
}

void
RpuExporter::exportVariableIndirect (const SourceObjRef& obj)
{
    SourceExtractor se (obj, output_);
    int collide = 0;
    const string& name = obj->name();

    collide += RPU::globalObjects_.count (name);
    collide += RPU::staticCollisions_.count (name);

    if (collide > 1)
        se.addReplacement (name, mangledVarName (obj));

    se.extractExternDeclaration ();
}

void
RpuExporter::exportToFile (void)
{
    if (rpu_->file_) {
        std::string path = Options::outPath() + "/" + rpu_->path();
        SourceFileParser::Preprocess(*(rpu_->file_), path);
        rpu_->file_->keepPreprocFile();
        rpu_->outputted_ = true;
        return;
    }

    // If there's nothing to output, return early.
    if (rpu_->needed_types_.size() == 0) return;

    // Assume that there will be output, since we know there are needed types.
    rpu_->outputted_ = true;

    set<SourceObjRef>::iterator it, end;

    Options::summary() << "\n" << rpu_->path() << "\n";

    // We rely on the needed_types_ being ordered. Note that for one file a
    // partial definition may suffice, while in another we need the complete
    // definition. As a result, we must fully explore the tree of needed types
    // for each imported function. At this stage we simply avoid duplicate
    // imports.
    end = rpu_->needed_types_.end();
    for (it = rpu_->needed_types_.begin(); it != end; ++it)
    {
        const SourceObjRef object = *it;
        const string& name = object->globalName();

        if (knownTypes_.count (name) > 0) continue;

        // Don't register extern declarations to allow a later declaration
        // with storage to be imported too
        if (!object->isExtern()) knownTypes_.insert (name);

        if (object->isVariable())      exportVariable (object);
        else if (object->isFunction()) exportFunction (object);
        else exportObj (object);
    }
    rpu_->needed_types_.clear();
}

static void
cleanName (string& name)
{
    size_t loc;
    while ((loc = name.find_last_of (".")) != string::npos)
        name.replace (loc, 1, 1, '_');
    while ((loc = name.find_last_of ("-")) != string::npos)
        name.replace (loc, 1, 1, '_');
    while ((loc = name.find_last_of ("\"")) != string::npos)
        name.replace (loc, 1, 1, '_');
}

// provide a mangled version of the object's name, given that the object
// resides in the local rpu
string
RpuExporter::mangledName (const SourceObjRef& obj)
{
    string res = obj->mangledName() + "_RPU_" + rpu_->name_;
    cleanName (res);
    return res;
}

// provide a mangled version of the object's name as it would be created for
// the rpu that owns the object
string
RpuExporter::mangledTargetName (const SourceObjRef& obj)
{
    string rpuName;
    string globalName = obj->globalName();

    if (obj->isFunction() && !obj->isDefinition())
        globalName = kFunction + globalName;

    if (RPU::globalObjects_.count (globalName)) {
        // look for an RPU that owns the target object
        RPU* owner = (*(RPU::globalObjects_).find (globalName)).second;
        rpuName = owner->name_;
    } else {
        // otherwise assume that the original file has the leftover
        rpuName = obj->file().name();
    }
    string res = obj->mangledName() + "_RPU_" + rpuName;
    cleanName (res);
    return res;
}

string
RpuExporter::mangledVarName (const SourceObjRef& obj)
{
    assert (obj->isVariable() && "mangledVarName called with non-var");

    string res = obj->name();
    // static objects are unique to their source file, while global objects are
    // uniquely owned by one RPU
    if (obj->isStatic()) {
        res += "_";
        res += obj->file().name();
    } else if (RPU::globalObjects_.count (obj->name())) {
        RPU* owner = (*(RPU::globalObjects_).find (obj->name())).second;
        res += "_RPU_" + owner->name_;
    } else {
        assert (0 && "global object is un-owned");
    }
        
    cleanName (res);
    return res;
}
