//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <iostream>
#include <sstream>

#include <assert.h>
#include <libgen.h>

#include "graph/graph.h"
#include "graph/graphutils.h"
#include "profiling/hpcdatabase.h"
#include "rpu/rpu.h"
#include "rpu/rpufactory.h"
#include "utils/options.h"
#include "sourcefile.h"
#include "sourceobj.h"

using namespace aap;
using namespace std;

static const string kFunction = "function ";

static string
rpuName (Agraph_t* graph)
{
    string name = agnameof (graph);
    name = name.substr (Options::rpuPrefix().length(), string::npos);
    return name;
}

static void
cleanName (string& name)
{
    size_t loc = 0;
    while ((loc = name.find_first_of ('-', loc)) != string::npos)
        name[loc] = '_';
}

RPU* 
RpuFactory::ConvertDagSubgraph (Agraph_t* graph)
{
    RPU* rpu = new RPU();
    rpu->name_ = rpuName (graph);
    Agnode_t* node = agfstnode (graph);
    Agnodeinfo_t* nodeInfo;
    const string& rpuPrefix = Options::rpuPrefix();
    while (node)
    {
        nodeInfo = GraphUtils::getInfo (node);
        const string& name = nodeInfo->procedure->name();
        // if the edge to this node isn't in the subgraph, then this function
        // is an entry point for the RPU
        if (agfstin (graph, node) == NULL) {
            rpu->entryPoints_.insert (kFunction + name);
        }
        // if any of the edges out of this node aren't in the subgraph then we
        // need to record the target RPU
        Agraph_t* root = graph->root;
        Agedge_t* edge = agfstout (root, node);
        while (edge != NULL) {
            if (!agsubedge (graph, edge, FALSE)) {
                Agnode_t* head = aghead (edge);
                Agnodeinfo_t* headInfo = GraphUtils::getInfo (head);
                if (headInfo->subg) {
                    string rpuName = agnameof (headInfo->subg);
                    rpuName.replace (0, rpuPrefix.length(), "");
                    string key = name + "/" + headInfo->procedure->name();
                    rpu->callTargets_.insert (make_pair (key, rpuName));
                }
            }
            edge = agnxtout (root, edge);
        }

        const SourceObjRef func = nodeInfo->procedure->function();
        if (func) {
            SourceObjRef funcRef (func);
            rpu->procedures_.insert (make_pair (kFunction + name, funcRef));
            if (func->isStatic())
                rpu->procedures_.insert (make_pair (func->globalName(), funcRef));
        }
        node = agnxtnode (graph, node);
    }
    return rpu;
}

RPU*
RpuFactory::ConvertIndSubgraph (Agraph_t* graph)
{
    RPU* rpu = new RPU();
    rpu->name_ = rpuName (graph);
    Agnode_t* Node = agfstnode (graph);
    Agnodeinfo_t* NodeInfo;
    for (Node = agfstnode (graph); Node != NULL;
         Node = agnxtnode (graph, Node))
    {
        NodeInfo = GraphUtils::getInfo (Node);
        const string& name = NodeInfo->procedure->name();
        const SourceObjRef func = NodeInfo->procedure->function();
        if (func != NULL)
        {
            SourceObjRef funcRef (func);
            rpu->procedures_.insert (make_pair(kFunction + name, funcRef));
            if (func->isStatic())
                rpu->procedures_.insert (make_pair (func->globalName(),
                                                    funcRef));
        }
    }
    return rpu;
}

RPU_set
RpuFactory::ConvertGraph(Agraph_t* graph)
{
    int index = 0;
    RPU_set result;
    ostringstream* oss = new ostringstream();
    (*oss) << Options::rpuPrefix() << index++;

    Agraph_t* subg;
    char* name = strdup(oss->str().c_str());
    while ((subg = agsubg(graph, name, FALSE)))
    {
        // assign all nodes as belonging to their immediate parent graph
        Agnode_t* node = agfstnode (subg);
        while (node) {
            Agnodeinfo_t* nodeInfo = GraphUtils::getInfo (node);
            nodeInfo->subg = subg;
            node = agnxtnode (subg, node);
        }
        free(name);
        delete oss;
        oss = new ostringstream();
        (*oss) << Options::rpuPrefix() << index++;
        name = strdup(oss->str().c_str());
    }

    free(name);
    delete oss;
    index = 0;
    oss = new ostringstream();
    (*oss) << Options::rpuPrefix() << index++;
    name = strdup(oss->str().c_str());
    subg = agsubg (graph, name, FALSE);
    while (subg)
    {
        RPU *rpu;
        if (agisdirected (graph))
            rpu = ConvertDagSubgraph (subg);
        else
            rpu = ConvertIndSubgraph (subg);
        result.insert (rpu);
        free(name);
        delete oss;
        oss = new ostringstream();
        (*oss) << Options::rpuPrefix() << index++;
        name = strdup(oss->str().c_str());
        subg = agsubg (graph, name, FALSE);
    }
    free(name);
    delete oss;

    return result;
}

// Return of set of RPUs containing the functions from this file
// that were not in the graph (i.e. not in the profile at all)
// We have no metrics on how partitions them, so just do it based
// on the iterator
RPU_set*
RpuFactory::registerLeftovers (SourceFile& file)
{
    RPU_set* rpu_set = new RPU_set();
    unsigned int rpu_id = 0;
    unsigned int rpuFunctions = 0;
    unsigned int rpuLines = 0;

    RPU* rpu;

    // If the file hasn't been parsed, just return a dummy RPU.
    if (!file.IsParsed()) {
        rpu = new RPU(&file);
        rpu_set->insert(rpu);
        return rpu_set;
    }

    rpu = new RPU();
    rpu->path_ = new string ("rpu.");
    ostringstream oss;
    if (Options::splitLeftover()) oss << rpu_id++ << ".";
    oss << file.name();
    rpu->path_->append (oss.str());
    rpu->name_ = oss.str();    
    cleanName (rpu->name_);

    const SourceMap& functions = file.functions();
    SourceMap::const_iterator func_it = functions.begin();

    while (func_it != functions.end()) {
        const string& func_name = (*func_it).first;
        const SourceObjRef func = (*func_it).second;

        if (func->isDefinition()) {
            if (!hasExported (func_name, rpu->name_)) {
                const SourceLocation& loc = func->source();
                int lines = loc.lastLine() - loc.first_line();
                // If the RPU isn't empty, and adding the next function would
                // overflow the function or line limit, start a new RPU.
                if (Options::splitLeftover()
                    && (rpuFunctions && rpuLines)
                    && ((rpuFunctions + 1 > Options::rpuMaxFunctions())
                        || (rpuLines + lines > Options::rpuMaxLines())))
                {
                    rpu_set->insert(rpu);
                    rpu = new RPU();
                    rpuFunctions = 0;
                    rpuLines = 0;
                    rpu->path_ = new string ("rpu.");
                    oss.str("");
                    oss << rpu_id++ << "." << file.name();
                    rpu->path_->append (oss.str());
                    rpu->name_ = oss.str();
                    cleanName (rpu->name_);
                }

                rpu->requireDefinition (func_name, file);
                rpu->procedures_.insert (make_pair (func_name, func));
                string globalName = func_name + "/" + file.name();
                rpu->procedures_.insert (make_pair (globalName, func));

                rpuFunctions++;
                rpuLines += lines;
            }
        }
        func_it++;
    }
    rpu_set->insert(rpu);

    const SourceMap& globals = file.globals();
    SourceMap::const_iterator global_it = globals.begin();
    while (global_it != globals.end()) {
        string name = (*global_it).first;
        const SourceObjRef def = (*global_it).second;
        string globalName = def->globalName();

        if (def->isStatic()) {
            RPU::staticCollisions_.insert (name);
        } else if (!def->isExtern()) {
            // if the object is not owned by some RPU and would have been
            // exported from this file, we need to include its definition
            if (RPU::globalObjects_.count (globalName) == 0)
            {
                RPU_set::iterator rpu_it, rpu_end = rpu_set->end();
                for (rpu_it = rpu_set->begin(); rpu_it != rpu_end; rpu_it++)
                    (*rpu_it)->requireDefinition (name, file);
            }
        }

        global_it ++;
    }

    return rpu_set;
}

bool
RpuFactory::hasExported (const string& obj, const string& file)
{
    if (RPU::exportedObjects_.count (obj) == 0) return false;
    return (RPU::exportedObjects_[obj].count (file) > 0);
}

