#include <llvm/Support/CommandLine.h>
#include <llvm/Support/raw_ostream.h>

#include "rpu/rpuoutput.h"
#include "utils/options.h"

using namespace aap;

typedef std::basic_ostream<char, std::char_traits<char> > CoutType;

static llvm::cl::opt<bool>
injectLineDirectives_(
    "inject-line-directives",
    llvm::cl::desc("If true (default), the AAP will insert preprocessor style "
                   "line directives to indicate the origin of source code."),
    llvm::cl::Hidden, llvm::cl::init(true));

static llvm::cl::opt<std::string>
outputRpuList_(
    "output-rpu-list",
    llvm::cl::desc("If set, list generated RPU file names (one per line). "
                   "The optional argument indicates an output file to use. "
                   "(defaults to standard out)."),
    llvm::cl::Hidden, llvm::cl::init("-"));

static void outputRpuList(llvm::sys::Path& path)
{
    std::string errorInfo;
    llvm::raw_fd_ostream listOut(outputRpuList_.c_str(), errorInfo,
                                 llvm::raw_fd_ostream::F_Append);
    // If there are errors opening the list output, it will report_fatal_error
    // at destruction (at the close of this function).
    listOut << path.str() << "\n";
    listOut.close();
}

RpuOutput::RpuOutput(RPU& rpu)
    : line_(1), outputOwner_(true),
      pendingCppLine_(0), pendingCppFlag_(1), lastCppLine_(0)
{
    path_ = Options::outPath();
    path_.appendComponent(rpu.path());
    output_ = new std::ofstream(path_.c_str());
    if (outputRpuList_.getNumOccurrences()) outputRpuList(path_);
}

RpuOutput::RpuOutput(std::ostream& stream)
    : line_(1), path_("/dev/null/impossible-RPU.c"),
      output_(&stream), outputOwner_(false) { }

RpuOutput::~RpuOutput()
{
    if (line_ > 1) pccLineDirective(2, true);
    output_->flush();
    if (outputOwner_) {
        delete output_;
        output_ = NULL;
    }
}

/// Generates a preprocessor line directive. The default parameter value (2)
/// indicates a return to the file. These statements may be buffered unless
/// @param force is true.
void
RpuOutput::pccLineDirective(unsigned d, bool force)
{
    if (!injectLineDirectives_) return;

    lastCppLine_ = line_;
    if (force) {
        (*output_) << "# " << line_
                   << " \"" << path_.str() << "\" " << d << "\n";
        ++line_;
    } else {
        pendingCppFlag_ = d;
    }
}

/// Generates a preprocessor line directive indicating entry to a file. If the
/// most recent output was from the same source file, the output may be
/// condensed.
void
RpuOutput::pccEntryDirectiveForLocation(const SourceLocation& loc)
{
    if (!injectLineDirectives_) return;

    // if there is pending CPP output, and the line numbering doesn't match up,
    // then output it first.

    int line = loc.originalLine();
    if (line > 0) {
        const std::string& file = loc.originalFile();
        int a = line - pendingCppLine_;
        int b = (line_ - lastCppLine_) + 1;
        if (pendingCppFile_.compare(file) || (a != b)) {
            pccLineDirective(pendingCppFlag_, true);
            (*output_) << "# " << line << " \"" << file << "\" 1\n";
            pendingCppFile_ = file;
            ++line_;
        }
        pendingCppLine_ = line;
    }
}

RpuOutput&
RpuOutput::operator<<(RpuOutput&(*f)(RpuOutput&))
{
    assert (!(lastWasEndl_ = false));
    return f(*this);
}

/// Inserts a new line and increments the line count of the given RpuOutput.
RpuOutput&
RpuOutput::endl(RpuOutput& rpuOutput)
{
    *(rpuOutput.output_) << std::endl;
    rpuOutput.line_++;
    assert ((rpuOutput.lastWasEndl_ = true));
    return rpuOutput;
}

// Handles PragmInfo specially because they have a unique effect on the line
// count. Since injected pragma's aren't from the original source file, we have
// to adjust the CPP line directives a bit.
RpuOutput&
RpuOutput::operator<<(const PragmaInfo& pragma)
{
    (*output_) << "#pragma " << pragma.value() << "\n";
    ++line_;
    ++lastCppLine_;
}

// Accepts std::endl to ensure the line count is incremented.
RpuOutput&
RpuOutput::operator<<(std::basic_ostream<char, std::char_traits<char> >&(*c)
                      (std::basic_ostream<char, std::char_traits<char> >&))
{
    c(*output_);
    line_++;
    assert ((lastWasEndl_ = true));
    return *this;
}

/// Inserts the given string before the last character, which is assumed to be
/// a newline. The status of the previous character is confirmed only if
/// assertions are enabled.
void
RpuOutput::insertBeforeNewline(const std::string& s)
{
    assert (lastWasEndl_);
    output_->seekp(-1, std::ios_base::cur);
    // Use "\n" here to bypass the endl line increment.
    (*output_) << s << "\n";
}
