#ifndef AAP_RPUOUTPUT_H
#define AAP_RPUOUTPUT_H

#include <fstream>

#include "config/llvm.h"

#include "analysis/pragmainfo.h"
#include "rpu/rpu.h"

namespace aap {

/// Represents the final RPU output file. Used during extraction to track the
/// state of the file, including the current line position (for preprocessor
/// information). Streaming to the RpuOutput should use endl rather than "\n"
/// to ensure the line count is correctly maintained.
class RpuOutput
{
public:
    /// Constructs an RpuOutput wrapper for a given RPU.
    RpuOutput(RPU&);
    /// Constructs an RpuOutput wrapper that streams output to the given
    /// ostream. Ownership of the stream is retained by the caller.
    RpuOutput(std::ostream&);
    ~RpuOutput();

    void pccLineDirective(unsigned d = 2, bool force = false);
    void pccEntryDirectiveForLocation(const SourceLocation&);

    RpuOutput& operator<<(const PragmaInfo&);
    RpuOutput& operator<<(RpuOutput&(*)(RpuOutput&));
    RpuOutput& operator<<(std::basic_ostream<char, std::char_traits<char> >&
                          (*)(std::basic_ostream<char,
                              std::char_traits<char> >&));
    static RpuOutput& endl(RpuOutput& rpuOutput);


    template<class T>
    RpuOutput& operator<<(const T& value)
    {
        (*output_) << value;
        return *this;
    }

    void insertBeforeNewline(const std::string&);

private:
    unsigned line_;             // The current line count.
    llvm::sys::Path path_;      // The path of the output file.
    std::ostream* output_;      // Stream to which output is actually sent.
    bool outputOwner_;          // True when the output stream is ours.

    // CPP line directives are buffered, in case they prove to be redundant.
    std::string pendingCppFile_;
    unsigned    pendingCppLine_;
    unsigned    pendingCppFlag_;
    unsigned    lastCppLine_;   // The output line of the last directive.

#ifndef NDEBUG
    bool lastWasEndl_;          // True when endl was the last thing output.
#endif
};

} // end namespace aap

#endif
