//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <limits.h>
#include <stdlib.h>

#include <sysexits.h>

#include <fstream>
#include <iostream>
#include <map>
#include <memory>
#include <sstream>
#include <string>

#include "sourcefile.h"
#include "sourceobj.h"
#include "sourceextractor.h"
#include "utils/callconverter.h"

using namespace std;
using namespace aap;

static bool extractPreprocInfo_ = true;
static bool extractPragmas_ = true;

/// @param obj The source object to extract.
/// @param output RpuOutput to receive extracted source.
SourceExtractor::SourceExtractor (const SourceObjRef obj, RpuOutput& output)
    : obj_ (obj),
      output_ (output)
{
    stripStorage_ = false;
    declOnly_     = false;
    replace_      = NULL;
}

SourceExtractor::~SourceExtractor (void)
{
    if (replace_) delete replace_;
}

void
SourceExtractor::addReplacement (const string& from, const string& to)
{
    if (!replace_) replace_ = new map<string,string>;
    replace_->insert (make_pair (from,to));
}

void
SourceExtractor::addReplacements (const map<string,string>& replace)
{
    if (!replace_)
        replace_ = new map<string,string> (replace.begin(), replace.end());
    else
        replace_->insert (replace.begin(), replace.end());
}

/// Returns true for characters that are valid in C identifiers.
static inline bool
identifierCharacter(char t)
{
    return (isalnum (t) || (t == '_'));
}

// return true of the region of the string is lexically an independent string
// (is not preceded or followed by valid identifier characters)
static bool
standAlone(const string& line, int begin, int len)
{
    unsigned int end = begin + len;
    if (begin > 0){
        if (identifierCharacter (line[begin-1]))
            return false;
        if (line[begin-1] == '.')
            return false;           // we don't rename struct members
        if (line[begin-1] == '>' && line[begin-2] == '-')
            return false;
    }
    if ((end < line.length()) && (identifierCharacter (line[end]))) 
        return false;

    return true;
}

static bool
replaceString (string& line, const string& a, const string& b, bool once)
{
    size_t loc = line.find (a);
    bool found = false;

    if (loc == string::npos) return false;

    while (loc != string::npos) {
        if (standAlone (line, loc, a.length())) {
            line.replace (loc, a.length(), b);
            if (once){
                return true;
            } else {
                loc = line.find (a, loc + b.length());
                found = true;
            }
        } else {
            loc = line.find (a, loc + 1);
        }
    }
    return found;

}

static void
applyReplacements (const map<string,string>& replace, string& line)
{
    if (&replace == NULL) return;
    map<string,string>::const_iterator it;
    for (it = replace.begin(); it!=replace.end(); it++) {
        replaceString (line, (*it).first, (*it).second, false);
    }
}

// Trim the given string by the amount of input read beyond location end
static void
trimLine (string& line, ifstream& input, const SourceLocation& loc)
{
    unsigned end = loc.endPosition();
    unsigned pos = input.tellg();
    if (pos >= end) {
        int len = line.length() - (pos  - end - 1);
        line = line.substr (0, len);            
    }
}

void
SourceExtractor::extractFunctionDeclaration (ifstream& input)
{
    string line;
    const SourceLocation& source = obj_->source();

    // determine what type of character marks the end of the declaration
    char stop = obj_->isDefinition() || source.declarationOnly() ? '{' : ';';

    unsigned end = source.endPosition();
    do {
        getline (input, line, '\n');
        // if we've read past the end position, trim off the excess
        trimLine (line, input, source);

        size_t loc = line.find_first_of (stop);
        if (loc != string::npos) {
            line = line.substr (0, loc);
            line.append (";");
            input.seekg (end);
        }
        applyReplacements (*replace_, line);
        output_ << line << endl;
    } while (!input.eof() && ((unsigned) input.tellg() < end));
}

void
SourceExtractor::extractComplete (string& type, ifstream& input)
{
    if (type.length() > 0) {
        applyReplacements(*replace_,type);
        output_ << type << " ";
    }

    set<PragmaInfo>::const_iterator pEnd = obj_->pragmas().end();
    unsigned offset = 0;

    const SourceLocation& source = obj_->source();

    string line;
    unsigned end = source.endPosition();
    do {
        getline (input, line, '\n');
        trimLine (line, input, source);

        size_t loc = line.find_first_of('=');
        if (declOnly_ && obj_->isVariable() && loc != string::npos) {
            line.resize (loc);
            input.seekg (end);
        }

        while (nextPragma_ != pEnd && nextPragma_->offset() <= offset) {
            output_ << (*nextPragma_);
            ++nextPragma_;
        }
        applyReplacements (*replace_, line);
        output_ << line << endl;
        ++offset;
    } while (!input.eof() && ((unsigned) input.tellg() < end));

    if (obj_->GetPostName().size() > 0)
        output_ << obj_->GetPostName() << endl;

    output_.insertBeforeNewline(";");
}

static void
readError (const SourceLocation& source, const istream& input)
{
    cerr << "error: cannot extract from " 
         << source.file().name() << endl;
    if (input.eof())
        cerr << "\t end of file reached" << endl;
    if (input.bad())
        cerr << "\t internal loss of integrity" << endl;
    else if (input.fail())
        cerr << "\t internal operation failure" << endl;

    exit(EX_DATAERR);
}

void
SourceExtractor::extract (void)
{
    const SourceLocation& source = obj_->source();

    if (source.isUnknownLocation()) return;
    if (&(source.file()) == NULL) return;

    declOnly_ |= source.declarationOnly();

    string line, type = obj_->type();

    ifstream input (source.file().preProcPath().c_str());
    if (!input.is_open()) readError (source, input);
    input.seekg (source.startPosition());

    if (extractPreprocInfo_)
        output_.pccEntryDirectiveForLocation(source);

    // Output any pragmas that belong before the object.
    if (extractPragmas_) nextPragma_ = obj_->pragmas().begin();
    else nextPragma_ = obj_->pragmas().end();

    pragmaEnd_ = obj_->pragmas().end();
    while (nextPragma_ != pragmaEnd_ && nextPragma_->offset() == 0) {
        output_ << *nextPragma_;
        ++nextPragma_;
    }

    output_ << prefix_;

    if (stripStorage_) {
        replaceString (type, "static", "", true);
        replaceString (type, "extern", "", true);
    }

    if (!declOnly_) {
        extractComplete(type, input);
    } else if (obj_->isFunction()) {
        output_ << type << " ";
        extractFunctionDeclaration (input);
    } else {
        size_t n = type.find("=");
        if (n != string::npos)
            type.resize(n);
        applyReplacements(*replace_,type);
        output_ << type << ";" << endl;
    }

    input.close();
    output_.pccLineDirective();

    // reset in case the extractor is used again
    stripStorage_ = false;
    declOnly_ = false;
    prefix_ = "";
}

void
SourceExtractor::extractExtern (void)
{
    if (!obj_->isExtern()) {
        if (obj_->isVariable()) declOnly_ = true;
        stripStorage_ = true;
        prefix_ = "extern ";
    }
    extract ();
}

void
SourceExtractor::extractNonExtern (void)
{
    if (obj_->isExtern())
        stripStorage_ = true;
    extract();
}

void
SourceExtractor::extractNonStatic (void)
{
    if (obj_->isStatic()) {
        stripStorage_ = true;
    }
    extract ();
}

void
SourceExtractor::extractStatic (void)
{
    if (!obj_->isStatic()) {
        if (obj_->isExtern()) stripStorage_ = true;
        prefix_ = "static ";
    }
    extract ();
}

/// Function bodies and variable initializations are elided (as applicable).
void
SourceExtractor::extractDeclaration (void)
{
    declOnly_ = true;
    extract ();
}

void
SourceExtractor::extractExternDeclaration (void)
{
    declOnly_ = true;
    stripStorage_ = true;
    prefix_ = "extern ";
    extract ();
}

static string
getDeclaration (const SourceObjRef obj)
{
    std::ostringstream oss;
    RpuOutput rpuOutput(oss);
    SourceExtractor se (obj, rpuOutput);
    se.extractDeclaration ();
    return oss.str();
}

static void
cleanupDecl (string& dec)
{
    size_t loc = dec.find_first_of ('{');
    if (loc != string::npos)
        dec.replace (loc, string::npos, "");
    // the extractDeclaration will append a ;
    loc = dec.find_last_of (';');
    if (loc != string::npos)
        dec.replace (loc, string::npos, "");
}

// We want to remove \n to make it easier to location parts of a declaration,
// but we should avoid messing with preprocessor directives (which need to be
// on a new line)
static void
removeNewlines (string& dec)
{
    size_t loc;
    size_t length = dec.length();
    while ((loc = dec.find_first_of('\n')) != string::npos) {
        if (loc < length && dec.at (loc) != '#')
            dec.replace (loc, 1, 1, ' ');
    }
}

static void
removeStatic (string& dec)
{
    size_t loc = dec.find("static ");
    // this could probably be an assert
    if (loc != string::npos)
        dec.replace(loc, 7, "");
}

/// When applied to a SourceObj representing a function, extracts a function
/// with the given (mangled) name and the same parameter set as the
/// original. The body of the new function is a call to the original. In the
/// case of va_arg usage, a complete copy of the original is extracted using the
/// new name. Behavior on non-functions is undefined.
void
SourceExtractor::extractRedirection (const string& mangledName)
{
    extractPreprocInfo_ = false;

    // if the function uses va_args we can't wrap it
    if (obj_->usesVaArgs()) {
        addReplacement (obj_->baseName(), mangledName);
        stripStorage_ = true;
        extract();
    } else {
        extractPragmas_ = false;
        string call, dec = getDeclaration (obj_);
        extractPragmas_ = true;
        removeNewlines (dec);
        call = dec;
        replaceString (dec, obj_->baseName(), mangledName, true);
        removeStatic (dec);

        if (obj_->isDefinition()) {
            cleanupDecl (dec);
            CallConverter::convertDeclToCall (call);
            output_ << dec << endl << "{" << endl
                    << "    " << call << ";" << endl << "}" << endl;
        } else {
            output_ << dec << ";" << endl;
        }
    }

    extractPreprocInfo_ = true;
}
