//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/wait.h>

#include <iostream>

#include "sourcefile.h"
#include "utils/options.h"

using namespace std;
using namespace aap;

namespace {

/// Catalog of SourceFile objects (used internally by SourceFile).

/// SourceFile uses SourceFileCatalog to maintain a single static catalog of
/// the parsed source files. By allocating it on the heap we can easily handle
/// destruction of the catalog, and thus the individual source files (which
/// allows for preprocessed source to be removed).
class SourceFileCatalog 
{
public:
    map<string,SourceFile*> files_;
    ~SourceFileCatalog (void);
};

SourceFileCatalog::~SourceFileCatalog (void)
{
    map<string,SourceFile*>::iterator file_it = files_.begin();
    while (file_it != files_.end())
    {
        delete (*file_it).second;
        file_it++;
    }
}

} // anonymous namespace 

// global catalog of files
SourceFileCatalog SourceFile::catalog_;
// mapping from function name to the source file where it is defined
std::map<std::string,SourceFile*> SourceFile::global_functions_;
// file name aliases (e.g. from preprocessed to original source)
std::map<std::string,std::string> SourceFile::aliases_;

void
SourceFile::setName(const string& path)
{
    name_  = path;
    size_t sep = name_.find_first_of ('/');
    while (sep != string::npos) {
        name_.replace (sep, 1, 1, '_');
        sep = name_.find_first_of ('/', sep);
    }
}

SourceFile::SourceFile (const std::string& path)
{
    hasFlags_  = false;
    path_      = path;
    parsed_    = false;
    temp_      = NULL;
    unlink_    = true;
    setName (path);
    catalog_.files_.insert (make_pair (path, this));
}

SourceFile::~SourceFile (void)
{
    if (temp_ != NULL) {
        if (unlink_) unlink (temp_->c_str());
        delete temp_;
    }
}

const std::string&
SourceFile::name (void) const
{
    return name_;
}

const std::string& 
SourceFile::path (void) const
{
    return path_;
}

const std::string&
SourceFile::preProcPath (void) const
{
    if (temp_ != NULL) return *temp_;
    else return path();
}

const SourceMap&
SourceFile::functions (void) const
{
    return functions_;
}

const SourceMap&
SourceFile::definitions (void) const
{
    return definitions_;
}

const SourceMap&
SourceFile::globals (void) const
{
    return globals_;
}

bool
SourceFile::defIsKnown (const std::string& type_name)
{
    return  (definitions_.find(type_name) != definitions_.end());
}

/// Files that are included in the build options file are considered necessary,
/// while those in the profile are not. This is done so that parsing (or
/// locating) files identified by the profile can potentially fail without
/// requiring that the whole process be terminated.
bool
SourceFile::required (void) const
{
    return hasFlags_;
}

void
SourceFile::registerDef (SourceObjRef type)
{
    const string& name = type->name();

    if (type->isVariable() && !type->isExtern()) {
        // allow later non-extern definitions to clobber earlier definitions
        int count = definitions_.count (name);
        if (count > 0) {
            // tweak the extern_def name so it doesn't preclude the inclusion
            // of the object we're registering now
            string extern_name = "extern " + name;
            SourceObjRef extern_def = definitions_.at (name);
            definitions_.erase (name);
            // the same object is removed and re-inserted (with a difference
            // name), so no retain/release is needed
            definitions_.insert (make_pair (extern_name, extern_def));
            type->addDependency (extern_name);
        }

        globals_.insert (make_pair (name, type));
    }

    definitions_.insert (make_pair (name, type));

    if (type->isFunction())
    {
        functions_.insert (make_pair (name, type));
        if (type->isDefinition())
            global_functions_.insert (make_pair (type->globalName(), this));
    }
}

SourceObjRef
SourceFile::lookupDef (const std::string& type_name) const
{
    SourceMap::iterator type_it = definitions_.find (type_name);
    if (type_it != definitions_.end())
        return type_it->second;

    SourceObjRef res (SourceObj::UnknownType);
    return res;
}

const BuildOptions&
SourceFile::buildOptions (void) const
{
    return flags_;
}

/// Replaces existing definitions of alias with a similar definition
/// that only depends on name.
void
SourceFile::aliasDef (const std::string& name, const std::string& alias)
{
    SourceObj* alias_obj;
    alias_obj = new SourceObj (alias, *SourceLocation::UnknownLocation);
    alias_obj->addDependency (name);
    SourceMap::iterator alias_it = definitions_.find (alias);
    if (alias_it != definitions_.end())
    {
        SourceObj& oldAlias = *(alias_it->second);
        alias_obj->setIsVariable (oldAlias.isVariable());
        alias_obj->setIsFunction (oldAlias.isFunction());
        definitions_.erase (alias_it);
    }
    definitions_.insert (make_pair (alias,alias_obj));
}

SourceFile&
SourceFile::findFunction (const string& name)
{
    SourceFile* file = (*(global_functions_.find (name))).second;
    return *file;
}

SourceFile&
SourceFile::getFileForPath (const string& path)
{
    return lookupFile (path);
}

const map<string,SourceFile*>
SourceFile::allFiles (void)
{
    return catalog_.files_;
}

SourceFile&
SourceFile::lookupFile (const string& path)
{
    if (aliases_.count (path) > 0)
    {
        string alias = (*(aliases_.find (path))).second;
        return lookupFile (alias);
    }
    if (catalog_.files_.count (path) > 0) {
        return *((*(catalog_.files_.find (path))).second);
    }

    size_t loc = path.find_first_of ('/');
    while (loc != string::npos) {
        string alias = path.substr (loc + 1);
        if (catalog_.files_.count (alias) > 0) {
            return *((*catalog_.files_.find (alias)).second);
        } else if (aliases_.count (path) > 0) {
            alias = (*(aliases_.find (alias))).second;
            return lookupFile (alias);
        }
        loc = path.find_first_of ('/', loc+1);
    }

    SourceFile* res = new SourceFile (path);
    return *res;
}

static void
updatePrefix (string& prefix, const string& path)
{
    int compare = path.compare (0, prefix.length(), prefix);
    while (compare != 0 && prefix.length() != 0) {
        size_t loc = prefix.find_last_of ('/');
        if (loc == string::npos) loc = 0;
        prefix = prefix.substr (0,loc);
        compare = path.compare (0, prefix.length(), prefix);
    }
}

void
SourceFile::removePrefixes (set<SourceFile*> files, const string& prefix)
{
    SourceFile* file;
    if (prefix.length() && files.size() > 1) {
        set<SourceFile*>::iterator iter;
        for (iter = files.begin(); iter != files.end(); iter++){
            file = *iter;
            file->setName(file->path_.substr (prefix.length() + 1, string::npos));
        }
    }
}

/// Given a map of file paths and their build options, adds the files (keys) to
/// the set of known files and associates the build options with them.
void
SourceFile::setCompileFlags (const map<string,BuildOptions>& options)
{
    if (options.empty()) return;
    map<string,BuildOptions>::const_iterator opt_it = options.begin();
    set<SourceFile*> prefixSet;
    string prefix = (*opt_it).first;
    while (opt_it != options.end()) {
        string path = (*opt_it).first;
        updatePrefix (prefix, path);
        SourceFile* file = new SourceFile (path);
        prefixSet.insert (file);
        file->flags_ = (*opt_it).second;
        file->hasFlags_ = true;
        opt_it++;
    }
    removePrefixes (prefixSet, prefix);
}

void
SourceFile::keepPreprocFile (void) const
{
    unlink_ = false;
}

void SourceFile::SetIsParsed (bool TrueFalse)  {  parsed_ = TrueFalse; }
bool SourceFile::IsParsed (void) const         {  return parsed_;      }

void SourceFile::SetPreProcName (const std::string& FileName)
{
    if (temp_) delete temp_;
    temp_ = new string (FileName);
    aliases_.insert(make_pair(preProcPath(),path()));
}
