//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_SOURCEFILE_H
#define AAP_SOURCEFILE_H

#include <string>
#include <vector>

#include "buildoptions.h"
#include "sourceobj.h"

namespace {
class SourceFileCatalog;
}

namespace aap {

/// Representation of a input source file and SourceObj declared in it.
class SourceFile
{
public:
    ~SourceFile (void);

    /// Returns the name representing the source file.
    const std::string& name        (void) const;
    /// Returns the path to the source file.
    const std::string& path        (void) const;
    /// Returns the path to the pre-processed version of the file.
    const std::string& preProcPath (void) const;

    /// Returns a mapping of functions defined in this file.
    const SourceMap&  functions   (void) const;
    /// Returns a mapping of all objects defined in this file. 
    const SourceMap&  definitions (void) const;
    /// Returns a mapping of global variables defined in this file. 
    const SourceMap&  globals     (void) const;

    /// Returns true if a definition is know for the given object name.
    bool  defIsKnown   (const std::string&);

    /// Returns true if the file must be included.
    bool  required     (void) const;

    const BuildOptions& buildOptions (void) const;

    /// Looks up a function defined in this file. 
    static SourceFile& findFunction (const std::string&);

    /// Returns the SourceFile associated with the given path string.
    static SourceFile& getFileForPath (const std::string&);

    /// Returns all known files, keyed by their path.
    static const std::map<std::string,SourceFile*> allFiles (void);

    /// Adds files to set of known files and associates build options with them.
    static void setCompileFlags (const std::map<std::string, BuildOptions>&);

    /// Returns the named source object as defined in this file.
    SourceObjRef lookupDef (const std::string&) const;

    /// Registers the given object with this file.
    void registerDef  (SourceObjRef);

    /// Makes an existing definition depend on a single other object.
    void aliasDef (const std::string& name, const std::string& alias);

    /// Check and set file parsing
    void SetIsParsed (bool);
    bool IsParsed    (void) const;

    /// Set Preprocess file name
    void SetPreProcName (const std::string&);

    // Indicates that the preprocessed file should be kept after parsing.
    void keepPreprocFile (void) const;

private:

    SourceFile (const std::string& path);

    static SourceFile& lookupFile (const std::string& path);

    // Strips a [common] prefix from the path of several files
    static void removePrefixes (std::set<SourceFile*>, const std::string&);

    void setName (const std::string&);

    std::string          name_;
    std::string          path_;        // path of the parsed file
    mutable SourceMap    definitions_; // object name -> definition (location)
    SourceMap            functions_;   // a subset of definitions
    SourceMap            globals_;     // data objects, a subset of definitions_
    mutable std::string *temp_;        // temporary file (preprocessed)
    mutable bool         unlink_; // should temp_ be removed at destruction
    bool                 parsed_; // has this file been parsed

    BuildOptions flags_;
    bool         hasFlags_;

    // track the files already parsed for reuse
    static SourceFileCatalog catalog_;

    // map an alternate name (e.g. preprocessed) to a canonical one
    static std::map<std::string,std::string> aliases_;

    // map function names to the source file containing them
    static std::map<std::string,SourceFile*> global_functions_;


};

} // namespace aap

#endif
