//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2011, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "sourcefileparser.h"

#include <fstream>
#include <iostream>
#include <set>
#include <sysexits.h>
#include <sys/wait.h>

#include "ast/aapdiagnosticclient.h"
#include "ast/aapaction.h"
#include "ast/aapaddinfoaction.h"
#include "buildoptions.h"
#include "graph/graphutils.h"
#include "graph/partitioner.h"
#include "profiling/hpcdatabase.h"
#include "profiling/SloDatabase.h"
#include "sourcefile.h"
#include "utils/options.h"
#include "ast/attrinfo.h"

#include "config/config.h"

using namespace std;

static const string kFunction = "function ";

#include "clang/Driver/Compilation.h"
#include "clang/Driver/Driver.h"
#include "clang/Driver/Tool.h"
#include "clang/Frontend/ChainedDiagnosticClient.h"
#include "clang/Frontend/CompilerInvocation.h"
#include "clang/Frontend/CompilerInstance.h"
#include "clang/Frontend/DiagnosticOptions.h"
#include "clang/Frontend/FrontendDiagnostic.h"
#include "clang/Frontend/TextDiagnosticPrinter.h"

#include "llvm/LLVMContext.h"
#include "llvm/Module.h"
#include "llvm/ADT/OwningPtr.h"
#include "llvm/ADT/SmallString.h"
#include "llvm/ExecutionEngine/ExecutionEngine.h"
#include "llvm/Support/Debug.h"
#include "llvm/Support/ManagedStatic.h"
#include "llvm/Support/raw_ostream.h"
#include "config/llvm.h"

using namespace clang;
using namespace clang::driver;
using namespace clang::frontend;
using namespace llvm;
using namespace aap;


llvm::sys::Path GetExecutablePath(const char *Argv0) {
  // This just needs to be some symbol in the binary; C++ doesn't
  // allow taking the address of ::main however.
  void *MainAddr = (void*) (intptr_t) GetExecutablePath;
  return llvm::sys::Path::GetMainExecutable(Argv0, MainAddr);
}


static const char* argv0;
static clang::driver::Driver* TheDriver;
static clang::Diagnostic* Diags;
static bool diagnosticsOccured_ = false;

// Header include paths.
static cl::list<std::string> includes_(
    "I", cl::value_desc("path"), cl::Hidden, cl::Prefix,
    cl::desc("header search path"));

// Header include paths to be treated like system headers.
static cl::list<std::string> isystem_(
    "isystem", cl::value_desc("path"), cl::Hidden,
    cl::desc("header search path to be treated as a system header paths"));

void SourceFileParser::Initialize(const char* Argv0)
{
    argv0 = Argv0;
    llvm::sys::Path Path = GetExecutablePath(argv0);
    TextDiagnosticPrinter* DiagClient
        = new TextDiagnosticPrinter(llvm::errs(), DiagnosticOptions());

    Diags = new Diagnostic(DiagClient);
    TheDriver = new Driver(Path.getBasename(), llvm::sys::getHostTriple(),
                           "a.out", false, false,*Diags);

    TheDriver->setTitle("AAP rpu generator");
}


void
SourceFileParser::ParseFilesInGraph(Agraph_t* graph)
{
    Agnode_t*     node;
    Agnodeinfo_t* nodeInfo;
    ProcedureRef  procedure;
    Agraphinfo_t* graphInfo;

    if (Options::Attributes()) {
        graphInfo = GraphUtils::getInfo(graph);
        AttrInfo::SetTotalTime(graphInfo->total_time);
    }
    for (node = agfstnode(graph); node != NULL; node = agnxtnode(graph, node))
    {
        nodeInfo = GraphUtils::getInfo (node);
        if (!nodeInfo) continue;

        string name (kFunction);
        string path;
        procedure = nodeInfo->procedure;
        name.append(procedure->name());
        path = procedure->file();

        if (0 != path.compare("~unknown-file~") && (0 == path.find("./src/")))
            path = path.substr(5);

        SourceFile& file = SourceFile::getFileForPath(path);
        SourceFileParser::ParseFile(file);
        const SourceObjRef function = file.lookupDef(name);
        if (function && function != SourceObj::UnknownType) {
            procedure->setFunction(function);
            if (Options::Attributes()) {
                AttrInfo *AT;
                if (function->isStatic())
                    AT = AttrInfo::SearchAttr(procedure->name(),file.name());
                else
                    AT = AttrInfo::SearchAttr(procedure->name());
                if (AT)
                    AT->AddTime(nodeInfo->time);
            }
        }
    }
}

void SourceFileParser::Preprocess (SourceFile &file, const string& outname)
{
    string OutputName;
    int pid, status;
    int FileFlagCount=0, GlobalFlCount=0, TotalFlgCount;
    const BuildOptions& Flags = file.buildOptions();
    const int Argc = 6;
    const vector<string>& FileFlags = Flags.preprocFlags();
    const vector<string>& GlobalFls = BuildOptions::globalPreprocFlags();

    if (outname.length()) {
        OutputName = outname;
    } else {
        OutputName = (Options::outPath());
        OutputName.append("aap.preproc.");
        OutputName.append(file.name());
    }
    file.SetPreProcName(OutputName);

    vector<string> argVector;
    argVector.push_back(CLANG_PATH);
    argVector.push_back("-cc1");
    argVector.push_back("-E");
    argVector.push_back(file.path());
    argVector.push_back("-o");
    argVector.push_back(OutputName);

    argVector.insert(argVector.end(), FileFlags.begin(), FileFlags.end());
    argVector.insert(argVector.end(), GlobalFls.begin(), GlobalFls.end());

    vector<string>::const_iterator iter, end;
    for (iter = includes_.begin(), end = includes_.end(); iter != end; ++iter)
        argVector.push_back(string("-I") + *iter);
    for (iter = isystem_.begin(), end = isystem_.end(); iter != end; ++iter)
        argVector.push_back(string("-I") + *iter);

    if (Options::verbose()) {
        for (unsigned i = 0; i < argVector.size(); i++)
            llvm::errs() << argVector[i] << " ";
        llvm::errs() << "\n";
    }

    // Create a NULL terminated argument list for exec style call.
    char** Args = new char*[argVector.size() + 1];
    assert(Args != NULL);
    Args[argVector.size()] = NULL;
    for (unsigned i = 0; i < argVector.size(); ++i)
        Args[i] = strdup(argVector[i].c_str());

    status = 0;
    pid = fork();
    if (pid) wait(&status);     // parent waits
    else execvp (Args[0], Args); // child preprocesses

    // if the preprocessing fails and we have been explicitly told how to
    // compile this file, then fail. otherwise, we assume the file may be a
    // system header indicated by the profiling system and allow failure
    if (WEXITSTATUS(status)) {
        cerr << "preprocessing failed" << endl;
        exit (WEXITSTATUS (status));
    }

    for (unsigned i=0; i < argVector.size(); i++)
        free (Args[i]);
    delete Args;
}

static void addHeaderSearchPaths(clang::HeaderSearchOptions& options)
{
    // Angled and System are defined in clang::frontend.
    for (unsigned i = 0; i < includes_.size(); ++i)
        options.AddPath(includes_[i], Angled, true, false, false);
    for (unsigned i = 0; i < isystem_.size(); ++i)
        options.AddPath(isystem_[i], System, true, false, false);
}

void SourceFileParser::ParseFile(SourceFile& file)
{
  if (file.IsParsed()) return;

  llvm::SmallVector<const char *, 16> Args;
  Preprocess(file);
  Args.push_back(argv0);
  Args.push_back(file.preProcPath().c_str());
  Args.push_back("-fsyntax-only");
  llvm::OwningPtr<Compilation> C(TheDriver->BuildCompilation(Args.size(),
                                                             Args.data()));
  if (!C) exit(EXIT_SUCCESS);

  const driver::JobList &Jobs = C->getJobs();

  driver::JobList::const_iterator jobIter = Jobs.begin();
  driver::JobList::const_iterator jobEnd = Jobs.end();
  while (jobIter != jobEnd) {
    const clang::driver::Job* job = *jobIter;
    if (!isa<driver::Command>(*job)) {
      llvm::SmallString<256> Msg;
      llvm::raw_svector_ostream OS(Msg);
      C->PrintJob(OS, *job, "; ", true);
      Diags->Report(diag::err_fe_expected_compiler_job) << OS.str();
      exit(EX_SOFTWARE);
    }

    const driver::Command *Cmd = cast<driver::Command>(*Jobs.begin());
    if (llvm::StringRef(Cmd->getCreator().getName()) != "clang") {
      Diags->Report(diag::err_fe_expected_clang_command);
      exit(EX_SOFTWARE);
    }

    // Initialize a compiler invocation object from the clang (-cc1) arguments.
    const driver::ArgStringList &CCArgs = Cmd->getArguments();
    llvm::OwningPtr<CompilerInvocation> CI(new CompilerInvocation);
    const char** data = const_cast<const char **>(CCArgs.data());
    CompilerInvocation::CreateFromArgs(*CI, data, data + CCArgs.size(), *Diags);

    // Show the invocation, with -v.
    if (CI->getHeaderSearchOpts().Verbose) {
      llvm::errs() << "clang invocation:\n";
      C->PrintJob(llvm::errs(), C->getJobs(), "\n", true);
      llvm::errs() << "\n";
    }

    // Create a compiler instance to handle the actual work.
    CompilerInstance Clang;
    Clang.setLLVMContext(new llvm::LLVMContext);
    Clang.setInvocation(CI.take());
    addHeaderSearchPaths(Clang.getHeaderSearchOpts());

    // Create the compilers actual diagnostics engine.
    Clang.createDiagnostics(CCArgs.size(), const_cast<char**>(data));
    if (!Clang.hasDiagnostics()) {
      llvm::errs() << "failed to create diagnostic\n";
      exit(EX_SOFTWARE);
    }

    DiagnosticClient* client = new AAPDiagnosticClient();
    Diagnostic& diag = Clang.getDiagnostics();
    DiagnosticClient* OldClient = diag.takeClient();
    if (Options::verbose())
        client = new clang::ChainedDiagnosticClient(client, OldClient);
    diag.setClient(client);
    diag.setIgnoreAllWarnings(true);

    // Infer the builtin include path if unspecified.
    if (Clang.getHeaderSearchOpts().UseBuiltinIncludes &&
        Clang.getHeaderSearchOpts().ResourceDir.empty())
    {
      void *MainAddr = (void*) (intptr_t) GetExecutablePath;
      Clang.getHeaderSearchOpts().ResourceDir =
        CompilerInvocation::GetResourcesPath(argv0, MainAddr);
    }

    llvm::OwningPtr<AAPAction> action(new AAPAction);
    action->setFile(file);
    if (!Clang.ExecuteAction(*action)) {
        if (diag.hasErrorOccurred()) {
            llvm::errs() << "errors parsing " << file.path() << "\n";
            diagnosticsOccured_ = true;
        }
    }

    if (Options::Attributes()) {
        llvm::OwningPtr<AAPAddInfoAction> infoaction(new AAPAddInfoAction);
        infoaction->setFile(file);
        if (!Clang.ExecuteAction(*infoaction)) {
            if (diag.hasErrorOccurred()) {
                llvm::errs() << "errors parsing " << file.path() << "\n";
                diagnosticsOccured_ = true;
            }
        }
    }

    ++jobIter;
  }
  file.SetIsParsed(true);
}

void SourceFileParser::ParseKnownFiles (void)
{
    const map<string,SourceFile*>& files = SourceFile::allFiles();
    map<string,SourceFile*>::const_iterator I, E = files.end();
    for (I = files.begin(); I != E; I++)
        ParseFile(*(I->second));
}

bool SourceFileParser::DiagnosticsOccured(void)
{
    return diagnosticsOccured_;
}
