//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <limits.h>
#include <stdlib.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <string>

#include "clang/Basic/SourceLocation.h"
#include "clang/Basic/SourceManager.h"

#include "sourcefile.h"
#include "sourcelocation.h"

using namespace std;
using namespace aap;

const SourceLocation* SourceLocation::UnknownLocation = new SourceLocation ();

// constructor used only for the unknown location
SourceLocation::SourceLocation(void) :
  declaration_only_  (false),
  needs_semi_colon_  (false),
  isUnknownLocation_ (true),
  original_line_     (0)
{}

SourceLocation::SourceLocation(clang::SourceRange& range,
                               clang::SourceManager& manager,
                               unsigned int EndOffset,
                               unsigned int StartOffset) :
  declaration_only_  (false),
  needs_semi_colon_  (false),
  isUnknownLocation_ (false),
  original_line_     (0)
{
    start_ = range.getBegin();
    StartLine = manager.getInstantiationLineNumber(start_);
    end_ = range.getEnd();
    if (end_.isValid()) {
        EndLine = manager.getInstantiationLineNumber(end_);
        StartPos =
            manager.getDecomposedInstantiationLoc(start_).second + StartOffset;
        EndPos   =
            manager.getDecomposedInstantiationLoc(end_).second + EndOffset;
    }
    else {
        EndLine   = StartLine;
        StartPos  = 0;
        EndPos    = 0;
    }

    file_ = manager.getBufferName(manager.getInstantiationLoc(start_));
    original_file_ = manager.getPresumedLoc(start_).getFilename();
    original_line_ = manager.getPresumedLoc(start_).getLine();
}

SourceLocation
SourceLocation::declaration (void) const
{
    SourceLocation dec = *this;
    dec.declaration_only_ = true;
    return dec;
}

int
SourceLocation::first_line(void) const
{
    return StartLine;
}

int
SourceLocation::lastLine (void) const
{
    return EndLine;
}

int
SourceLocation::originalLine (void) const
{
    return original_line_;
}

unsigned
SourceLocation::startPosition (void) const
{
    return StartPos;
}

unsigned
SourceLocation::endPosition (void) const
{
    return EndPos;
}

const SourceFile&
SourceLocation::file(void) const
{
    return SourceFile::getFileForPath(file_);
}

bool
SourceLocation::declarationOnly (void) const
{
    return declaration_only_;
}

bool
SourceLocation::needsSemiColon (void) const
{
    return needs_semi_colon_;
}

bool
SourceLocation::isUnknownLocation (void) const
{
    return isUnknownLocation_;
}

bool
SourceLocation::operator< (const SourceLocation& other) const
{
    int start = StartLine;
    int otherStart = other.StartLine;
    int end = EndLine;
    int otherEnd = other.EndLine;

  if (start < otherStart) return true;
  if (start == otherStart && end < otherEnd) return true;
  return false;
}

bool
SourceLocation::operator== (const SourceLocation& other) const
{
    return (start_ == other.start_ && end_ == other.end_);
}

void
SourceLocation::setOriginalLocation (int line, string& file)
{
    original_line_ = line;
    original_file_ = file;
}

string
SourceLocation::originalFile (void) const
{
    return original_file_;
}

void
SourceLocation::set_needs_semi_colon (bool needs)
{
    needs_semi_colon_ = needs;
}
