//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>

#include <sstream>
#include <string>

#include "sourceextractor.h"
#include "sourcefile.h"
#include "sourcelocation.h"
#include "sourceobj.h"

using namespace std;
using namespace aap;

SourceObjRef SourceObj::UnknownType ((SourceObj*) NULL);

SourceObj::SourceObj (const string& name,
                      const SourceLocation& source,
                      const string& type, 
                      const string& post, unsigned long ID)
    : name_          (name),
      source_        (source),
      type_          (type),
      initSize_      (0),
      isDefinition_  (false),
      isExtern_      (false),
      isFunction_    (false),
      isStatic_      (false),
      isVariable_    (false),
      usesVaArgs_    (false)
{
    ObjectID = ID;
    PostName = post;
    isFunctionScope_ = false;
    hasAttribute_ = false;
}

static string
nonFunctionName (const string& name)
{
    if (name.compare (0, 9, "function ") == 0)
        return name.substr(9, string::npos);
    else
        return name;
}

/// This method is only valid for objects representing a function definition.
SourceObjRef
SourceObj::prototype (void) const
{
    assert (isFunction_ && isDefinition_);
    SourceObjRef proto (new SourceObj());
    proto->name_ = name_.substr (9); //"function "
    proto->source_ = source_.declaration();
    string originalFile = source_.originalFile();
    proto->source_.setOriginalLocation (source_.originalLine(), originalFile);

    proto->isFunction_ = true;
    proto->isStatic_ = isStatic_;
    return proto;
}

string
SourceObj::name(void) const
{
    return name_;
}

string
SourceObj::baseName (void) const
{
    if (isFunction_) return nonFunctionName (name_);
    else return name_;
}

string
SourceObj::mangledName (void) const
{
    string res = nonFunctionName(name_) + "_" + source_.file().name();
    return cleanName (res);
}

unsigned
SourceObj::initSize (void) const
{
    return initSize_;
}

bool
SourceObj::isVariable (void) const
{
    return isVariable_;
}

bool
SourceObj::isDefinition (void) const
{
    return isDefinition_;
}

bool
SourceObj::isExtern (void) const
{
    return isExtern_;
}

bool
SourceObj::isFunction (void) const
{
    return isFunction_;
}

bool
SourceObj::isStatic (void) const
{
    return isStatic_;
}

bool
SourceObj::usesVaArgs (void) const
{
    return usesVaArgs_;
}

void
SourceObj::setInitSize (unsigned size)
{
    initSize_ = size;
}

void
SourceObj::setIsVariable (bool isVariable)
{
    isVariable_ = isVariable;
}

void
SourceObj::setIsDefinition (bool isDefinition)
{
    isDefinition_ = isDefinition;
}

void
SourceObj::setIsExtern (bool isExtern)
{
    isExtern_ = isExtern;
}

void
SourceObj::setIsFunction (bool isFunction)
{
    isFunction_ = isFunction;
}

void
SourceObj::setIsStatic (bool isStatic)
{
    isStatic_ = isStatic;
}

void
SourceObj::setUsesVaArgs (bool usesVaArgs)
{
    usesVaArgs_ = usesVaArgs;
}

string
SourceObj::globalName (void) const
{
    if (isStatic_ && (&(file())) != NULL) {
        return name_ + "/" + file().name();
    } else {
        return name_;
    }
}

string SourceObj::GetPostName (void) const
{
    return PostName;
}

string
SourceObj::type (void) const
{
    return type_;
}

bool
SourceObj::operator< (const SourceObj& other) const
{
    if (ObjectID != other.ObjectID)
        return ObjectID < other.ObjectID;
    if (source_ == other.source_) {
        return (name_.compare (other.name_) < 0);
    } else {
        return source_ < other.source_;
    }
}

void
SourceObj::stripLocation (void)
{
    source_ = *SourceLocation::UnknownLocation;
}

void
SourceObj::addDependency (const string& type_name)
{
    if (type_name.length() == 0) return;
    if (type_name.compare (name_) != 0)
        dependencies_.insert (type_name);
}

bool
SourceObj::compare (const SourceObj& a, const SourceObj& b)
{
    return a.name_ < b.name_;
}

const set<string>&
SourceObj::dependencies (void) const
{
    return dependencies_;
}

const SourceFile&
SourceObj::file (void) const
{
    return source_.file();
}

const SourceLocation&
SourceObj::source (void) const
{
    return source_;
}

string
SourceObj::cleanName (const string& input)
{
    size_t loc;
    string res = input;
    while ((loc = res.rfind (".")) != string::npos)
        res.replace (loc, 1, 1, '_');
    while ((loc = res.rfind ("-")) != string::npos)
        res.replace (loc, 1, 1, '_');
    while ((loc = res.rfind ("/")) != string::npos)
        res.replace (loc, 1, 1, '_');
    return res;
}

void SourceObj::setFunctionScope (const std::string& FunctionName)
{
    isFunctionScope_ = true;
    ScopeName = FunctionName;
}

bool SourceObj::isFunctionScope (void)
{
    return isFunctionScope_;
}

std::string SourceObj::ScopeFunction (void) const
{
    return ScopeName;
}

void SourceObj::addPragma(const PragmaInfo& pragma)
{
    pragmas_.insert(pragma);
}

void SourceObj::removePragma(const PragmaInfo& pragma)
{
    pragmas_.erase(pragma);
}

void SourceObj::SetAttribute (const std::string Attribute)
{
    if (hasAttribute_ == false) {
        type_.append(Attribute);
        hasAttribute_ = true;
    }
}

const std::set<PragmaInfo>& SourceObj::pragmas() const
{
    return pragmas_;
}
