//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_FUNCTIONPARSER_TYPEDEF_H
#define AAP_FUNCTIONPARSER_TYPEDEF_H

#include <map>
#include <set>
#include <string>

#include "analysis/pragmainfo.h"
#include "sourcelocation.h"

#include "llvm/ADT/IntrusiveRefCntPtr.h"

namespace aap {

class SourceFile;
class SourceObj;

typedef llvm::IntrusiveRefCntPtr<SourceObj> SourceObjRef;

/// Represents an object (variable, function, type definition) in source code.
class SourceObj : public llvm::RefCountedBase<SourceObj>
{
public:
    SourceObj (const std::string& name = "",
               const SourceLocation& loc = *(SourceLocation::UnknownLocation),
               const std::string& type = "",
               const std::string& post = "", unsigned long ID=0);

    std::string name         (void) const;
    std::string baseName     (void) const;
    std::string globalName   (void) const;
    std::string mangledName  (void) const;
    std::string type         (void) const;
    std::string GetPostName  (void) const;

    unsigned initSize  (void) const;

    bool directExport  (void) const;
    bool isVariable    (void) const;
    bool isDefinition  (void) const;
    bool isExtern      (void) const;
    bool isFunction    (void) const;
    bool isStatic      (void) const;
    bool usesVaArgs    (void) const;

    // Remove the location information so the object will not be exported
    void stripLocation (void);

    void addDependency  (const std::string&);

    void setInitSize     (unsigned);

    void setIsVariable    (bool);
    void setIsDefinition  (bool);
    void setIsExtern      (bool);
    void setIsFunction    (bool);
    void setIsStatic      (bool);
    void setUsesVaArgs    (bool);

    void setFunctionScope (const std::string&);
    bool isFunctionScope (void);
    std::string ScopeFunction (void) const;

    const std::set<std::string>& dependencies (void) const;
    const std::set<std::string>& localNames   (void) const;
    const std::set<std::string>& localStatics (void) const;

    const SourceFile&            file         (void) const;
    const SourceLocation&        source       (void) const;

    bool operator<      (const SourceObj&) const;
    static bool compare (const SourceObj&, const SourceObj&);

    /// Returns the prototype for a function.
    SourceObjRef prototype (void) const;

    void addPragma(const PragmaInfo&);
    void removePragma(const PragmaInfo&);
    const std::set<PragmaInfo>& pragmas(void) const;

    /// Add function attribute
    void SetAttribute (const std::string);    

    /// Singleton representation of an invalid object.
    static SourceObjRef UnknownType;

    /// Returns true if functions should record names of local variables
    static bool trackFunctionLocals (void);

    /// Returns a version of the input that is safe to use as an object name.
    static std::string cleanName (const std::string&);

private:
    SourceObj& operator= (const SourceObj&); // (use SourceObjRef, don't copy)

    std::string    name_;
    SourceLocation source_;
    std::string    type_;
    unsigned       initSize_;
    std::string    PostName;

    std::set<std::string>  dependencies_;

    bool isDefinition_;         // versus declaration aka prototype
    bool isExtern_;
    bool isFunction_;
    bool isStatic_;
    bool isVariable_;
    bool usesVaArgs_;           // true for functions that take va_args (...)

    bool isFunctionScope_;// extern or static variable declared within function
    bool hasAttribute_;
    std::string ScopeName;// For static and extern variables within function

    void clearFlags (void);
    unsigned long ObjectID; // Determines the order of source object creations

    std::set<PragmaInfo> pragmas_;
};

typedef std::map<std::string, SourceObjRef> SourceMap;

} // namespace aap

#endif
