//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <stdlib.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <string>

using namespace std;

namespace CallConverter 
{

size_t
findParenMatch (string& dec, size_t open)
{
    size_t close;
    size_t next = open + 1;
    int count = 1;
    assert (open != string::npos);

    // loop until we have balanced parentheses
    while (count > 0) {
        close = dec.find_first_of (')', next);
        open = dec.find_first_of ('(', next);
        if (open < close) {
            count ++;
            next = open + 1;
        } else {
            count--;
            next = close + 1;
        }
    }
    return close;
}

// strip everything after the parameter list
static void
removePostParen (string& dec)
{
    // find the open paren
    size_t open = dec.find_first_of ('(');
    size_t close = findParenMatch (dec, open);
    dec.replace (close+1, string::npos, "");
}

static inline bool
identifierCharacter(char t)
{
    return (isalnum (t) || (t == '_'));
}

static string
extractFunctionPointerName (const string& param)
{
    // extract the word within the first paren pair
    size_t open  = param.find_first_of ('(');
    size_t close = param.find_first_of (')', open);
    size_t aster = param.find_first_of ('*', open);

    if (aster != string::npos && aster > open)
        open = aster;

    return param.substr (open + 1, close - open - 1);
}

string
extractParamName (const string& param)
{
    // ignore any array dimensions
    size_t brace = param.find_first_of ('[');

    if (param.find_first_of ('(') != string::npos && brace == string::npos)
        return extractFunctionPointerName (param);

    // the last character of the last 'word'
    size_t back = param.find_last_not_of (' ', brace - 1);
    // the white space before the name
    size_t front = param.find_last_of (' ', back);

    // pass over any trailing white space
    while (front > back && front != string::npos)
        front = param.find_last_of (' ', front-1);
    if (front == string::npos) front = 0;
    else front++;

    // locate the last asterisk
    size_t aster = param.find_last_of ('*', back);
    if (aster != string::npos && aster >= front)
        front = aster + 1;

    // pass over any array indicators
    while (param[back] == ']') {
        size_t open = param.find_last_of ('[', back);
        if (open !=string::npos) back = open - 1;
    }

    while (param[back] == ')') back--;

    // take whatever is after that point
    return param.substr (front, back-front+1);
}

    // Finds the first comma that is not within a pair of parentheses
static size_t
nextParamSeparator (string& params)
{
    size_t close = 0;
    size_t open  = params.find_first_of ('(');
    size_t comma = params.find_first_of (',', close);

    while (open < comma) {
        int count = 0;
        close = params.find_first_of (')', open + 1);
        // count the number of open parens before that first close paren
        if (close != string::npos) {
            while (open != string::npos && open < close) {
                open = params.find_first_of ('(', open + 1);
                count++;
            }
        }
        // find the matching close paren
        while (count > 1) {
            close = params.find_first_of (')', close + 1);
            assert (close != string::npos);
            count--;
        }
        // find the next open paren, and next comma
        open = params.find_first_of ('(', close + 1);
        comma = params.find_first_of (',', close + 1);
        if (comma < open) open = string::npos;
    }
    return comma;
}

void removeTypeInfo (string& call)
{
    size_t open = call.find_first_of ('(') + 1;
    size_t close = call.find_last_of (')');
    string old_params = call.substr (open, close-open) + ", ";
    string new_params;
    string param;
    size_t comma = nextParamSeparator (old_params);
    while (comma != string::npos) {
        param = old_params.substr (0, comma);
        old_params = old_params.substr (comma + 1, string::npos);
        param = extractParamName (param);
        if (new_params.length() > 0) new_params.append (", ");
        new_params.append (param);
        comma = nextParamSeparator (old_params);
    }
    if (new_params.compare ("void") == 0) new_params = "";
    call.replace (open, close-open, new_params);
}

static
void adjustCallReturn (string& call)
{
    string prefix = "";

    if (call.find ("void") == string::npos) prefix = "return ";

    size_t loc = call.find_first_of ('(');
    assert (loc != string::npos);
    loc = call.find_last_not_of (' ', loc-1);
    loc = call.find_last_of (' ', loc);
    while (call[loc+1] == '*') ++loc;
    call.replace (0, loc+1, prefix);
}

    void convertDeclToCall (string& call)
    {
        removePostParen (call);
        removeTypeInfo (call);
        adjustCallReturn (call);
    }
}
