//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <assert.h>
#include <limits.h>
#include <stdlib.h>

#include <iostream>
#include <fstream>
#include <string>

#include "utils/options.h"

using namespace std;
using namespace llvm;

namespace Options {

cl::opt<bool> breakout_("breakout",
                        cl::desc("Stop after intermediate file generation"),
                        cl::init(false));

cl::opt<bool> padArrays_("pad-arrays",
                        cl::desc("Pad array dimensions"),
                        cl::init(false));

cl::opt<bool> promoteStatic_(
  "promote-statics",
  cl::desc("Make variable declarations 'static' when possible"),
  cl::init(false));

cl::opt<bool> keep_graph_("g", cl::desc("Dump intermediate graph files"),
                          cl::init(false));
cl::alias keep_graph_long("graphs", cl::desc("alias of -g"),
                          cl::aliasopt(keep_graph_));

cl::opt<std::string>
inputGraph_("r", cl::value_desc("filename"),
            cl::desc("Input RPU graph for partitioning (profile is ignored)"));
cl::alias inputGraph_long("rpu-graph", cl::desc("alias of -r"),
                          cl::aliasopt(inputGraph_));

cl::opt<std::string>
outPath_("o",cl::desc("Absolute path to output directory"),
        cl::init("/tmp/RPUs/"));
cl::alias outPath_long("output", cl::desc("alias of -o"),
                       cl::aliasopt(outPath_));

cl::opt<std::string>
programName_("n", cl::desc("name for the output program"), cl::init("a.out"));
cl::alias programName_long("program-name", cl::desc("alias of -n"),
                           cl::aliasopt(programName_));

cl::opt<std::string>
rpuPrefix_("rpu-prefix",cl::desc("Graph prefix for rpu names"),
           cl::Hidden, cl::init("cluster_rpu"));

cl::opt<double>
pruneThreshold_("c", cl::init(0),
                cl::desc("Threshold percentage for pruning"));
cl::alias pruneThreshold_long("prune-threshold", cl::desc("alias of -c"),
                              cl::aliasopt(pruneThreshold_));

cl::opt<unsigned>
rpuMaxFunctions_("f",
                 cl::desc("Maximum number of functions per RPU"),
                 cl::init(10));
cl::alias rpuMaxFunctions_long("max-functions", cl::desc("alias of -f"),
                               cl::aliasopt(rpuMaxFunctions_));


cl::opt<unsigned>
rpuMaxLines_("l",
             cl::desc("Maximum number of lines of code per RPU"),
             cl::init(UINT_MAX));
cl::alias rpuMaxLine_long("max-lines", cl::desc("alias of -l"),
                          cl::aliasopt(rpuMaxLines_));

cl::opt<Options::PartitionerType>
partitioner_("partitioner",
            cl::desc("Partitioner Type (default: mincut)"),
            cl::values(
              clEnumValN(Options::MINCUT, "mincut", "simple minimum edge cut"),
              clEnumValN(Options::RECOMBINE, "recombine",
                "mincut with recombination"),
              clEnumValN(Options::COMMUNITY, "community",
                         "shortest path betweenness"),
              clEnumValN(Options::RANDOMWALKS, "randomwalks",
                         "random walk betweenness"),
              clEnumValN(Options::LIMBO, "limbo", "agglomerative"),
              clEnumValN(Options::COMMUNLIMBO, "communlimbo",
                         "community then limbo"),
              clEnumValN(Options::RANDOMLIMBO, "randomlimbo",
                         "randomwalks then limbo"),
              clEnumValEnd));

cl::opt<Options::Database>
database_ ("database",
           cl::desc("Profiling Database type for Graph Creation (default: hpc-db)"),
           cl::values(
               clEnumValN(Options::HPCDBS, "hpc-db",
                          "HPC database" ),
               clEnumValN(Options::SLODBS, "slo-db",
                          "SLO database"),
               clEnumValN(Options::HPCSLO, "hpcslo",
                          "HPC database with SLO features"),
               clEnumValN(Options::SLOHPC, "slohpc",
                          "SLO database with HPC features"),
               clEnumValEnd),
           cl::init(Options::HPCDBS));

cl::opt<bool> verbose_("verbose", cl::desc("verbose output"), cl::init(false));

cl::opt<bool>
splitLeftover_("split-leftovers", cl::init(false),
              cl::desc("divide leftover source when generating RPUs"));

}

cl::opt<bool>
attributes_("attributes", cl::init(false),
            cl::desc("analyze files and set attributes"));

cl::opt<bool>
debugattributes_("debug-attributes", cl::init(false),
                 cl::desc("debug attribute information"));

/// If no input graph is given, an empty string is returned.
const string&
Options::inputGraph (void)
{
    return inputGraph_;
}

/// The default value is /tmp/RPUs/.
const string&
Options::outPath (void)
{
    return outPath_;
}

/// The default value is "a.out".
const string&
Options::programName (void)
{
    return programName_;
}

/// Defaults to false.
bool
Options::breakout (void)
{
    return breakout_;
}

/// Defaults to false.
bool
Options::hasGraph (void)
{
    return inputGraph_.getNumOccurrences();
}

/// Defaults to false.
bool
Options::padArrays (void)
{
    return padArrays_;
}

/// By default, only the files mentioned in the call-graph need to be
/// parsed. In some cases it is necessary to parse all files (for example if
/// any form of global analysis is performed).
bool
Options::parseAllFiles (void)
{
    return padArrays_ || promoteStatic() || splitLeftover();
}

/// Code that is not explicitly placed into an RPU (e.g., based on profile
/// information) is considered 'leftover'. This code can optionally be left
/// intact when it is output as an RPU. If leftover code is split, then all
/// files have to be parsed.
bool
Options::splitLeftover (void)
{
    return splitLeftover_;
}

/// Objects that are only used in a single RPU can potentially be marked with
/// static storage. This may allow later compilation to perform greater
/// optimization, but also requires more analysis by the AAP.
bool
Options::promoteStatic (void)
{
    return promoteStatic_;
}

/// Defaults to false.
bool
Options::keep_graph (void)
{
    return keep_graph_;
}

/// Defaults to false.
bool
Options::verbose (void)
{
    return verbose_;
}

/// Defaults to false.
bool
Options::Attributes (void)
{
    return attributes_;
}

/// Defaults to false.
bool
Options::DebugAttribute (void)
{
    return debugattributes_;
}

/// Defaults to "cluster_rpu". This is the value used within the graph
/// representations, not the file name prefix for output source RPUs.
const string&
Options::rpuPrefix (void)
{
    return rpuPrefix_;
}

/// Defaults to zero, which disables pruning.
double
Options::pruneThreshold (void)
{
    return pruneThreshold_;
}

/// Defaults to ten.
unsigned
Options::rpuMaxFunctions (void)
{
    return rpuMaxFunctions_;
}

/// Defaults to unlimited lines (actually, to the maximum unsigned integer
/// value).
unsigned
Options::rpuMaxLines (void)
{
    return rpuMaxLines_;
}

/// Defaults to MINCUT.
Options::PartitionerType
Options::partitioner (void)
{
    return partitioner_;
}

cl::opt<bool> summary_("summary", cl::init(false),
                       cl::desc("generate a summary of RPU changes"));

ostream& Options::summary (void)
{
    static ostream* stream = NULL;
    if (!stream) {
        if (summary_) {
            stream = &(std::cerr);
        } else {
           stream = new ofstream("/dev/null");
        }
    }
    return *stream;
}

static cl::opt<bool> ignoreProfile_
("ignore-profile", cl::init(false), cl::Hidden,
 cl::desc("skip graph generation from profiles and simply split input files"));

bool Options::ignoreProfile(void)
{
    return ignoreProfile_;
}

static cl::bits<Options::PragmaType>
injectPragmas_("inject-pragma",
               cl::desc("Comma separated list of pragmas to inject."),
               cl::CommaSeparated,
               cl::ValueOptional,
               cl::values(
                   clEnumValN(Options::kHpcProfilePragmas, "hpcprof",
                              "profile information from an HPC toolkit profile"),
                   clEnumValN(Options::kArrayPaddingPragmas, "padding",
                              "indicate paddable arrays"),
                   clEnumValN(Options::kAllPragmas, "",
                              "(empty list activates all)"),
                   clEnumValEnd));

bool Options::injectPragmas(PragmaType type)
{
    return (injectPragmas_.isSet(kAllPragmas) || injectPragmas_.isSet(type));
}

Options::Database Options::ProfilingDatabase (void)
{
    return database_;
}
