//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_OPTIONS_H
#define AAP_OPTIONS_H

#include <string>

#include "llvm/Support/CommandLine.h"
#include "config/llvm.h"

/// AAP command line options.
namespace Options
{

/// Represents the possible types of partitioner to use.
    enum PartitionerType
    {
        MINCUT,
        RECOMBINE,
        COMMUNITY,
        RANDOMWALKS,
        LIMBO,
        COMMUNLIMBO,
        RANDOMLIMBO
    };

enum PragmaType
{
    kAllPragmas,
    kHpcProfilePragmas,
    kArrayPaddingPragmas
};

/// Represents the possible types of partitioner to use.
enum Database
{
    HPCDBS, // HPC database
    SLODBS, // SLO database
    HPCSLO, // HPC database with additional information from SLO database
    SLOHPC  // SLO database with additional information from HPC database
};

/// Returns true if execution should stop after producing intermediate files.
    bool breakout      (void);
/// Returns true if an RPU graph was provided on the command line.
    bool hasGraph      (void);
/// Returns true if array padding analysis is requested.
    bool padArrays     (void);
/// Returns true if all known files need to be parsed.
    bool parseAllFiles (void);
/// Returns true if objects should be make static when possible.
    bool promoteStatic (void);
/// Returns true if intermediate graphs should be generated.
    bool keep_graph    (void);
/// Returns true if code not allocated to a specific RPU should be split.
    bool splitLeftover (void);
/// Returns true if run-time output should be verbose.
    bool verbose       (void);
/// Returns if attribute analysis is requested
    bool Attributes    (void);
/// Debug attribute information
    bool DebugAttribute(void);

/// Returns the path to the provided RPU graph.
    const std::string& inputGraph  (void);
/// Returns the path where generated files should be placed.
    const std::string& outPath     (void);
/// Returns the name of the final program.
    const std::string& programName (void);
/// Returns the name prefix to give the RPUs.
    const std::string& rpuPrefix   (void);

/// Returns the percentage of run-time below which nodes should be ignored.
    double pruneThreshold (void);
/// Returns the maximum number of functions allowed in an RPU.
    unsigned rpuMaxFunctions   (void);
/// Returns the maximum number of source code lines allowed in an RPU.
    unsigned rpuMaxLines       (void);

/// Returns the type of partition to apply.
    PartitionerType partitioner (void);

/// Returns true if the profile should be ignored.
bool ignoreProfile(void);

/// Returns the stream to which user summary should be generated.
std::ostream& summary(void);

/// Returns true if the given pragma type should be injected into RPUs.
bool injectPragmas(PragmaType);

/// Returns the type of profiling database to use
Database ProfilingDatabase (void);

} // namespace Options

#endif
