/* PACE Project, Rice University
 *
 * This code measures the parameters of the cache memory system.
 * 
 * First, it determines L1 size, associativity, and line size.
 * It uses size to determine associativity.
 *
 *
 */

#include "MemTest.h"
#include "BlackJackTimer.h"

void exit( int x );
double sqrt(double x);
int errno;

char * TimeStamp();
int GetOSPageSize();
char HostName[128];

double SingleTrial( int s );
void SetOfTrials( int sa[], int ca[], double ta[], int c );

static int    Sizes[MAX_TESTS_PER_RUN];
static int    Cycles[MAX_TESTS_PER_RUN];
static double Times[MAX_TESTS_PER_RUN];

int PreCon;

void PrintArray( char *s, int array[], int size )
{
  int i;

  fprintf(stderr,"\n\nContents of %s: (%d)\n",s,size);
  for (i=0;i<size;i++)
    {
      if ((i % 4) == 0)
	fprintf(stderr,"\n%10d:  ",i);
      fprintf(stderr,"%10d, ",array[i]);
    }
  fprintf(stderr,"\n");
}

static void DieWithMessage(char *currArg, int argNumber)
{
  fprintf(stderr,"Did not understand argument '%s' (%d).\n",
	  currArg,argNumber);
  fprintf(stderr,"Valid arguments are:\n");
  fprintf(stderr,"\t-g \tActivate internal debugging checks.\n");
  fprintf(stderr,"\t-l <name> \tUse <name> for log file.\n");
  fprintf(stderr,"\t-p \tDon't precondition the data.\n");
  fprintf(stderr,"\t-v \tPrint results (& debug info) incrementally.\n");
  exit(-1);  
}

int main(int argc, char *argv[])
{
  int i, j, arg, argNumber;
  char *currArg, *LogFileName;

  int    Count;

  Verbose   = 0;
  Debug     = 0;
  HeartBeat = 1;
  PreCon    = 1;  /* precondition the data */

  LogFileName = "./Log";

  /* Parse the command line parameters */
  for (argNumber = 1; argNumber < argc; argNumber++)
  {
    currArg = argv[argNumber];

    if (*currArg == '-')
    {
      currArg++; 
      while (*currArg != '\0')
      {
	switch(*currArg)
        {
	  case 'd':
	    Debug++;
	    break;
	  case 'g':
	    Debug++;
	    break;
  	  case 'h':
	    HeartBeat++;
	    break;
	  case 'l':
	    argNumber++;
	    LogFileName = argv[argNumber];
	    break;
	  case 'v':
            Verbose++;
	    break;
	  default:
	    DieWithMessage(currArg,argNumber);
	}
	currArg++;
      }
    }
  }

  /* Set up the Log File */
  fprintf(stderr,"Log File Name is '%s'.\n",LogFileName);

  LogFile = fopen(LogFileName,"w");
  if (LogFile == NULL) /* cannot use abort() since it writes 'LogFile' */
  {
    fprintf(stderr,"fopen() of 'CacheTestLog' failed\n");
    exit(-2);
  }

  (void) SetupAndMark( LogFile, "LogFile");
  fprintf(LogFile,"\"CacheOnly Reference String, Integer Analsis\"\n\n");

  /* test for the presence of the environment variable for temp results */
  (void) TestEnvVar();  /* dies if environment variable is missing */

  /* Some initializations */
  UnitSize   = sizeof(void *);           /* tests use a C 'void *' */
  L1LineSize = GetL1LineSize();
  if (L1LineSize <= 0)  /* failure of L1 Tests */
    L1LineSize = 256; /* bytes assumed -- close enough for tests */
  L1LineSize = L1LineSize / UnitSize;
  PageSize   = GetOSPageSize() / UnitSize; 

  fprintf(stderr,"\nUnitSize is %d.\n",UnitSize);

  if (HeartBeat)
    fprintf(stderr,"Providing HeartBeat @ Level %d.\n",HeartBeat);

  /* Find an appropriate timer interval to allow 0.5% accuracy */
  TimerTick = FindTimerTick();
  MinTime = MIN_TICKS * TimerTick;

  fprintf(LogFile,"\"Timer tick is %s, Minimum time is %s\"\n",
	  PrintDNum(TimerTick), PrintDNum(MinTime));

  /* Initializations */
  BigInt = FindBigInt();
  i = (int) fmod(getticks(),1000000.0);
  fprintf(stderr,"Random seed is %s.\n",PrintNum(i));
  srand( i ); /* no real reason to believe in a better seed */

  AddCostInNsecs = GetAddCost();

  int FullRun = 1;
  if (FullRun)     
  {
    int ct, LB, UB, BS;

    LB = LOWERBOUNDINBYTES / UnitSize; 
    UB = UPPERBOUNDINBYTES / UnitSize;  /* upper bound is 32 mb */
    BS = PageSize / UnitSize;
    
    ct = GenerateLogSet( Sizes, MAX_TESTS_PER_RUN, LB, UB );

    (void) SetOfTrials( Sizes, Cycles, Times, ct );
  }


  int AssocRun = 0;
  if (AssocRun)
  {
    int ThisSize, LastSize, Result;
    char buf[32];

    LastSize = GetOSPageSize();
    for(i=1;1;i++)
    {
      sprintf(buf,"CL%02dS",i);
      ThisSize = ReadResultSoft(buf);
      ThisSize = ThisSize / UnitSize;
      if (ThisSize < 1)
	break;

      if (ThisSize > 0)
      {
	Result = AssocTrial( ThisSize, L1LineSize );
	fprintf(stderr,"Associativity @ %s is %d.\n",
		PrintNum(ThisSize*UnitSize),Result);
	fprintf(LogFile,"Associativity @ %s is %d.\n",
		PrintNum(ThisSize*UnitSize),Result);
	sprintf(buf,"CL%02dA",i);
	WriteResult(buf,Result);
      }
      LastSize = ThisSize;
    } 
  }

  int Test = 0;
  if (Test)
  {
    (void) SingleTrial(1024);
  }

  i = PACE_AllocationCheck();
  if (i != 0)
    fprintf(stderr,"Memory leak claimed %d aligned items.\n",i);
}
