/* PACE Project, Rice University
 *
 * AllocMem() returns a page-aligned chunk of the heap.
 * Objects allocated with AllocMem() must be freed with 
 * FreeMem().
 * 
 * A limited number of these aligned objects can be allocated.
 * Adjust NObjects below to increase that number.
 *
 * This code is not particularly elegant, but it was written 
 * for a limited-purpose application.  A more elegant and 
 * efficient solution can be written.
 * 
 * -- kdc 
 *
 */

#include "MemTest.h"

#define NObjects 25

static char errbuf[128];

static int PS = 0;
static void *Allocated[NObjects], *Reported[NObjects];

static int Allocations = 0;
static int Frees = 0;

void *PACE_AllocMem( int minsize /* measured in bytes */)
{
  int i, index;
  long int long_convert;
  void *result;

  if (PS == 0) /* initialize the necessary state */
  {
    PS = GetOSPageSize();
    for (i=0;i<NObjects;i++)
    {
      Allocated[i] = (void *) 0;
      Reported[i]  = (void *) 0;
    }
  }

  result = (void *) malloc(PS+minsize);
  if (result == NULL)
  {
    sprintf(errbuf,"malloc() fails to allocate object of %s bytes.\n",
	    PrintNum(minsize));
    Abort(errbuf,-1);
  }

  index = -1;
  for (i=0;i<NObjects;i++)
  {
    if (Allocated[i] == (void *) 0)
    {
      index = i;
      break;
    }
  }
  if (index == -1)
  {
    fprintf(stderr, "AllocMem() runs out of internal tracking slots.\n");
    fprintf(LogFile,"AllocMem() runs out of internal tracking slots.\n");
    exit(-1);
  }

  Allocated[index] = result;

  /* Now, bump the "result" pointer up to an integral page boundary */
  long_convert = (long) result;
  while ((long_convert % PS) != 0)
  {
    result++;
    long_convert = (long) result;
  }
  Reported[index] = result;

  if (Debug>1)
    fprintf(LogFile,"AllocMem( %s b): original %s, converted %s, slot %2d.\n",
	    PrintNum(minsize),
	    PrintNum((long) Allocated[index]), 
	    PrintNum((long) Reported[index]), index);

  Allocations++;
  return result;
}

void PACE_FreeMem( void * ptr )
{
  int i, index;

  for (i=0;i<NObjects;i++)
  {
    if (Reported[i] == ptr)
    {
      if (Debug>1)
	fprintf(LogFile,"FreeMem( %s ) frees %s in slot %d.\n",
		PrintNum((long) ptr), PrintNum((long)Allocated[i]), i);
      (void) free( Allocated[i] );
      Allocated[i] = (void *) 0;
      Reported[i]  = (void *) 0;
      Frees++;
      return;
    }
  }
  fprintf(LogFile,"FreeMem( %s ) fails.\n",PrintNum((long)ptr));
}

int PACE_AllocationCheck()
{
  fprintf(LogFile,
	  "\nAllocation report:\nPool Size %d\n  %s Allocations\n  %s Frees\n\n",
	  NObjects,PrintNum(Allocations),PrintNum(Frees));

  return Allocations - Frees;
}
