/*
 * PACE Project, Rice University
 *
 * Preconditioning filter for cache benchmark data
 *
 */
#include "pace_rc_memlib.h"

double round( double d );

/* The idea behind this preconditioner is quite simple.
 * If we have three experimental values, a b & c, and
 * |a-c| < Epsilon, then we expect b to fall near the 
 * linear interpolant of a & c.  (This observation is 
 * based on the physical behavior of the memory system.
 * The data should increase monotonically, unless we see
 * a point where the set of spots being accessed map 
 * completely into a lower level of cache.)
 *
 */

void  PreCondition( int Sizes[], double Times[] , int Count, double Epsilon )
{
  int i, a, b, c, FirstTime;
  double interpolant;

  FirstTime = 1;
  
  /* First, smooth the line by pulling up values that are below
   * the linear interpolant, but only if the endpoints (a & c) 
   * are close ( < Epsilon )
   */

  for (i=2; i<Count; i++)
  {
    a = i-2;
    b = i-1;
    c = i;
         
    if ((DAbs(Times[a] - Times[c])/Times[c]) < Epsilon)
    { /* region is flat, within Epsilon */
      interpolant = round(Times[a] + Times[c]) / 2.0;
      if (Times[b] < interpolant)
      {
	if (FirstTime)
	{ 
	  fprintf(stderr,"\nPreconditioning the data.\n");
	  FirstTime = 0;
	}
	Times[b] = interpolant;
      }
    }
  }

  /* finally, pull high values down */
  for (i=2; i<Count; i++)
  {
    a = i-2;
    b = i-1;
    c = i;
         
    if ((DAbs(Times[a] - Times[c])/Times[c]) < Epsilon)
    { /* region is flat, within Epsilon */
      interpolant = round(Times[a] + Times[c]) / 2.0;
      if (Times[b] > interpolant)
      {
	if (FirstTime)
	{ 
	  fprintf(stderr,"\nPreconditioning the data.\n");
	  FirstTime = 0;
	}
	Times[b] = interpolant;
      }
    }
  }
  if (FirstTime)
    fprintf(stderr,"Preconditioner made no changes.\n");
}

