/* PACE Project, Rice University
 *
 * Support code for the PACE RC memory tests
 * 
 *
 */

#include "pace_rc_memlib.h"

void Shuffle( int Array[], int Size )
{
  int i, j, k, t;

  for (i=0; i<2; i++)
  {
    for (j=0; j<Size; j++)
    {
      k        = rand() % Size;
      t        = Array[j];
      Array[j] = Array[k];
      Array[k] = t;
    }
  }
}

int *GenerateLinearSet( int Array[], int Size, int Stride )
{
  int i, j;

  j = 0;
  for (i=0; i<Size; i++)
  {
    Array[i] = j;
    j += Stride;
  }
  return Array;
}

int *GenerateSetForAssoc( int Array[], int Size, int Stride, int Ways )
{
  int i, j, k, limit;

  limit = Size / Ways;

  j = 0;
  k = 0;
  for (i=0; i<Size; i++)
  {
    Array[i] = j + (k * Size);
    j += Stride;
    if (j >= limit)
    {
      j = 0;
      k++;
    }
  }
  return Array;
}

int GenerateLogSet( int Array[], int Size,  int LB, int UB )
{
  int i, j, k, increment;

  if (LB < 4)
  {
    fprintf(LogFile,"GenerateLogSet: LB (%d) is too small.  Run aborted",LB);
    fprintf(stderr,"GenerateLogSet: LB (%d) is too small.  Run aborted",LB);
    exit(-1);
  }

  if (power(2,log2(LB)) != LB)
  {
    fprintf(LogFile,"GenerateLogSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(LogFile,
	    "\tThis indicates a structural problem and may cause problems.\n");
    fprintf(stderr,"GenerateLogSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(stderr,
	    "\tThis indicates a structural problem and may cause problems.\n");
  }

  if (power(2,log2(UB)) != UB)
  {
    fprintf(LogFile,"GenerateLogSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(LogFile,
	    "\tSet may run past its upper bound.\n");
    fprintf(stderr,"GenerateLogSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(stderr,
	    "\tSet may run past its upper bound.\n");
  }

  increment = LB / 4;

  i = LB;  /* i will keep the running size */
  j = 0;   /* j is the index into array    */
  k = 0;   /* k counts 0 to 3 to govern increments to "increment" */

  while ( i <= UB )
  {
    if (j < Size)
      Array[j] = i;
    i = i + increment;
    j++;
    k++;
    if (k == 4)
    {
      increment = increment + increment;
      k = 0;
    }
  }
    if (j > Size)
    {
      fprintf(stderr,
	      "GenerateLogSet: Array Size (%s) is too small (%s to %s).\n",
	      PrintNum(Size),PrintNum(LB),PrintNum(UB));
      fprintf(stderr,
	      "Increase array to at least %s for these bounds.\n",
	      PrintNum(j));
    }
    return j;
}

int GenerateHalfStepSet( int Array[], int Size,  int LB, int UB )
{
  int i, j, k, increment;

  if (LB < 4)
  {
    fprintf(LogFile,
	    "GenerateHalfStepSet: LB (%d) is too small.  Run aborted",LB);
    fprintf(stderr,
	    "GenerateHalfStepSet: LB (%d) is too small.  Run aborted",LB);
    exit(-1);
  }

  if (power(2,log2(LB)) != LB)
  {
    fprintf(LogFile,
	    "GenerateHalfStepSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(LogFile,
	    "\tThis indicates a structural problem and may cause problems.\n");
    fprintf(stderr,
	    "GenerateHalfStepSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(stderr,
	    "\tThis indicates a structural problem and may cause problems.\n");
  }

  if (power(2,log2(UB)) != UB)
  {
    fprintf(LogFile,
	    "GenerateHalfStepSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(LogFile,
	    "\tSet may run past its upper bound.\n");
    fprintf(stderr,"GenerateLogSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(stderr,
	    "\tSet may run past its upper bound.\n");
  }

  increment = LB / 2;

  i = 0;    /* index in array */
  j = LB;   /* initial set element */
  while (j <= UB)
  {
    if (i < Size)
      Array[i++] = j;

    if (i < Size && (j + increment) <= UB)
      Array[i++] = j + increment;


    if (i>Size)
    {
      fprintf(stderr,"GenerateHalfStepSet: exceeds set size.\n");
      (void) exit(-1);
    }
    increment = j;
    j = j + j;
  }

  return i;
}

int GenerateLogBlockSet( int Array[], int Size,  int LB, int UB, int BlockSize )
{
  int i, j, k, increment;

  /* Four is a magic number in this routine.
   * The routine generates a set that includes all powers of two from LB to UB,
   * called the primary points.  Between each pair of primary points, it includes
   * three uniformly spaced intermediate points.  Thus, for a pair of primary
   * points 2^i and 2^(i+1), it will include 2^i * 1.25, 2^i * 1.5, and 2^i * 1.75.
   *
   * If the test uses blocked data (as happens in many of the memory tests),
   * it must be the case that each intermediate point is also an integral 
   * multiple of the blocksize.  Thus, this version of the generator drops out
   * points where that condition does not hold.
   * 
   * Arithmetically, it can only happen below 4 * BlockSize.
   *
   */

  if (LB < 4)
  {
    fprintf(LogFile,"GenerateLogBlockSet: LB (%d) is too small.  Run aborted",LB);
    fprintf(stderr,"GenerateLogBlockSet: LB (%d) is too small.  Run aborted",LB);
    exit(-1);
  }

  if (power(2,log2(LB)) != LB)
  {
    fprintf(LogFile,"GenerateLogBlockSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(LogFile,
	    "\tThis indicates a structural problem and may cause problems.\n");
    fprintf(stderr,"GenerateLogBlockSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(stderr,
	    "\tThis indicates a structural problem and may cause problems.\n");
  }

  if (power(2,log2(UB)) != UB)
  {
    fprintf(LogFile,"GenerateLogBlockSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(LogFile,
	    "\tSet may run past its upper bound.\n");
    fprintf(stderr,"GenerateLogBlockSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(stderr,
	    "\tSet may run past its upper bound.\n");
  }

  increment = LB / 4;

  i = LB;  /* i will keep the running size */
  j = 0;   /* j is the index into array    */
  k = 0;   /* k counts 0 to 3 to govern increments to "increment" */

  while ( i <= UB )
  {
    if ((BlockSize * (i/BlockSize) == i)  /* test relation between i & BlockSize */
	&& j < Size)                      /* test for overflow on "Array" */
      Array[j++] = i;

    i = i + increment;
    k++;
    if (k == 4)
    {
      increment = increment + increment;
      k = 0;
    }
  }
    if (j > Size)
    {
      fprintf(stderr,
	      "GenerateLogBlockSet: Array Size (%s) is too small (%s to %s).\n",
	      PrintNum(Size),PrintNum(LB),PrintNum(UB));
      fprintf(stderr,
	      "Increase array to at least %s for these bounds.\n",
	      PrintNum(j));
    }
    return j;
}


int GenerateEighthStepSet( int Array[], int Size,  int LB, int UB )
{
  int i, j, k, increment;

  if (LB < 8)
  {
    fprintf(LogFile,
	    "GenerateEighthStepSet: LB (%d) is too small.  Run aborted",LB);
    fprintf(stderr,
	    "GenerateEighthStepSet: LB (%d) is too small.  Run aborted",LB);
    exit(-1);
  }

  if (power(2,log2(LB)) != LB)
  {
    fprintf(LogFile,
	    "GenerateEighthStepSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(LogFile,
	    "\tThis indicates a structural problem and may cause problems.\n");
    fprintf(stderr,
	    "GenerateEighthStepSet: Lower bound (%s) is not a power of 2.\n",
	    PrintNum(LB));
    fprintf(stderr,
	    "\tThis indicates a structural problem and may cause problems.\n");
  }

  if (power(2,log2(UB)) != UB)
  {
    fprintf(LogFile,
	    "GenerateEighthStepSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(LogFile,
	    "\tSet may run past its upper bound.\n");
    fprintf(stderr,"GenerateEighthSet: Upper bound (%s) is not a power of 2.\n",
	    PrintNum(UB));
    fprintf(stderr,
	    "\tSet may run past its upper bound.\n");
  }

  i = 0;    /* index in array */
  j = LB;   /* initial set element */
  while (j < UB)
  {
    increment = j / 8;
    for (k=0; k < 8; k++)
    {
      if (i < Size)
	Array[i++] = j + k * increment;
    }

    if (i>Size)
    {
      fprintf(stderr,"GenerateEighthStepSet: exceeds set size.\n");
      (void) exit(-1);
    }
    j = j + j;
  }
  if (i<Size)
    Array[i++] = j;

  return i;
}
