/* PACE Project, Rice University
 *
 * This code measures the parameters of the cache memory system.
 * 
 */

#include "MemTest.h"
#include "BlackJackTimer.h"

void exit( int x );
double sqrt(double x);
int errno;

char *TimeStamp();

char HostName[128];

void PrintArray( char *s, int array[], int size )
{
  int i;

  fprintf(stderr,"\n\nContents of %s: (%d)\n",s,size);
  for (i=0;i<size;i++)
    {
      if ((i % 4) == 0)
	fprintf(stderr,"\n%10d:  ",i);
      fprintf(stderr,"%10d, ",array[i]);
    }
  fprintf(stderr,"\n");
}

static void DieWithMessage(char *currArg, int argNumber)
{
  fprintf(stderr,"Did not understand argument '%s' (%d).\n",
	  currArg,argNumber);
  fprintf(stderr,"Valid arguments are:\n");
  fprintf(stderr,"\t-g \tActivate internal debugging checks.\n");
  fprintf(stderr,"\t-l <name> \tUse <name> for log file.\n");
  fprintf(stderr,"\t-p \tTurn off preconditioning.\n");
  fprintf(stderr,"\t-v \tPrint results (& debug info) incrementally.\n");
  exit(-1);  
}

int main(int argc, char *argv[])
{
  int i, j, arg, argNumber;
  char *currArg, *LogFileName;

  Verbose   = 0;
  Debug     = 0;
  HeartBeat = 1;
  PreCon    = 1;

  LogFileName = "./Log";

  /* Parse the command line parameters */
  for (argNumber = 1; argNumber < argc; argNumber++)
  {
    currArg = argv[argNumber];

    if (*currArg == '-')
    {
      currArg++; 
      while (*currArg != '\0')
      {
	switch(*currArg)
        {
 	  case 'd':
	    Debug++;
	    break;
	  case 'g':
	    Debug++;
	    break;
  	  case 'h':
	    HeartBeat++;
	    break;
	  case 'l':
	    argNumber++;
	    LogFileName = argv[argNumber];
	    break;
	  case 'p':
	    PreCon = 0;
	    break;
	  case 'v':
            Verbose++;
	    break;
	  default:
	    DieWithMessage(currArg,argNumber);
	}
	currArg++;
      }
    }
  }

  /* Set up the Log File */
  fprintf(stderr,"Log File Name is '%s'.\n",LogFileName);

  LogFile = fopen(LogFileName,"w");
  if (LogFile == NULL) /* cannot use abort() since it writes 'LogFile' */
  {
    fprintf(stderr,"fopen() of '%s' failed\n",LogFileName);
    exit(-2);
  }

  (void) SetupAndMark( LogFile, "LogFile" );
  fprintf(LogFile,"\"TLB Only Test with Integer Analysis\"\n");

  if (HeartBeat)
    fprintf(stderr,"Providing HeartBeat @ Level %d.\n",HeartBeat);

  /* Set the process' CPU affinity mask */
  /* i = TieProcess(); */
  /* fprintf(LogFile,"\"Process tied to CPU %d.\"\n",i); */

  /* Test to ensure that the environment variable for the temporary */
  /* directory is set. Aborts if it fails. */
  (void) TestEnvVar();

  /* Some initializations */
  UnitSize = sizeof(void**);          /* tests use a C 'void **' */
  L1Size = ReadResultSoft("CacheL1Sa") / UnitSize; /* assumes L1 test has run */
  L1LineSize = GetL1LineSize() / UnitSize;
  PageSize = GetOSPageSize() / UnitSize;
  AddCost = GetAddCost();

  fprintf(LogFile,"Using Unitsize %d, L1Size %s b, L1 Linesize %d b, ",
	  UnitSize,PrintNum(L1Size*UnitSize),L1LineSize*UnitSize);
  fprintf(LogFile,"PageSize %s b, AddCost %f\n",
	  PrintNum(PageSize*UnitSize),AddCost);

  /* Find an appropriate timer interval to allow 0.5% accuracy */
  TimerTick = FindTimerTick();
  MinTime = MIN_TICKS * TimerTick;

  fprintf(LogFile,"\"Timer tick is %s, Minimum time is %s\"\n",
	  PrintDNum(TimerTick), PrintDNum(MinTime));

  /* Initializations */
  BigInt = FindBigInt();
  i = (int) fmod(getticks(),1000000.0);
  fprintf(stderr,"Random seed is %s.\n",PrintNum(i));
  srand( i ); /* no real reason to believe in a better seed */

  MemArray = (void **) NULL;

  int TLBOnly = 1;
  if (TLBOnly)
  {
    fprintf(stderr,"Page size is %s b.\n",PrintNum(PageSize*UnitSize));
    fprintf(LogFile,"\"Page size is %s b, from 'sysconf()'.\"\n",
	    PrintNum(PageSize*UnitSize));

    /* Run the TLB test with 1, 2, 3, and 4 Spots */
    TLBTrial( 1 /* TLB 1-spot */ );
    TLBTrial( 2 /* TLB 2-spot */ );
    TLBTrial( 3 /* TLB 3-spot */ );
    TLBTrial( 4 /* TLB 4-spot */ );

    MiniReconcile();
  }
  int Test = 0;
  if (Test)
  {
    BuildTNK(1024*1024/UnitSize,16*1024/UnitSize,64/UnitSize,1);
    fprintf(stderr,"\n--------\n");
    BuildTNK(1024*1024/UnitSize,16*1024/UnitSize,64/UnitSize,2);
    fprintf(stderr,"\n--------\n");
    BuildTNK(1024*1024/UnitSize,16*1024/UnitSize,64/UnitSize,3);
    fprintf(stderr,"\n--------\n");
    BuildTNK(1024*1024/UnitSize,16*1024/UnitSize,64/UnitSize,4);
    fprintf(stderr,"\n--------\n");
  }

  i = PACE_AllocationCheck();
  if (i != 0)
    fprintf(stderr,"Memory leak claimed %d aligned items.\n",i);

}
  

