#include "MemTest.h"
#include "BlackJackTimer.h"

int GlobalZero;

static int *ColumnIndexSet   = NULL;
static int ColumnIndexSize   = 0;
static int *RowIndexSet      = NULL;
static int RowIndexSize      = 0;

static int NotYetComplained = 0;
static int RandomStaticFlag = 0;

static struct AccessCount NAccesses; 

static char buf[256];

static int  ConfirmTLB( int s );

int  IsItATLB( int s );


/**  Tests for case 2 -- the suspect point is out beyond the largest TLB
 **  and we are simply seeing a cache response from some larger cache ...
 **
 **  Here, we build a TLB Only Permutation, except that we thread it 
 **  through four lines in each page. When it runs, it should have 1/4
 **  as many TLB misses as the original. If the time is due to TLB, it
 **  should change (by about 4x).  If it is due to cache, it should not
 **  change.  (Decrease in time => suspect is a TLB)
 **/

static Counter = 0;

void BuildTNK( int ArraySize,
	       int PageSize, /* TLB Page Size */
	       int LineSize,
	       int NSpots )
{
  int i, j, k, dist, ThisElt, LastElt, count;
  int  NColumns, NPoints, Separation;
  void **Old, **p;

  
  if (LineSize >= PageSize)
  {
    fprintf(stderr,"BuildOddTLBPerm() receives invalid parameters, see log\n");
    fprintf(LogFile,"\nBuildOddTLBPerm: invalid parameters.\n");
    fprintf(LogFile,"Line size %04d, page size %04d\n",LineSize,PageSize);
    fprintf(LogFile,"Test makes no sense in this context.\n");
    (void) exit(-1);
  }

  NColumns = PageSize / (LineSize * NSpots);
  Separation = PageSize / NSpots;
  NPoints = ArraySize * NSpots / PageSize;

  ColumnIndexSet = PACE_AllocMem( NColumns * sizeof(int) );
  RowIndexSet    = PACE_AllocMem( NPoints * sizeof(int) );

  /* Generate the index sets we need & shuffle them */
  (void) GenerateLinearSet( ColumnIndexSet, NColumns, LineSize );
  (void) Shuffle( ColumnIndexSet, NColumns );

  (void) GenerateLinearSet( RowIndexSet, NPoints, Separation );
  (void) Shuffle( RowIndexSet, NPoints );

  /* Allocate the Memory Array */
  Old = MemArray;
  MemArray = (void **) PACE_AllocMem(ArraySize * UnitSize);
  if (Old != (void **) NULL)
    PACE_FreeMem(Old); 

  if (Debug>1)
    fprintf(LogFile,"\nBuilding New Permutation %d\n",Counter);

  /* Assemble the permutation */
  LastElt = -1;
  j = 0;
  for (i=0;i<NPoints;i++)
  {
    ThisElt = RowIndexSet[i] + ColumnIndexSet[j];
    if (LastElt != -1)
      MemArray[ThisElt] = &MemArray[LastElt];

    LastElt = ThisElt;

    j++;
    if (j >= NColumns)
      j = 0;
  }
  /* and, set last point */
  StartingPoint = RowIndexSet[0] + ColumnIndexSet[0];
  MemArray[StartingPoint] = &MemArray[LastElt];

  /* Check the permutation */
  if (Debug>1)
    fprintf(LogFile,"\nM[%s] <- %s. ** starting point **\n",
	    PrintNum(StartingPoint),PrintNum(LastElt));

  /* verify permutation */
  p = MemArray[StartingPoint];
  i = 0;
  count = NPoints;
  while (p != &MemArray[StartingPoint])
  {
    p = *p;
    i++;
    if (i > count)
    {
      fprintf(stderr,"%d: Cycle did not return to starting point.\n",Counter);
      fprintf(stderr,"Cycle length is %s of %s.\n",
	      PrintNum(i),PrintNum(count));
      break;
    }
  }
  if ((Debug) && (i == count-1))
    fprintf(LogFile,"\"Maximal length permutation.\"\n");
  if (i < count-1)
    fprintf(LogFile,"\n\"Short permutation (%d vs %d).\"\n",i,count-1);

  PACE_FreeMem(ColumnIndexSet);
  ColumnIndexSet = (int *) NULL;

  PACE_FreeMem(RowIndexSet);
  RowIndexSet = (int *) NULL;
}


/* TLBTest 
 *
 * Conducts a specialized test to look for a TLB
 *
 * PARAMETERS:  ArraySize, PageSize, LineSize, NA
 *
 * RETURNS:     An elapsed time, in microseconds, as a double
 *
 */

double TLBTest( int ArraySize, /* footprint for test, in WORDS            */
	        int PageSize,  
	        int LineSize,  
		struct AccessCount NA,  /* number of iterations     */
		int NRefs
	       )
{
  int i;
  double result;
  
  if (HeartBeat > 1)
    fprintf(stderr,"Trial @ %s b: ",PrintNum(ArraySize*UnitSize));

  /* Initialize MemArray */
  BuildTNK(ArraySize,PageSize,LineSize,NRefs); 
  if (HeartBeat > 1)
    fprintf(stderr," -- built tnk");
   /*   Flush(ArraySize); */

  /* Run the test */
   result = TimePermPtr(MemArray, StartingPoint, NA);
   if (HeartBeat > 1)
     fprintf(stderr," -- ran tnk");

   PACE_FreeMem(MemArray);
   MemArray = (void **) NULL;

  if (HeartBeat > 1)
    fprintf(stderr,"%s usec\n",PrintDNum(result));

  return result;
}

/* TLBTrial
 *
 * PARAMETERS: PageSize, UB, LineSize
 *
 * Runs TLBTest() on from PageSize to UB, recording its results in 
 * the LogFile.
 *
 */

void TLBTrial ( int NRefs )
{
  int i, j, k, Count;
  double Trial, Events;
  
  int Small, Large;

  int    AllSizes[MAX_TESTS_PER_RUN], ExtractSizes[MAX_TESTS_PER_RUN];
  int    Counters[MAX_TESTS_PER_RUN], NotDone;
  int    Cycles  [MAX_TESTS_PER_RUN];
  double AllTimes[MAX_TESTS_PER_RUN], ExtractTimes[MAX_TESTS_PER_RUN];
  struct IntList *p, *q;

  NotYetComplained = 1;
  GlobalZero = 0;

  /* The TLB test & permutation only makes sense at integral multiples of
   * the PageSize; rather than write yet another generator over in Lib,
   * we generate the full set of eighth step sizes into ExtractSizes
   * and then eliminate sizes that do not make sense.
   */

  Small = 2 * PageSize;
  Large = Min(2048*PageSize,1024*1024*1024 / UnitSize);
  
  Count = GenerateEighthStepSet(ExtractSizes,MAX_TESTS_PER_RUN, Small, Large);

  j = 0;
  for (i=0; i< Count; i++)
  {
    if (PageSize * (ExtractSizes[i]/PageSize) == ExtractSizes[i])
    {
      AllSizes[j] = ExtractSizes[i];
      Counters[j] = TRIALS_FOR_MIN; /* set the counter for this entry */
      j++;
    }
  }
  Count = j;

  NAccesses.outer = 1;
  NAccesses.inner = 1000000;

  fprintf(LogFile,"\nTLB %d Spot Test of %s points between %s b to %s b.\n",
	  NRefs, PrintNum(Count), PrintNum(AllSizes[0]*UnitSize),
	  PrintNum(AllSizes[Count-1]*UnitSize));
  fprintf(LogFile,"Page Size %s b, Line Size of %s b.\n",
	  PrintNum(PageSize*UnitSize),PrintNum(L1LineSize*UnitSize));
  fprintf(LogFile,"( %s ; %s ) accesses.\n",
	  PrintNum(NAccesses.outer),PrintNum(NAccesses.inner));

  /* run the trial until every point has a "good" time */
  NotDone = 1;
  i = 1;
  k = 0;
  while(NotDone)
  {
    if (HeartBeat)
      fprintf(stderr,"Starting TLB %d-Spot Series %2d.",NRefs,i);

    NotDone = 0;
    for (j=0; j<Count;j++)
    { 
      if (Counters[j])
      {
	Trial = TLBTest(AllSizes[j],PageSize,L1LineSize,NAccesses,NRefs);
	k++;
	if (i == 1)
	  AllTimes[j] = Trial;
	else if (Trial < AllTimes[j])
	{
	  AllTimes[j] = Trial;
	  Counters[j] = TRIALS_FOR_MIN;
	}
	else
	{
	  Counters[j]--;
	}
	if (Counters[j] > 0)
	  NotDone = 1;
      }
    }
    if (HeartBeat)
      fprintf(stderr," Tested %d points.\n",k);
    k = 0; i++;
  }
  fprintf(stderr,"\n");

  /* Set up for the integer analysis package */
  if (NAccesses.outer == 1)
    Events = (double) NAccesses.inner;
  else
    Events = (((double) NAccesses.outer) - 1.0) * (double) BigInt
      + (double) NAccesses.inner;

  for (i=0;i<Count;i++)
  {
    Cycles[i] = round(1000.0 * AllTimes[i] / (Events * AddCost));
  }

  /* Write the data to the Log File */
  fprintf(LogFile,"\nSize\tTime\tCycles\n");
  for (i=0; i<Count; i++)
    fprintf(LogFile,"%s\t%s\t%s\n",
	    PrintNum(AllSizes[i]*UnitSize),PrintDNum(AllTimes[i]),
	    PrintNum(Cycles[i]));

  AnalyzeTLB( AllSizes, Cycles, Count, "TLBI", /* no smoothing */ 1,
	      NRefs );

  return;
}

