/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
/*
 *  affinity.c
 *  
 *  Implements affinity.h interface
 *  Either HWLOC_AFFINITY, PTHREAD_AFFINITY, or NO_AFFINITY must be defined.
 *  Optionally define  __DEBUG__ macro 
 *
 *
 *  Created by Anna Youssefi on 4/27/10.
 *  Copyright 2010 __MyCompanyName__. All rights reserved.
 *
 */

// #define __DEBUG__


#define _GNU_SOURCE
#include <pthread.h>

#include <sched.h>
#include <stdlib.h>
#include <stdio.h>





///////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef HWLOC_AFFINITY

#include <hwloc.h>

hwloc_topology_t topology;


/* Initializes and sets cpucount to total number of cpus.
*  Returns success (0) or error code */
int init_affinity(int *cpucount){

	int i, res;
	const hwloc_cpuset_t set = hwloc_cpuset_alloc();
	
	i = 0;
	
	//initialize
	if (hwloc_topology_init(&topology)) {
		#ifdef __DEBUG__
			fprintf(stderr, "ERROR: hwloc_topology_init failed!\n");
		#endif
		return 1; 
	}
	if (hwloc_topology_load(topology)){
		#ifdef __DEBUG__
			fprintf(stderr, "ERROR: hwloc_topology_load failed!\n");
		#endif
		return 1; 
	}


	// find cpu count
	for ( i =0; i < HWLOC_NBMAXCPUS; i++){
		hwloc_cpuset_set(set, i);
		res = hwloc_set_cpubind(topology, set, HWLOC_CPUBIND_THREAD | HWLOC_CPUBIND_STRICT );
		if (res) break;
	}
	
	*cpucount = i;
	#ifdef __DEBUG__
		fprintf(stderr, "Cpu count %d\n", *cpucount);
	#endif 
	
	if (i==0){
		#ifdef __DEBUG__
			fprintf(stderr, "CPU count = 0, set affinity returned code %d: ", res);
			if (res == ENOSYS) fprintf(stderr, "ENOSYS (not possible to bind requested kind of proc/thread).\n");
			else if (res == EXDEV) fprintf(stderr, "EXDEV (requested cpu set can't be enforced).\n");
			else fprintf(stderr, "Unknown.\n");
		#endif
		return res;
	}
	else if (i ==  HWLOC_NBMAXCPUS) {
		#ifdef __DEBUG__
			fprintf(stderr, "Warning, cpucount = HWLOC_NBMAXCPUS!\n");  //this should never happen but just in case!
		#endif
	}
	
	//////////////////////////////////////////////////////////////////
	// TODO:  SEE IF THERE'S A WAY TO GET A DEFAULT CPUSET TO RESTORE!
	// restore original affinity for thread
	hwloc_cpuset_set_range (set, 0, i-1);
	res = hwloc_set_cpubind(topology, set, HWLOC_CPUBIND_THREAD);

	return 0;

}

/* Binds calling thread to cpu cpunum.
*  Returns success (0) or error code */
int set_affinity(int cpunum){
	int res;
	const hwloc_cpuset_t set = hwloc_cpuset_alloc();
	hwloc_cpuset_set(set, cpunum);
	res = hwloc_set_cpubind(topology, set, HWLOC_CPUBIND_THREAD | HWLOC_CPUBIND_STRICT );
	
	if (res) {  //binding failed
		#ifdef __DEBUG__
			fprintf(stderr, "ERROR:  set thread affinity failed for cpu %d, returned code %d: ", cpunum, res);
			if (res == ENOSYS) fprintf(stderr, "ENOSYS (not possible to bind requested kind of proc/thread).\n");
			else if (res == EXDEV) fprintf(stderr, "EXDEV (requested cpu set can't be enforced).\n");
			else fprintf(stderr, "Unknown.\n");
		#endif
		return res;
	}
	else return 0;

}

#endif //end HWLOC_AFFINITY

///////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef PTHREAD_AFFINITY

#include <limits.h>

#define MAX_CPUS INT_MAX


cpu_set_t mask;
cpu_set_t myaffinity;


/* Sets cpucount to total number of cpus.
*  Returns success (0) or error code */
int init_affinity(int *cpucount){

	int i, res;

	//save thread's affinity for restoring at end of function
	CPU_ZERO(&myaffinity);
	res = pthread_getaffinity_np(pthread_self(), sizeof(cpu_set_t), &myaffinity);
	
	for ( i =0; i < MAX_CPUS; i++){
		CPU_ZERO(&mask);
		CPU_SET(i, &mask);
		res = pthread_setaffinity_np(pthread_self(), sizeof(cpu_set_t), &mask);
		if (res) break;
	}
	
	*cpucount = i;
	#ifdef __DEBUG__
		fprintf(stderr, "MAX_CPUS %d\n", MAX_CPUS);
		fprintf(stderr, "Cpu count %d\n", *cpucount);
	#endif
	
	if (i == 0){
		#ifdef __DEBUG__
			fprintf(stderr, "CPU count = 0, set affinity returned code %d: ", res);
		#endif
		return res;
	}
	
	else if (i ==  MAX_CPUS){
		#ifdef __DEBUG__
			fprintf(stderr, "Warning, cpucount = %d, MAX_CPUS!\n", MAX_CPUS);
		#endif
	}
	
	//clear binding
	res = pthread_setaffinity_np(pthread_self(), sizeof(cpu_set_t), &myaffinity);
	CPU_ZERO(&myaffinity);
	
	return 0;
}


/* Binds calling thread to cpu cpunum.
*  Returns success (0) or error code */
int set_affinity(int cpunum){
	int aff;
	
	CPU_ZERO(&mask);
	CPU_SET(cpunum, &mask);

	aff = pthread_setaffinity_np(pthread_self(), sizeof(cpu_set_t), &mask);
	return aff;

}

#endif //end PTHREAD_AFFINITY

///////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef NO_AFFINITY

/* 
 *  Always returns 1 (error).
 *  Cpucount set to 0.
 */
int init_affinity(int *cpucount){
	*cpucount = 0;
	return 1;
}

/* 
*  Always returns success (0).
*/
int set_affinity(int cpunum){
	return 0;
}

#endif
