/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
/* PACE Project, Rice University
 *
 * This code measures the parameters of the cache memory system.
 * 
 *
 *
 */

#include "MemTest.h"
#include "BlackJackTimer.h"

void exit( int x );
double sqrt(double x);

int errno;

int PreCon = 0;

void shuffle( int array[], int size )
{
  int i, j, t;

  for (i=0; i<size; i++)
    array[i] = i;

  for (i=0;i<size;i++)
    {
      j        = rand() % size;
      t        = array[i];
      array[i] = array[j];
      array[j] = t;
    }
}

void PrintArray( char *s, int array[], int size )
{
  int i;

  fprintf(stderr,"\n\nContents of %s: (%d)\n",s,size);
  for (i=0;i<size;i++)
    {
      if ((i % 4) == 0)
	fprintf(stderr,"\n%10d:  ",i);
      fprintf(stderr,"%10d, ",array[i]);
    }
  fprintf(stderr,"\n");
}

static void DieWithMessage(char *currArg, int argNumber)
{
  fprintf(stderr,"Did not understand argument '%s' (%d).\n",
	  currArg,argNumber);
  fprintf(stderr,"Valid arguments are:\n");
  fprintf(stderr,"\t-g \tActivate internal debugging checks.\n");
  fprintf(stderr,"\t-h \tActivate HeartBeat to stderr.\n");
  fprintf(stderr,"\t-l <name> \tUse <name> for log file.\n");
  fprintf(stderr,"\t-p \tPrecondition the data before analysis.\n");
  fprintf(stderr,"\t-v \tPrint results (& debug info) incrementally.\n");
  exit(-1);  
}

int main(int argc, char *argv[])
{
  int i, j, arg, argNumber;
  char *currArg, *LogFileName;

  int    Sizes [MAX_TESTS_PER_RUN];
  int    Cycles[MAX_TESTS_PER_RUN];
  double Times [MAX_TESTS_PER_RUN];
  int    Count;

  Verbose   = 0;
  Debug     = 0;
  HeartBeat = 1;

  LogFileName = "./Log";

  /* Parse the command line parameters */
  for (argNumber = 1; argNumber < argc; argNumber++)
  {
    currArg = argv[argNumber];

    if (*currArg == '-')
    {
      currArg++; 
      while (*currArg != '\0')
      {
	switch(*currArg)
        {
	  case 'g':
	    Debug++;
	    break;
  	  case 'h':
	    HeartBeat++;
	    break;
	  case 'l':
	    argNumber++;
	    LogFileName = argv[argNumber];
	    break;
  	  case 'p':
	    PreCon++;
	    break;
	  case 'v':
            Verbose++;
	    break;
	  default:
	    DieWithMessage(currArg,argNumber);
	}
	currArg++;
      }
    }
  }

  /* Set up the Log File */
  fprintf(stderr,"Log File Name is '%s'.\n",LogFileName);

  LogFile = fopen(LogFileName,"w");
  if (LogFile == NULL) /* cannot use abort() since it writes 'LogFile' */
  {
    fprintf(stderr,"fopen() of '%s' failed.\n",LogFileName);
    exit(-2);
  }

  (void) SetupAndMark( LogFile, "LogFile" );
  fprintf(LogFile,"\"BC Microbenchmark with Integer Analysis\"\n");

  if (HeartBeat)
    fprintf(stderr,"Providing HeartBeat @ Level %d.\n",HeartBeat);

  /* Test to ensure that the environment variable for the temporary */
  /* directory was set; aborts if environment variable is missing   */
  (void) TestEnvVar();  

  /* Some initializations */
  UnitSize = sizeof(void **);          /* tests use a C 'void **' */
  PageSize = GetOSPageSize() / UnitSize;
  AddCost = GetAddCost();

  fprintf(LogFile,"Using UnitSize %d, PageSize %s, & AddCost %f\n",
	  UnitSize,PrintNum(PageSize*UnitSize),AddCost);

  /* Find an appropriate timer interval to allow 0.5% accuracy */
  TimerTick = FindTimerTick();
  MinTime = MIN_TICKS * TimerTick;

  fprintf(LogFile,"\"Timer tick is %s, Minimum time is %s\"\n",
	  PrintDNum(TimerTick), PrintDNum(MinTime));

  /* Initializations */
  BigInt = FindBigInt();
  i = (int) fmod(getticks(),1000000.0);
  fprintf(stderr,"Random seed is %s.\n",PrintNum(i));
  srand( i ); /* no real reason to believe in a better seed */

  int StrideStudy = 0;
  if (StrideStudy)
  {
    int stride, blocksize;

    arg = GenerateHalfStepSet(Sizes, MAX_TESTS_PER_RUN, 8192, 8 * 1024 * 1024 );

    stride    = 128;    /* words */
    blocksize = 1024; /* words */
    while(blocksize <= 8192)
    {
	fprintf(LogFile,"\n\"Blocksize: %s,\tStride: %s\"\n",
		PrintNum(blocksize),PrintNum(stride));
	fprintf(stderr, "\n\"Blocksize: %s,\tStride: %s\"\n",
		PrintNum(blocksize),PrintNum(stride));

	BCTrial( Sizes, Times, arg, blocksize, stride );
	blocksize = blocksize + blocksize;
    }
  }

  int FullRun = 1;
  if (FullRun)
  {
    int LB, UB;
    LB = LOWERBOUNDINBYTES / UnitSize;
    UB = UPPERBOUNDINBYTES / UnitSize;

    arg = GenerateLogSet(Sizes, MAX_TESTS_PER_RUN, LB, UB);
    BCTrial(Sizes, Times, Cycles, arg, PageSize/2, 64);
  }

  i = PACE_AllocationCheck();
  if (i != 0)
    fprintf(stderr,"Memory leak has claimed %d aligned objects.\n",i);
}
  
