/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
#include "MemTest.h"

#define SL  4
#define RA  2
#define YES 1
#define NO  0

static int  ExitEarly = 0;
static int  CorDFlattensAt;

static int L1Size = 0;
static int L1Assoc = 0;
static int L1Gran  = 0;
static int PageSize = 0;

static char buffer[32];  /* POSIX file name limit is 14 */

void AnalyzeL1( int s[], double t[], int c, double r);
int  CorDTest( int s, int min, int max );
int  LogTest (int s[], double t[], int c, double r); /* returns index */

static void FindLatency( int s[], double t[], int c, int l1 );

/* When we interpret these graphs, we always use a log scale on the Sizes axis.
 * In mathematical terms, that corresponds to treating the changes in size as 
 * being roughly uniform. 
 */

void AnalyzeL1(int Sizes[], double Times[], int Count, double Resolution )
{
  int i, j;


  fprintf(LogFile,"\nFinding L1 Cache Parameters.\n");

  L1Size = SharpRiseTest( Sizes, Times, Count, 
		    0.10 /* sensitivity in %; L1 is easy to find */ );

  if (L1Size == 0)
  {
    fprintf(stderr,"\nNo L1 Cache found.\n");
    fprintf(LogFile,"\nNo L1 Cache found.\n");
    return;
  }

  fprintf(LogFile,"L1 Cache Size is %s b.\n",PrintNum(L1Size*UnitSize));
  fprintf(stderr,"\nL1 Cache Size is %s b.\n",PrintNum(L1Size*UnitSize));

  sprintf(buffer,"%sS%s",TESTNAME,TESTSUFFIX);
  WriteResult(buffer,L1Size*UnitSize);

  if (CorDTest(L1Size, 4 , 512 ))
  {
    fprintf(stderr, "->CorD confirms that %s b is a Cache Level.\n",
	    PrintNum(L1Size*UnitSize));
    fprintf(LogFile,"\"->CorD confirms L1.\"\n");
  }
  fprintf(LogFile,"\"-------- End of Confirmation Run --------\"\n\n");

  /* LineSize is returned in bytes, so correct for it */
  L1Gran = FindGranularity( L1Size ) / UnitSize;
    sprintf(buffer,"%sG%s",TESTNAME,TESTSUFFIX);
  WriteResult(buffer,L1Gran*UnitSize);

  L1Assoc = AssocTrial( L1Size, L1Gran );

  if (L1Assoc)
  {
    fprintf(stderr,"L1 Associativity is %d.\n",L1Assoc);
    fprintf(LogFile,"\"L1 Associativity is %d.\"\n",L1Assoc);

    sprintf(buffer,"%sA%s",TESTNAME,TESTSUFFIX);
    WriteResult(buffer,L1Assoc);
  }
  else
  {
    fprintf(stderr,"L1 is fully associative.\n");
    fprintf(LogFile,"L1 is fully associative.\n");
    sprintf(buffer,"%sA%s",TESTNAME,TESTSUFFIX);
    WriteResult(buffer,0);
  }

  /* Pull a latency number from "Times[]" */
  FindLatency(Sizes,Times,Count,L1Size); 

}


/* can find page size by running the L1 CorD test until curve flattens */
/* TLB curve flattens at page size. */
 int CorDTest( int SuspectSize, int InitStride, int MaxStride )
{
  int Sizes[3];
  int Strides[MAX_TESTS_PER_RUN];
  double Times[MAX_TESTS_PER_RUN][3];
  double Trial;

  int InitBlockSize = 512;

  int Stride, Confirm;
  int i, j, nTests, SavedHeartBeat;
  double t1, t2, ratio;

  if (HeartBeat>1)
    fprintf(stderr,"CorD: Testing suspect point @ %sb.\n",
	    PrintNum(SuspectSize*UnitSize));

  SavedHeartBeat = HeartBeat;
  HeartBeat = 0;

  CorDFlattensAt = -1;

  fprintf(LogFile,"CorD: Testing suspect point @ %sb.\n",
	  PrintNum(SuspectSize*UnitSize));

  if (SuspectSize < 1024*1024)
  {
    Sizes[0] = SuspectSize - (SuspectSize/4);
    Sizes[1] = SuspectSize;
    Sizes[2] = SuspectSize + (SuspectSize/4); 
  }
  else 
  {
    Sizes[0] = SuspectSize - 64 * 1024;
    Sizes[1] = SuspectSize;
    Sizes[2] = SuspectSize + 64 * 1024;
  }

  nTests = 0;
  Stride = InitStride;
  while (Stride <= MaxStride)
  {
    BCTrial(Sizes, &Times[nTests][0], 3, InitBlockSize, Stride); 
    Strides[nTests] = Stride;

    Stride = Stride + Stride;
    nTests++;
  }

  Confirm = 1;
  for (i=0;i<nTests;i++)
  {
    t1 = fmax(Times[i][1] - Times[i][0],1.0);
    t2 = fmax(Times[i][2] - Times[i][1],1.0);
    ratio = t2 / t1;

    if (Debug>1)
      fprintf(LogFile,"@ Stride %6s b, ratio is %f / %f = %f.\n",
	      PrintNum(Strides[i]*UnitSize),t1, t2, ratio); 

    if (CorDFlattensAt < 1 && ratio < 1.1)
      CorDFlattensAt = Strides[i];

    if ((t1 == 1.0 && t2 > fmax(TimerTick * 0.1,2)) ||
	ratio > CorDThreshold)
      Confirm  = Confirm && 1;
    
    else 
      Confirm = 0;
  }

  if (Confirm)
    fprintf(LogFile,"-> CorD test confirms %s b as a valid level.\n",
	    PrintNum(Sizes[1]*UnitSize));
  else  
    fprintf(LogFile,"-> CorD test denies %s b as a valid level.\n",
	    PrintNum(Sizes[1]*UnitSize));
  HeartBeat = SavedHeartBeat;
  return Confirm;
}

/* Pull a latency number out of the Times[] data */
#define MAX_L1_LATENCY 100
static void FindLatency( int s[], double t[], int c, int l1 )
{
  int i, L1Index, max_ct, max_index, cycles;
  int L1Latencies[MAX_L1_LATENCY];

  for (i=0;i<c;i++)
  {
    if (s[i] == l1)
    {    
      L1Index = i;
      break;
    }
  }
  fprintf(LogFile,"\nLooking for L1 Latency\n");
  fprintf(LogFile,"FindLatency( *, *, %d, %s ) finds L1 at index %d (%s b).\n",
	  c,PrintNum(l1*UnitSize),L1Index,PrintNum(s[L1Index]*UnitSize));

  for (i=0;i<MAX_L1_LATENCY;i++)
    L1Latencies[i] = 0;

  for (i=0;i<=L1Index;i++)
  {
    cycles = (int) t[i];
    if (cycles < MAX_L1_LATENCY)
      L1Latencies[cycles]++;
  }
  fprintf(LogFile,"\nHistogram of Latencies:\nCycles\tCount\n");
  for (i=0;i<MAX_L1_LATENCY;i++)
  {
    if (L1Latencies[i] > 0)
      fprintf(LogFile,"%3d\t%3d\n",i,L1Latencies[i]);
  }

  max_ct = 0;
  max_index = 0;

  for (i=0;i<MAX_L1_LATENCY;i++)
  {
    if (L1Latencies[i] > max_ct)
    {
      max_ct = L1Latencies[i];
      max_index = i;
    }  
  }
  fprintf(LogFile,"Found latency of %d cycles (%d occurrences).\n",
	  max_index,max_ct);

  sprintf(buffer,"%sL%s",TESTNAME,TESTSUFFIX);
  WriteResult(buffer,max_index);

  fprintf(stderr,"->L1 Latency is %d cycles.\n",max_index);
}
