/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
/*
 *  The PACE Project, Rice University
 *
 *  Simple analysis routines used in the PACE Resource Characterization 
 *  Tools.
 *
 *
 */

#include "pace_rc_memlib.h"

int UnitSize;
int ExitEarly = 0;

/* Simple utility to reverse an IntList 
 *
 * Straight out of COMP 210 at Rice ...
 *
 */

static struct IntList *Reverse( struct IntList *list )
{
  struct IntList *p, *q, *r;

  if (list == (struct IntList *) NULL)
    return list;  /* empty list */

  if (list->Next == (struct IntList *) NULL)
    return list;  /* singleton list is already reversed */

  p = list;
  q = (struct IntList *) NULL;

  while (p != (struct IntList *) NULL)
  {
    r = p->Next;
    p->Next = q;
    q = p;
    p = r;
  }
  return q;
}


  /* The idea behind this test is simple.  We are looking for a sharp transition
   * in the data, such as that caused by the L1 cache.
   *
   * So, we scale the data so that a change of "+ Sensitivity" percent from
   * the initial point (Sizes[0] & Times[0]) will make the number 1.0.  Now,
   * we can scane the data for the point where the sign of the natural log 
   * changes. At that point, say i, we know that Sizes[i-1] fits in the memory
   * level and Sizes[i] does not.
   *
   * Assuming that we are sampling at the right granularity and points so that
   * the test hits the hardware boundary, it will be Sizes[i-1].  The choice of
   * sizes and granularity is the domain of the testing routine, not the analysis
   * routine.
   * 
   */

int SharpRiseTest( int Sizes[], double Times[], int Count, double Sensitivity )
{
  int i, result;
  double scale;

  if (Count < 2)
  {
    fprintf(LogFile,"Sharp Rise Test invoked with too few elements (%d) to analyze.\n",
	    Count);
    Abort("Sharp Rise Test has too few elements, see log file.",-1);
  }

  scale = (1.0/(1.0+Sensitivity)) / Times[0];
  result = 0;
  for (i=1; i<Count; i++)
  {
    if ((Times[i] * scale) > 1.0)
    {
      result = Sizes[i-1];
      break;
    }
  }
  return result;
}

static void InflErr1( int i, int j, char *s )
{
  if (ExitEarly == 0)
  {
    fprintf(LogFile,"In routine '%s':\n",s);
    fprintf(LogFile,"Difference of two consecutive sizes is <= 0 (%s & %s).\n",
	    PrintNum(i),PrintNum(j));
    fprintf(LogFile,"This error indicates a serious problem in the data.\n");
    fprintf(LogFile,"Slope is undefined for these points.\n");
    fprintf(LogFile,"Ending run; fix the code and try again.\n");
    ExitEarly++;
  }
  else
    fprintf(LogFile,"InflectionTest: ... also sizes (%s & %s).\n",
	    PrintNum(i),PrintNum(j));
}

/* This test looks for inflection points in the piecewise linear function
 * represented by the test data.
 *
 *
 */

#define NO  0
#define YES 1
#define RSlope 2
#define RDT    4
#define RAvg   8

struct IntList *InflectionTest(int Sizes[], double Times[], int Count, int Unit)
{
  int i, j, pt, rising, falling;

  double S[Count], DS[Count], DT[Count];
  double TSum, TAvg;
  int    Suspect[Count];


  if (Count < 3)
    Abort("InflectionTest: too few points to Analyze.\n",-1);

  Suspect[0] = NO;
  for (i=1; i<Count; i++)
  {
    DT[i] = Times[i] - Times[i-1];
    DS[i] = (double) (Sizes[i] - Sizes[i-1]);
    if (DS[i] <= 0) 
      InflErr1(Sizes[i],Sizes[i-1],"InflectionTest");
    else 
      S[i] = DT[i] / DS[i];
    Suspect[i] = NO;
  }
  if (ExitEarly)
    Abort("InflectionTest: Consecutive identical sizes - see log file.",-1);

  rising = NO;
  pt = 0;
  j = 0;

  /* print data in a separate pass so that LogFile is readable */
  fprintf(LogFile,"\n\"In Inflection Test:\"\nSize\t\tSlope\tDelta T\tChange\n");
  for (i=1;i<Count;i++)
  {
    fprintf(LogFile,"%3d:\t%10s\t%.4f\t%6s\t%f\n",
            i,PrintNum(Sizes[i]*Unit),S[i],
	    PrintDNum(DT[i]),DT[i]/Times[i-1]);
  }

  /* now, look for suspect points */
  for (i=1;i<Count;i++)
  {
    TAvg  = TSum / (double) i;

    if (rising &&
	(S[i] <= S[i-1]) &&
	((DT[i]/Times[i-1]) > 0.01)
      )
    {
      fprintf(LogFile,"\n\"Inflection Test finds point at size %s.\"\n",
	      PrintNum(Sizes[i-2]*Unit));
      fprintf(LogFile,"\"Change is %f\"\n",DT[i] / Times[i-1]);
      Suspect[i] = Suspect[i] | RSlope;
    }

    if (DT[i] > DMax(TAvg,10.0))
    {
      Suspect[i] = Suspect[i] | RAvg;
    }
    else 
    {
      TSum += DT[i];
    }

    if ((DT[i] > 20) &&
	(DT[i] > DT[i-1]) &&
	(DT[i] > DT[i+1]))
    {
      fprintf(LogFile,"\n\"Suspect at %s.\"\n",PrintNum(Sizes[i-1]*Unit));
      Suspect[i] = Suspect[i] | RDT;
    }

    if (S[i] >= S[i-1])
      rising = YES;
    else 
      rising = NO;

  }
  /* print data in a separate pass so that LogFile is readable */
  fprintf(LogFile,"\n\"Suspect points:\n");
  for (i=1;i<Count;i++)
  {
    if (Suspect[i])
    {
      fprintf(LogFile,"%3d: %10s\t",i,PrintNum(Sizes[i]*Unit));
      if (Suspect[i] & RSlope)
	fprintf(LogFile,"slope\t");
      else
	fprintf(LogFile,"\t");

      if (Suspect[i] & RDT)
	fprintf(LogFile,"Delta T\t");
      else
	fprintf(LogFile,"\t");

      if (Suspect[i] & RAvg)
	fprintf(LogFile,"Rising Avg\t");
      else
	fprintf(LogFile,"\t\t");

      fprintf(LogFile,"\n");
    }
  }
}

struct IntList *LogInflectionTest( 
		      int    X[],         /* Data on the X axis */
		      double Y[],         /* Function values for X[i] */
		      int Count,          /* Number of data points */
		      int (*Confirm)(int s) )
{
  int i, j, rising, pt;

  double DT[Count], DS[Count], S[Count];
  double SumDT, AvgDT;
  int    Suspect[Count];

  struct IntList *Result = (struct IntList *) NULL;
  struct IntList *p;

  if (Count < 3)
    Abort("LogInflectionTest: Too few points to analyze.\n",-1);

  ExitEarly = 0;
  Suspect[0] = NO;
  AvgDT = 0.0;
  for (i=1; i<Count; i++)
  {
    DT[i] = Y[i] - Y[i-1];
        DS[i] = log((double) (X[i]-X[i-1]));

    if (DS[i] <= 0)
      InflErr1(X[i],X[i-1],"LogInflectionTest");
    else
      S[i] = DT[i] / DS[i];
 
    if (DT[i] > 0)
      AvgDT += DT[i];
    else
      AvgDT -= DT[i];

    Suspect[i] = NO;
  }
  if (ExitEarly)
    Abort("LogInflectionTest: Consecutive identical sizes - see log file.",-1);

  AvgDT = AvgDT / (double) Count;
 
  if (0)
  {
    fprintf(LogFile,"\"LogInflectionTest Data\"\nSize      \tDT\t\tDS\t\tS\n");
    for (i=1;i<Count;i++)
      fprintf(LogFile,"%10s\t%f\t%f\t%f\n",
	      PrintNum(X[i]*UnitSize),DT[i],DS[i],S[i]);
  }

  rising = YES;  /* is this the right initialization, or do we need a more */
                 /* complex scheme...  This should prevent X[0] as a suspect */
  j = 0;
  SumDT = 0.0;

  /* Need to incorporate notion of slope already rising
   * along with some self-scaling tolerance, such as the average
   * Delta T to this point
   *
   * Looking for a rise of > tolerance after a region of no significant
   * (e.g. ><tolerance) rise
   *
   */

  fprintf(LogFile,"\n\n\"AvgDT: %.2f\"\n\"Searching for suspects:\"\n",AvgDT);
  /* now look for suspect points */
  for (i=1;i<Count;i++)
  {
    if (DT[i] > 0)
      SumDT += DT[i];
    else
      SumDT -= DT[i];
    AvgDT = SumDT / (double) i;

    fprintf(LogFile,"%12s\tDT: %8s\t(avg: %.2f)\tS: %.5f",
	    PrintNum(X[i]*UnitSize),PrintDNum(DT[i]),AvgDT,S[i]);
    if (rising == YES)
      fprintf(LogFile,"\t(rising)\n");
    else
      fprintf(LogFile,"\t(not)\n");

    if ((DT[i] > AvgDT) &&
	(rising == NO)  &&
        (S[i] > S[i-1]))
    {
      Suspect[i-1] = YES;
    }

    if ((S[i] > 0) &&
	(DT[i] > AvgDT))
    {
      rising = YES;
    }

    if (S[i] < S[i-1])
    {
      rising = NO;
    }
  }
  for (i=0; i<Count; i++)
  {
    if (Suspect[i] == YES)
    {
      fprintf(LogFile,"\n\"Suspect point @ %s b.\"\n",PrintNum(X[i]*UnitSize));
      if (Confirm(X[i]))
      {
        fprintf(LogFile,"--> %s b confirmed.\n",PrintNum(X[i]*UnitSize));
	p = (struct IntList *) malloc(sizeof(struct IntList));
        if (p == (struct IntList *) NULL)
	{
	  fprintf(LogFile,"In LogInflectionTest: malloc returns NULL.\n");
          fprintf(stderr, "In LogInflectionTest: malloc returns NULL.\n");
	  exit(-1);
	}
        p->Next = Result;
	p->Val = X[i];
	Result = p;
      }
      else
        fprintf(LogFile,"--> %s b denied.\n",PrintNum(X[i]*UnitSize));
    }
  }
  return Reverse(Result);
}

/* log-y test was DT[i] > 0.01 && DT[i] > DT[i-1] && DT[i] > DT[i+1] */ 

/* need to swap notion of rising and falling for convexity,
 * make log-x and log-y be options
 *
 */
