/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
#include <stdio.h>
#include <stdlib.h>
#include "../Lib/pace_rc_memlib.h"

FILE *Log;
char buf[256];

#define NValues 10
static int Values[NValues];
static int PointerComment = 1;
static FILE *RCDB;

void ZeroVals()
{
  int i;

  for (i=0;i<NValues;i++)
    Values[i] = 0;
}

char *SetName( char *name, int level, char *buf)
{
  sprintf(buf,name,level);
  return buf;
}

static int Close( int i, int j, double frac )
{
  int result;
  double x, y, ratio;

  x = (double) i;
  y = (double) j;

  if (x < y)
    ratio = 1.0 - (x / y);
  else
    ratio = 1.0 - (y / x);

  if (ratio <= frac) /* linear half of a power of 2 */
    result = 1;
  else 
    result = 0;

  return result;
}

static int EmergencyPlan()
{
  return 2*1024*1024;
}

int ProcessL1()
{
  int Size, Assoc, Latency;

  int L1, Gap, BC, Cache;

  ZeroVals();
  Size    = 0;
  Assoc   = 0;
  Latency = 0;

  fprintf(Log,"--------------------------------\nL1 Numbers:\n");

  L1 = ReadResultSoft("CacheL1Sa");
  fprintf(Log," L1 test found L1 cache size %s b.\n",PrintNum(L1));

  Gap = ReadResultSoft("AltCL1S");
  fprintf(Log," New Gap test found L1 cache size %s b.\n",PrintNum(Gap));

  if (L1 == Gap)
  {
    fprintf(Log," -> agreement.\n");
    Size = L1;
  }
  else if (Gap > 0 && Close(L1,Gap,0.25))
  {
    fprintf(Log," @ level 1, L1 (%s) and Gap (%s) are close.\n",
	    PrintNum(L1),PrintNum(Gap));
    Size = Gap;
    fprintf(Log," -> L1 and Gap are close, using Gap.\n");
  }
  else /* Backup Plan */  /* ugly case logic */
  {
    BC = ReadResultSoft("BC01S");
    fprintf(Log," BC test found L1 cache size %s b.\n",PrintNum(BC));

    Cache = ReadResultSoft("CL01S");
    fprintf(Log," Cache Only test found L1 cache size %s b.\n",
	    PrintNum(Cache));

    if (L1 <= 0 && Gap > 0 && (Gap == Cache || Gap == BC))
    {
      Size = Gap;
      fprintf(Log," -> using Gap (confirmed by one of Cache or BC)\n");
    }
    else if (L1 > 0 && Gap <= 0 && (L1  == Cache || L1  == BC))
    {
      Size = L1;
      fprintf(Log," -> using L1 (confirmed by one of Cache or BC)\n");
    }
    else if ((L1 > 0) && (Gap > 0) &&
	     (L1  == Cache || L1  == BC) &&
	     (Gap == Cache || Gap == BC))
    { /* the diabolical case of two x and two y */
      fprintf(Log," -> 2x vs 2y case, using Min(L1,Gap).\n");
      Size = Min(L1,Gap);
    }
    else if (L1 > 0 && Gap > 0 && (L1 == Cache || L1 == BC))
    {
      Size = L1;
      fprintf(Log,
	" -> L1 and Gap disagree, using L1 (confirmed by Cache or BC)\n");
    }
    else if (L1 > 0 && Gap > 0 && (Gap == Cache || L1 == BC))
    {
      Size = Gap;
      fprintf(Log,
	" -> L1 and Gap disagree, using Gap (confirmed by Cache or BC)\n");
    }
    else if (Cache > 0 && BC == Cache)
    {
      Size = Cache;
      fprintf(Log," -> Cache and BC agree, using Cache.\n");
    }
    else if (L1 > 0)
    {
      Size = L1;
      fprintf(Log," -> Only L1 ran, using L1.\n");
    }
    else if (Gap > 0)
    {
      Size = L1;
      fprintf(Log," -> Only L1 ran, using L1.\n");
    }
    else /* both l1test and gap test failed */
    {
      Size = EmergencyPlan();
    }
  }

  L1 = ReadResultSoft("CacheL1Aa");
  fprintf(Log,"\n L1 test found L1 associativity %d.\n",L1);
  Gap = ReadResultSoft("AltCL1A");
  fprintf(Log," New Gap test found L1 Associativity %d.\n",Gap);

  if (L1 == Gap)
  {
    fprintf(Log," -> agreement.\n");
    Assoc = L1;;
  }
  else if (L1 <= 0 && Gap > 0)
  {
    Assoc = Gap;
    fprintf(Log," -> using New Gap test value.\n");
  }
  else if (L1 > 0 && Gap <= 0)
  {
    Assoc = L1;
    fprintf(Log," -> using L1 test value.\n");
  }
  else if (L1 > 0 && Gap > 0)
  {
    Assoc = Min(L1,Gap);
    fprintf(Log," -> using smaller of L1 and New Gap = %d.\n",Assoc);
  }

  L1 = ReadResultSoft("CacheL1La");
  fprintf(Log,"\n L1 test found L1 latency %d.\n",L1);
  Gap = ReadResultSoft("AltCL1L");
  fprintf(Log," New Gap test found L1 latency %d.\n",Gap);

  BC = ReadResultSoft("BC01L");
  fprintf(Log," BC test found L1 latency %d.\n",BC);

  Cache = ReadResultSoft("CL01L");
  fprintf(Log," Cache test found L1 Latency %d.\n",Cache);

  if (L1 > 0 && L1 == Gap)
  {
    fprintf(Log," -> agreement.\n");
    Latency = L1;
  }
  else if (Gap > 0 && (Gap == BC || Gap == Cache))
  {
    Latency = Gap;
    fprintf(Log," -> using New Gap test value (confirmed by BC or Cache).\n");
  }
  else if (L1 > 0 && (L1 == BC || L1 == Cache))
  {
    Latency = L1;
    fprintf(Log," -> using L1 test value (confirmed by BC or Cache).\n");
  }
  else if (Cache > 0 && (Cache == BC))
  {
    Latency = Cache;
    fprintf(Log," -> using Cache test value (confirmed by BC).\n",Latency);
  }
  else if (Gap > 0)
  {
    Latency = Gap;
    fprintf(Log," -> using Gap test value.\n",Cache);
  }
  else if (L1 > 0)
  {
    Latency = L1;
    fprintf(Log," -> using L1 test value.\n",Cache);
  }
  else if (Cache > 0)
  {
    Latency = Cache;
    fprintf(Log," -> using Cache test value.\n",Cache);
  }
  else if (BC > 0)
  {
    Latency = BC;
    fprintf(Log," -> using BC test value.\n",BC);
  }
  else 
  {
    Latency = -1;
    fprintf(Log," -> no latency measurement succeeded.\n");
  }
  if (Size >= 0)
  {
    fprintf(RCDB,"cache.kind.data.level.%d.total_size %d\n",1,Size);
    fprintf(stdout,"L1 Cache: Size %12s",PrintNum(Size));
  }
  if (Latency >= 0)
  {
    fprintf(RCDB,"cache.kind.data.level.%d.load_cost %d\n",1,Latency);
    fprintf(stdout,"; Latency %d",Latency);
  }
  if (Assoc >= 0)
  {
    fprintf(RCDB,"cache.kind.data.level.%d.associativity %d\n",1,Assoc);
    fprintf(stdout,"; Associativity %d",Assoc);
  }
  fprintf(stdout,"\n");
  fprintf(Log,"--------------------------------\n");
  return Size;
}

/* Handle the upper levels of cache, with fewer tests and fewer 
 * measurements 
 *
 */
void ProcessUpperCacheLevels()
{
  int i, j, level, Result;
  int Cache[NValues], CacheL[NValues], cachei, CacheCount;
  int BC[NValues], BCL[NValues], bci, BCCount;
  int Size, Assoc, Gran, Latency;

  char buf[32];

  fprintf(Log,"Reading upper level cache values\n");
  for (i=0;i<NValues;i++)
  {
    BC[i]  = ReadResultSoft(SetName("BC%02dS", i+2, buf));
    BCL[i] = ReadResultSoft(SetName("BC%02dL", i+2, buf));
    if (BC[i] < 0)
    {
      BCCount = i;
      break;
    }

    else fprintf(Log,"  Found BC level %3d @ %12s with latency %d.\n",
		 i+2,PrintNum(BC[i]),BCL[i]);
  }

  fprintf(Log,"\n");

  for (i=0;i<NValues;i++)
  {
    Cache[i]  = ReadResultSoft(SetName("CL%02dS", i+2, buf));
    CacheL[i] = ReadResultSoft(SetName("CL%02dL", i+2, buf));
    if (Cache[i] < 0)
    {
      CacheCount = i;
      break;
    }
    else fprintf(Log,"  Found Cache level %3d @ %12s with latency %d.\n",
		 i+2,PrintNum(Cache[i]),CacheL[i]);
  }


  level  = 2;
  bci    = 0;
  cachei = 0;
 
  while(1) 
  {
    fprintf(Log,"--------------------------------\n");
    fprintf(Log,"Looking for level %2d.\n",level);
    Size = 0;
    Latency = 0;
    if (Cache[cachei] > 0 && Close(Cache[cachei],BC[bci],0.5))
    {
      fprintf(Log,"\n @ level %02d, Cache (%s) and BC (%s) are close.\n",
	      level,PrintNum(Cache[cachei]),PrintNum(BC[bci]));
      fprintf(Log," Consider them a single level.\n");

      if (Cache[cachei] == BC[bci])
      {
        Size = Cache[cachei];
	fprintf(Log," -> Cache and BC agree on size %s.\n",PrintNum(Size));
	if ((CacheL[cachei] > 0) && (BCL[bci] > 0)) /* take smaller latency */
	  Latency = Min(CacheL[cachei],BCL[bci]);
	else if (CacheL[cachei] > 0)
	  Latency = CacheL[cachei];
	else if (BCL[bci] > 0)
	  Latency = BCL[bci];

	cachei++;
	bci++;
      }
      else if (Cache[cachei] < BC[bci])
      {
	Size = Cache[cachei];
	fprintf(Log," -> Using Cache because it is smaller.\n");
	if ((CacheL[cachei] > 0) && (BCL[bci] > 0)) /* take smaller latency */
	  Latency = Min(CacheL[cachei],BCL[bci]);
	else if (CacheL[cachei] > 0)
	  Latency = CacheL[cachei];
	else if (BCL[bci] > 0)
	  Latency = BCL[bci];

	cachei++;
	bci++;
      }
      else 
      {
	Size = BC[bci];
	fprintf(Log," -> Using BC because it is smaller.\n");
	if ((CacheL[cachei] > 0) && (BCL[bci] > 0)) /* take smaller latency */
	  Latency = Min(CacheL[cachei],BCL[bci]);
	else if (CacheL[cachei] > 0)
	  Latency = CacheL[cachei];
	else if (BCL[bci] > 0)
	  Latency = BCL[bci];

	cachei++;
	bci++;
      }
    }
    else if (Cache[cachei] > 0 && (Cache[cachei] < BC[bci] || bci >= BCCount))
    {
      Size = Cache[cachei];
      if ((CacheL[cachei] > 0) && (BCL[bci] > 0)) /* take smaller latency */
	Latency = Min(CacheL[cachei],BCL[bci]);
      else if (CacheL[cachei] > 0)
	Latency = CacheL[cachei];
      else if (BCL[bci] > 0)
	Latency = BCL[bci];

      cachei++;
    }
    else if (BC[bci] > 0 && (BC[bci] < Cache[cachei] || cachei >= CacheCount))
    {
      Size = BC[bci];
      if ((CacheL[cachei] > 0) && (BCL[bci] > 0)) /* take smaller latency */
	Latency = Min(CacheL[cachei],BCL[bci]);
      else if (CacheL[cachei] > 0)
	Latency = CacheL[cachei];
      else if (BCL[bci] > 0)
	Latency = BCL[bci];

      bci++;
    }

    /* Now do the corresponding latency test */
    if (Latency == 0)
    {
      fprintf(Log,"No valid latency data for level %d.\n",level);
      if (Size > 0)
      {
	Latency = 10 * level;
	fprintf(Log,
		"-> Size is valid, so using guess of %d for level %d latency.\n",
		Latency,level);
      }
    }
    if (Latency > 0)
      fprintf(Log," -> Chose latency %d.\n",Latency);

    /* write results */
    if (Size > 0)
    {
      fprintf(RCDB,"cache.kind.data.level.%d.total_size %d\n",level,Size);
      fprintf(stdout,"L%d Cache: Size %12s",level,PrintNum(Size));
    }
    if (Latency > 0)
    {
      fprintf(RCDB,"cache.kind.data.level.%d.load_cost %d\n",level,Latency);
      fprintf(stdout,"; Latency %d",Latency);
    }
    fprintf(stdout,"\n");

    if (Size == 0)
      break;
    level++;
  }
}


int ProcessTLBLevel( int level )
{
  int i, j, Result;
  int Size, Assoc, Gran;

  char buf[32], *p;

  fprintf(Log,"--------------------------------\n");
  fprintf(Log,"\nL%d TLB Values\n",level);

  /* Size */
  ZeroVals();

  Result = 1;
  Size = -1;
  Assoc = -1;
  Gran = GetOSPageSize();  /* a given */

  /* at this point, the TLBI numbers are not mature enough to use */
  /* so we rely on the TLBF numbers */
  
  p  = SetName("TLBI%02dS", level, buf);
  Size = ReadResultSoft(p);
  fprintf(Log,"  TLB-I test found size %s in '%s'.\n",PrintNum(Size),p);
 
  if (Size == -1)  /* read failure at this level on the size file */
  {               /* => return a 0 to exit loop in main() */
    return 0;
  }

  /* Associativity */ 
  Assoc = ReadResultSoft(SetName("TLBI%02dA", level, buf));
  fprintf(Log,"  TLB-I test found associativity %d.\n",Assoc);

  /* Doctor the value, if necessary */
  if (Assoc == 0)
  {
    fprintf(Log,"  -> zero means full associativity.\n");
  }
  else if (Assoc == -1) /* Assume a reasonable value */
  {
    fprintf(Log,"  -> no value detected; none reported.\n");
  }

  if (Result)
  {
    fprintf(Log,
	    "L%d TLB:   Size %10s; Associativity: %2s; Granularity %s\n",
	    level,
	    PrintNum(Size),PrintNum(Assoc),PrintNum(Gran));

    /* Write out prototype XML lines for the report generator */
    /* This stuff goes to PACE_RCDB */
    if (Size > 0)
    {
      fprintf(RCDB,"cache.kind.TLB.level.%d.total_size %d\n",level,Size);
      fprintf(stdout,"L%d TLB: Total Size %12s",level,PrintNum(Size));
    }
    if (Gran > 0)
    {
      fprintf(RCDB,"cache.kind.TLB.level.%d.pageorline_size %d\n",level,Gran);
      fprintf(stdout,"; Page Size %s",PrintNum(Gran));
    }
    if (Assoc >= 0)
    {
      fprintf(RCDB,
	      "cache.kind.TLB.level.%d.associativity %d\n",level,Assoc);
      fprintf(stdout,"; Associativity %d",Assoc);
    }
    fprintf(stdout,"\n");
  }
  else
    fprintf(Log,"-> no level %d found.\n",level);

  return Result;
}

int main ( int argc, char ** argv )
{
  int i,result;
  int Size, Assoc, Gran, Latency;



  fprintf(stderr,"PACE RC Memory Tests\nValue Reconciliation Tool\n");

  Log = fopen("Log","w");
  if (Log == NULL) 
  {
    fprintf(stderr,"fopen() of 'Log' failed.\n");
    exit(-2);
  }
  (void) SetupAndMark( Log, "Log");
  fprintf(Log,"Run of PACE RC Memory Test Value Reconciliation Tool.\n");

  RCDB = fopen("../../pace_rcdb","a");
  if (RCDB == NULL)
  {
    fprintf(stderr,"fopen() of PACE_RCDB file failed.\n");
    exit(-2);
  }

  result = ProcessL1();

  if (result > 0)
     ProcessUpperCacheLevels(i);


  /* To avoid false positives from the multiplicative echos of L2 TLBs, */
  /* we will limit ourselves to reporting just 2 TLBs.                  */
  /* The echo problem arises on a Power7 processor.                     */
  result = ProcessTLBLevel(1);
  result = ProcessTLBLevel(2);


  return 0;
}
