/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
#include "MemTest.h"
#include "BlackJackTimer.h"

int GlobalZero;

static int *ColumnIndexSet   = (int *) NULL;
static int *RowIndexSet = (int *) NULL;
static int StartingPoint= -1;

static int NotYetComplained = 0;
static int RandomStaticFlag = 0;

static struct AccessCount NAccesses; 

static char buf[256];

static void **MemArray = (void **) NULL;

static int  ConfirmTLB( int s );

double TLBTest( int s, int ps, int ls, struct AccessCount na );
int IsItACache( int s, int lbst, int ubst );

/**  Tests for case 1 -- We are seeing a response (upward jump) in the 
 **  TLB Only permutation from loss of associativity in a lower level 
 **  of the memory hierarchy.
 **
 **  In this case, we build the same set of lines with fewer pages and
 **  see what happens. If the response is a TLB response, we expect to
 **  see a much faster run.  If it is from the lower caches, we should
 **  see the same respnse.  (Packed is faster => suspect is a TLB)
 **/



/* BuildPackedPerm 
 *
 * Parameters: ArraySize, PageSize, LineSize 
 * 
 *
 *
 */

static void BuildTLBPackedPerm( int ArraySize,
				int PageSize, /* TLB Page Size */
				int LineSize )
{
  int i, j, col, ThisElt, LastElt, count;
  int NRows, NColumns, Row;
  void **p;

  NRows = ArraySize /  PageSize;
  NColumns = PageSize / LineSize;

  if (Debug>1)
    fprintf(LogFile,
	    "TLBPackePerm: %d rows x %d columns x %d stride = %s (=? %s w).\n",
	    NRows,NColumns,LineSize,PrintNum(NRows * NColumns * LineSize),
	    PrintNum(ArraySize));

  if (NRows * NColumns * LineSize < ArraySize && NotYetComplained)
  {
    NotYetComplained = 0;
    fprintf(stderr,
	    "\nSpacing between points is less than page size.\n");
    fprintf(stderr,
	    "Duplicate points at low end. (Run w/-g option to see.)\n\n");
    fprintf(LogFile,
	    "\"\nSpacing between points is less than page size.\"\n");
    fprintf(LogFile,
	    "\"Duplicate points at low end. (Run w/-g option to see.\"\n\n");
  }

  if (LineSize >= PageSize)
  {
    fprintf(stderr,"BuildOddTLBPerm() receives invalid parameters, see log\n");
    fprintf(LogFile,"\nBuildOddTLBPerm: invalid parameters.\n");
    fprintf(LogFile,"Line size %04d, page size %04d\n",LineSize,PageSize);
    fprintf(LogFile,"Test makes no sense in this context.\n");
    (void) exit(-1);
  }

  ColumnIndexSet  = PACE_AllocMem( NColumns * sizeof(int) );

  /* Generate the index sets we need & shuffle them */
  (void) GenerateLinearSet( ColumnIndexSet, NColumns, LineSize );
  (void) Shuffle( ColumnIndexSet, NColumns );


  /* Allocate the Memory Array */
  if (MemArray != (void **) NULL)
  {
    fprintf(LogFile,"Lost MemArray in IsItATLB - Site 1.\n");
  }
  MemArray = PACE_AllocMem(ArraySize * UnitSize);

  for (i=0;i<ArraySize;i++)  /* initialize the array to a known wrong value */
    MemArray[i] = (void **) NULL;

  LastElt = 0; /* distinct from uninitialized */
               /* MemArray[StartingPoint] gets written twice */
               /* Once in 1st iteration; once after the loop */
  j = 0;
  Row = 0;
  count = 0;
  for (i=0;i<NRows;i++)
  {
    ThisElt = Row + ColumnIndexSet[j];

    /* When this page is full, bump to the next page */
    if (MemArray[ThisElt] != (void **) NULL) 
    {
      Row += PageSize;
      ThisElt = Row + ColumnIndexSet[j];
    }

    MemArray[ThisElt] = &MemArray[LastElt];

    if (Debug>1)
      fprintf(LogFile,"%04d: M[%6s] <- %6s.\t\t(%s : %s)\n",count,
	      PrintNum(ThisElt),PrintNum(LastElt),PrintNum(Row),
	      PrintNum(ColumnIndexSet[j]));
    LastElt = ThisElt;
    count++;
    j++;
    if (j == NColumns)
      j = 0;
  }

  /* and, finally, MemArray[0][0] <- MemArray[NRows-1][NCols-1] */
  StartingPoint = ColumnIndexSet[0];
  MemArray[StartingPoint] = &MemArray[LastElt];
  if (Debug >1)
  {
    fprintf(LogFile,"%04d: M[%6s] <- %6s.\n", count+1,
	    PrintNum(StartingPoint),PrintNum(LastElt));
    fprintf(LogFile,
	    "BuildTLBPackedPerm used %d rows (of %d rows) with %d elts.\n",
	    1+ Row/PageSize,NRows,count);
  }

  /* Check the permutation */
  if (Debug>1)
    fprintf(LogFile,"M[%s] <- %s. ** starting point **\n",
	    PrintNum(StartingPoint),PrintNum(LastElt));

  /* verify permutation */
  p = MemArray[StartingPoint];
  i = 0;
  count = NRows;
  while (p != &MemArray[StartingPoint])
  {
    p = *p;
    i++;
    if (i > count)
    {
      fprintf(stderr,"Cycle did not return to starting point.\n");
      fprintf(stderr,"Cycle length is %s of %s.\n",
	      PrintNum(i),PrintNum(count));
      break;
    }
  }
  if ((Debug) && (i == count-1))
    fprintf(LogFile,"\"Maximal length permutation.\"\n");
  if (i < count-1)
    fprintf(LogFile,"\n\"Short permutation. (%s vs %s) -- 1\"\n",
	    PrintNum(i),PrintNum(count-1));

  PACE_FreeMem(ColumnIndexSet);
  ColumnIndexSet = (int *) NULL;
}

/* TLBCase1
 * 
 * Conducts a specialized test to look for a TLB
 * 
 * PARAMETERS:  ArraySize, PageSize, LineSize, NA   
 * 
 * RETURNS:An elapsed time, in microseconds, as a double 
 * 
 */

static double TLBCase1( int ArraySize, /* footprint for test, in WORDS  */
			 int PageSize,
			 int LineSize,
			 struct AccessCount NA    /* number of iterations*/
			 )
{
  int i;
  double result;

  if (HeartBeat > 1)
    fprintf(stderr,"Trial @ %s b: ",PrintNum(ArraySize*UnitSize));

  /* Initialize MemArray */
  BuildTLBPackedPerm(ArraySize,PageSize,LineSize);

  Flush(ArraySize);

  /* Run the test */
  result = TimePermPtr(MemArray, StartingPoint, NA);

  PACE_FreeMem(MemArray);
  MemArray = (void**) NULL;

  if (HeartBeat > 1)
    fprintf(stderr,"%s usec\n",PrintDNum(result));

  return result;
}

/**  Tests for case 2 -- the suspect point is out beyond the largest TLB
 **  and we are simply seeing a cache response from some larger cache ...
 **
 **  Here, we build a TLB Only Permutation, except that we thread it 
 **  through four lines in each page. When it runs, it should have 1/4
 **  as many TLB misses as the original. If the time is due to TLB, it
 **  should change (by about 4x).  If it is due to cache, it should not
 **  change.  (Decrease in time => suspect is a TLB)
 **/

static Counter = 0;

static void BuildVariantTLBPerm( int ArraySize,
				 int PageSize, /* TLB Page Size */
				 int LineSize )
{
  int i, j, k, dist, ThisElt, LastElt, count;
  int NRows, NColumns;
  void **p;

  NRows = ArraySize /  PageSize;
  NColumns = PageSize / LineSize;

  Counter++;

  if (Debug)
    fprintf(LogFile,
	    "TLB Perm: %d rows x %d columns x %d stride = %s (=? %s w).\n",
	    NRows,NColumns,LineSize,PrintNum(NRows * NColumns * LineSize),
	    PrintNum(ArraySize));

  if (NRows * NColumns * LineSize < ArraySize && NotYetComplained)
  {
    NotYetComplained = 0;
    fprintf(stderr,
	    "\nSpacing between points is less than page size.\n");
    fprintf(stderr,
	    "Duplicate points at low end. (Run w/-g option to see.)\n\n");
    fprintf(LogFile,
	    "\"\nSpacing between points is less than page size.\"\n");
    fprintf(LogFile,
	    "\"Duplicate points at low end. (Run w/-g option to see.\"\n\n");
  }

  if (LineSize >= PageSize)
  {
    fprintf(stderr,"BuildOddTLBPerm() receives invalid parameters, see log\n");
    fprintf(LogFile,"\nBuildOddTLBPerm: invalid parameters.\n");
    fprintf(LogFile,"Line size %04d, page size %04d\n",LineSize,PageSize);
    fprintf(LogFile,"Test makes no sense in this context.\n");
    (void) exit(-1);
  }

  ColumnIndexSet  = PACE_AllocMem( NColumns * sizeof(int) );
  RowIndexSet  = PACE_AllocMem( NRows * sizeof(int) );

  /* Generate the index sets we need & shuffle them */
  (void) GenerateLinearSet( ColumnIndexSet, NColumns, LineSize );
  (void) Shuffle( ColumnIndexSet, NColumns );

  (void) GenerateLinearSet( RowIndexSet, NRows, PageSize );
  (void) Shuffle( RowIndexSet, NRows );

  /* Allocate the Memory Array */
  if (MemArray != (void**) NULL)
  {
    fprintf(LogFile,"Lost MemArray in IsItATLB - Site 2.\n");
  }
  MemArray = (void **) PACE_AllocMem(ArraySize * UnitSize);

  dist = PageSize / 4;

  if (Debug>1)
    fprintf(LogFile,"\nBuilding New Permutation %d\n",Counter);

  /* Assemble the permutation */
  LastElt = -1;
  j = 0;
  for (i=0;i<NRows;i++)
  {
    ThisElt = RowIndexSet[i] + ColumnIndexSet[j];
    if (LastElt != -1)
      MemArray[ThisElt] = &MemArray[LastElt];

    if (Debug>1)
      fprintf(LogFile,"\nM[%s] <- %s.\t\tRow %s,\tCol %5s",
	      PrintNum(ThisElt),PrintNum(LastElt),
	      PrintNum(RowIndexSet[i]),PrintNum(ColumnIndexSet[j]));

    LastElt = ThisElt;

    for (k=1;k<4;k++) /* add extra refs per row (row is a page) */
    {
      ThisElt = ColumnIndexSet[j] + k * dist;
      if (ThisElt >= PageSize)
	ThisElt -= PageSize;

      if (Debug>1)
	fprintf(LogFile," + %5s",PrintNum(ThisElt));

      ThisElt = ThisElt +  RowIndexSet[i];

      MemArray[ThisElt] = &MemArray[LastElt]; 
      LastElt = ThisElt;
    }

    j++;
    if (j >= NColumns)
      j = 0;
  }
  /* and, finally, MemArray[0][0] <- MemArray[NRows-1][NCols-1] */
  StartingPoint = RowIndexSet[0] + ColumnIndexSet[0];
  MemArray[StartingPoint] = &MemArray[LastElt];

  /* Check the permutation */
  if (Debug>1)
    fprintf(LogFile,"\nM[%s] <- %s. ** starting point **\n",
	    PrintNum(StartingPoint),PrintNum(LastElt));

  /* verify permutation */
  p = MemArray[StartingPoint];
  i = 0;
  count = NRows * 4;
  while (p != &MemArray[StartingPoint])
  {
    p = *p;
    i++;
    if (i > count)
    {
      fprintf(stderr,"%d: Cycle did not return to starting point.\n",Counter);
      fprintf(stderr,"Cycle length is %s of %s.\n",
	      PrintNum(i),PrintNum(count));
      break;
    }
  }
  if ((Debug) && (i == count-1))
    fprintf(LogFile,"\"Maximal length permutation.\"\n");
  if (i < count-1)
    fprintf(LogFile,"\n\"Short permutation (%d vs %d).\"\n",i,count-1);

  PACE_FreeMem(ColumnIndexSet);
  ColumnIndexSet = (int *) NULL;

  PACE_FreeMem(RowIndexSet);
  RowIndexSet = (int *) NULL;
}


/* TLBCase2
 * 
 * Conducts a specialized test to look for a TLB
 * 
 * PARAMETERS:  ArraySize, PageSize, LineSize, NA   
 * 
 * RETURNS:An elapsed time, in microseconds, as a double 
 * 
 */

static double TLBCase2( int ArraySize, /* footprint for test, in WORDS  */
			 int PageSize,
			 int LineSize,
			 struct AccessCount NA    /* number of iterations*/
			 )
{
  int i;
  double result;

  if (HeartBeat > 1)
    fprintf(stderr,"Trial @ %s b: ",PrintNum(ArraySize*UnitSize));

  /* Initialize MemArray */
  BuildVariantTLBPerm(ArraySize,PageSize,LineSize);

  Flush(ArraySize);

  /* Run the test */
  result = TimePermPtr(MemArray, StartingPoint, NA);

  PACE_FreeMem(MemArray);
  MemArray = (void**) NULL;

  if (HeartBeat > 1)
    fprintf(stderr,"%s usec\n",PrintDNum(result));

  return result;
}


/* CheckForFalsePositive --
 * 
 * False positive can arise in two ways:
 * 
 * 1.) The code runs out of lines or associativity at some lower level of 
 *     the memory hierarchy.  To test this case, we will fold the TLB Only
 *     access pattern back into a smaller number of pages. If the TLB really
 *     exists and this point is not a false positive, then the rise in time
 *     should go away. If the resulting time is unchanged, then the effect
 *     arises from a lower level in the memory hierarchy and we should deny
 *     this point.
 *
 * 2.) The code runs past the size of the last TLB and measures a larger 
 *     level of cache. To detect this false positive, we compare the 
 *     results against the results of the memory test.
 *
 */

int IsItATLB( int  Size, 
	      int AllSizes[], 
	      double AllTimes[], 
	      int Count,
	      int PageSize, 
	      int L1LineSize )
{
  int i, j, k, s, Case1, Case2, count, ticker, trigger, SaveHeartBeat;
  double Original, Folded, Expanded, trial;

  struct AccessCount NA;

  NA.outer = 1;
  NA.inner = 1000000;

  SaveHeartBeat = HeartBeat;
  HeartBeat = 0;

  fprintf(LogFile,"\nIsItATLB( %s b ) [PS: %s b, LS: %s b, NA (%s : %s)].\n",
	  PrintNum(Size*UnitSize),PrintNum(PageSize*UnitSize),
	  PrintNum(L1LineSize*UnitSize),
	  PrintNum(NA.outer),PrintNum(NA.inner));

  trigger = -1;
  for (i=0;i<Count;i++)
  {
    if (AllSizes[i] == Size && (i+1) < Count)
    {
      trigger = i+1;
      break;
    }
  }

  if (trigger == -1) /* something is wrong in the handshake with TLBTest */
  {
    fprintf(stderr,"\nIsItATLB() doesn't find array size.\nReturns False\n");
    return 0;
  }

  s = AllSizes[trigger];
  fprintf(LogFile,"-> Testing lower level interference at next size %s.\n",
	  PrintNum(s*UnitSize));

  /**   Test for Case 1 -- Interference from lower levels of cache 
   **
   **/

  count = TRIALS_FOR_MIN;
  ticker = 1;
  Original = TLBTest( s, PageSize, L1LineSize, NA );
  while (count > 0)
  {
    trial =  TLBTest( s, PageSize, L1LineSize, NA );
    if (trial < Original)
    {
      Original = trial;
      count = TRIALS_FOR_MIN;
    }
    else
      count--;
    ticker++;
  }

  if (MemArray != (void**) NULL)
    PACE_FreeMem(MemArray);

  fprintf(LogFile,
	  "IsItATLB() took %s trials to find min %s at %s with TLB Only.\n",
	  PrintNum(ticker),PrintDNum(Original),PrintNum(s*UnitSize));

  count = TRIALS_FOR_MIN;
  ticker = 1;
  Folded = TLBCase1( s, PageSize, L1LineSize, NA );
  while (count > 0)
  {
    trial =  TLBCase1( s, PageSize, L1LineSize, NA );
    if (trial < Folded)
    {
      Folded = trial;
      count = TRIALS_FOR_MIN;
    }
    else
      count--;
    ticker++;
  }

  fprintf(LogFile,
	  "IsItATLB() took %s trials to find min %s at %s with Case 1.\n",
	  PrintNum(ticker),PrintDNum(Folded),PrintNum(s*UnitSize));
  
  if ((Original - Folded) > 0.02 * Original)
    Case1 = 1;
  else
    Case1 = 0;

  fprintf(LogFile,"-> Original %s, Folded %s => %1d\n",
	  PrintDNum(Original),PrintDNum(Folded),Case1);


  /**   Test for Case 2 -- Interference from lower levels of cache 
   **
   **/

  fprintf(LogFile,"Testing Case 2 with CorD Test at %s b.\n",
	  PrintNum(Size*UnitSize));

  Case2 = IsItACache( Size, L1LineSize, PageSize / 4 );

  if (Case2)  /* invert sense of the test and report it */
  {
    fprintf(LogFile,"-> IsItACache thinks it is a cache.\n");
    Case2 = 0;
  }
  else
  {
    fprintf(LogFile,"-> IsItACache thinks it is not a cache.\n");
    Case2 = 1;
  }

  fprintf(LogFile,"--> %1d\n",Case1&&Case2);

  if (ColumnIndexSet != (int*) NULL)
  {
    PACE_FreeMem(ColumnIndexSet);
    ColumnIndexSet = (int*) NULL;
  }

  if (RowIndexSet != (int*) NULL)
  {
    PACE_FreeMem(RowIndexSet);
    RowIndexSet = (int*) NULL;
  }

  if (MemArray != (void **) NULL)
  {
    PACE_FreeMem(MemArray);
    MemArray = (void**) NULL;
  }

  HeartBeat = SaveHeartBeat;

  return Case1 && Case2;
}
