/* Copyright 2011, Rice University.  All rights reserved.
   No warranty of usability express or implied.  Have a lovely day! */
/*
 *  PACE Project, Rice University
 *
 *  Functions to report into the PACE_RCDB directory
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

void PACE_RCDB_write_string(const char *key, const char *value);
void PACE_RCDB_write_int(const char *key, int value);
void PACE_RCDB_write_double(const char *key, double value);

void PACE_TEMP_write_string(const char *filename, const char *value);
void PACE_TEMP_write_int(const char *filename, int value);
void PACE_TEMP_write_double(const char *filename, double value);

int PACE_TEMP_read_string(const char *filename, char *value, unsigned int value_len);
int PACE_TEMP_read_int(const char *filename, int *value);
int PACE_TEMP_read_double(const char *filename, double *value);

static const char *PACE_RCDB_get_filename();
static const char *PACE_TEMP_get_dirname();




static const char *PACE_RCDB_get_filename()
{
  static const char *filename = NULL;
  if (filename == NULL) {
    filename = getenv("PACE_RCDB");
    if (filename == NULL)
      {
        fprintf(stderr,"WARNING:  Environment variable \"PACE_RCDB\" is not set.\n");
        fprintf(stderr,"          Using local \"PACE_RCDB\" instead.\n");
        filename = "../../../pace_rcdb";
      }
  }
  return filename;
}

static const char *PACE_TEMP_get_dirname()
{
  static const char *filename = NULL;
  if (filename == NULL) {
    filename = getenv("PACE_TEMP");
    if (filename == NULL)
      {
        fprintf(stderr,"WARNING:  Environment variable \"PACE_TEMP\" is not set.\n");
        fprintf(stderr,"          Using '.' instead.\n");
        filename = ".";
      }
  }
  return filename;
}

void PACE_RCDB_write_string(const char *key, const char *value)
{
  const char *filename = PACE_RCDB_get_filename();
  int  retval;
  FILE *OutFile;

  fprintf(stderr,"Writing to \"%s\".\n",filename);

  OutFile = fopen(filename,"a");
  if (OutFile == NULL)
  {
    fprintf(stderr,"Cannot open file \"%s\" for output.\n",filename);
    exit(-1);
  }
  fprintf(OutFile,"%s %s\n", key, value);
  retval = fclose(OutFile);
  if (retval != 0)
  {
    fprintf(stderr,"flose(\"%s\") failed.\n",filename);
    exit(-1);
  }
}


void PACE_RCDB_write_int(const char *key, int value)
{
  char buffer[128];
  sprintf(buffer, "%d", value);
  PACE_RCDB_write_string(key, buffer);
}

void PACE_RCDB_write_double(const char *key, double value)
{
  char buffer[128];
  sprintf(buffer, "%g", value);
  PACE_RCDB_write_string(key, buffer);
}




void PACE_TEMP_write_string(const char *filename, const char *value)
{
  const char *dirname = PACE_TEMP_get_dirname();
  int  retval;
  FILE *OutFile;

  {
    char buffer[1024];
    if (snprintf(buffer, 1024, "%s/%s", dirname, filename) >= 1024) {
      fprintf(stderr,"ERROR:  filename too long\n");
      exit(-1);
    }

    fprintf(stderr,"Writing to \"%s\".\n", buffer);
    OutFile = fopen(buffer,"w");
    if (OutFile == NULL)
      {
        fprintf(stderr,"Cannot open file \"%s\" for output.\n", buffer);
        exit(-1);
      }
  }


  fprintf(OutFile,"%s\n", value);
  retval = fclose(OutFile);
  if (retval != 0)
  {
    fprintf(stderr,"flose(\"%s\") failed.\n",dirname);
    exit(-1);
  }
}


void PACE_TEMP_write_int(const char *filename, int value)
{
  char buffer[128];
  sprintf(buffer, "%d", value);
  PACE_TEMP_write_string(filename, buffer);
}

void PACE_TEMP_write_double(const char *filename, double value)
{
  char buffer[128];
  sprintf(buffer, "%f", value);
  PACE_TEMP_write_string(filename, buffer);
}

int PACE_TEMP_read_string(const char *filename, char *value, unsigned int value_len)
{
  const char *dirname = PACE_TEMP_get_dirname();
  char pathname[1024];
  char buffer[1024];
  int  retval;
  FILE *InFile;

  int rc = 0;

  if (value_len < 1) {
    fprintf(stderr, "ERROR:  value_len < 1\n");
    exit(-1);
  }

  if (snprintf(pathname, 1024, "%s/%s", dirname, filename) >= 1024) {
    fprintf(stderr,"ERROR:  filename too long\n");
    exit(-1);
  }
  
  fprintf(stderr,"Reading from \"%s\".\n", pathname);
  InFile = fopen(pathname,"r");
  if (InFile == NULL) {
    return -1;
  }

  if (fscanf(InFile, "%1024s\n", buffer) == 1) {
    strncpy(value, buffer, value_len);
    value[value_len-1] = '\0';
    rc = 0;
  } else {
    rc = -1;
  }

  retval = fclose(InFile);
  if (retval != 0) {
    fprintf(stderr,"flose(\"%s\") failed.\n", pathname);
    exit(-1);
  }
  return rc;
}

int PACE_TEMP_read_int(const char *filename, int *value)
{
  char buffer[1024];
  int rc = PACE_TEMP_read_string(filename, buffer, 1024);
  if (rc != 0) return rc;
  *value = atoi(buffer);
  return rc;
}

int PACE_TEMP_read_double(const char *filename, double *value)
{
  char buffer[1024];
  int rc = PACE_TEMP_read_string(filename, buffer, 1024);
  if (rc != 0) return rc;
  *value = atof(buffer);
  return rc;
}






/* #include <assert.h> */

/* int main() { */
/*   char str_val[1024]; */
/*   int  int_val; */
/*   double double_val; */
  
/*   PACE_TEMP_write_string("deleteme.report.test1", "test1"); */

/*   assert(0 == PACE_TEMP_read_string("deleteme.report.test1", */
/*                                     str_val, sizeof(str_val)/sizeof(str_val[0]))); */
/*   assert(strcmp("test1", str_val) == 0); */

  
/*   PACE_TEMP_write_int("deleteme.report.test2", 2342); */
/*   assert(0 == PACE_TEMP_read_int("deleteme.report.test2", &int_val)); */
/*   assert(int_val == 2342); */


/*   PACE_TEMP_write_double("deleteme.report.test3", 5432432.235); */
/*   assert(0 == PACE_TEMP_read_double("deleteme.report.test3", &double_val)); */
/*   assert(double_val == 5432432.235); */

/*   assert(0 != PACE_TEMP_read_int("deleteme.report.test4", &int_val)); */

/*   return 0; */
/* } */
