#include "MemTest.h"
#include "BlackJackTimer.h"

int GlobalZero;

static int *ColumnIndexSet   = (int *) NULL;
static int *RowIndexSet      = (int *) NULL;
static int StartingPoint     = -1;

static int NotYetComplained = 0;
static int RandomStaticFlag = 0;

static struct AccessCount NAccesses; 

static char buf[256];

static void **MemArray = (void **) NULL;

static int  ConfirmTLB( int s );

double round( double x );  /* standard math library function */

int  IsItATLB( int s, int as[], double at[], int ct, int ps, int l1ls );

/* BuildTLBOnlyPerm:
 *
 * Builds a reference stream that accesses one line per page and spreads
 * those line references across the sets of the lower level cache.
 *
 * Parameters: ArraySize, PageSize, LineSize 
 *             All three are measured in UnitSize chunks, e.g., sizeof(void)
 *             
 */

void BuildTLBOnlyPerm( int ArraySize,
		      int PageSize, /* TLB Page Size */
		      int LineSize )
{
  int i, j, col, ThisElt, LastElt, count;
  int NRows, NColumns;
  void **Old, **p;

  NRows = ArraySize /  PageSize;
  NColumns = PageSize / LineSize;

  if (Debug)
    fprintf(LogFile,
	    "TLB Perm: %d rows x %d columns x %d stride = %s (=? %s w).\n",
	    NRows,NColumns,LineSize,PrintNum(NRows * NColumns * LineSize),
	    PrintNum(ArraySize));

  if (NRows * NColumns * LineSize < ArraySize && NotYetComplained)
  {
    NotYetComplained = 0;
    fprintf(stderr,
	    "\nSpacing between points is less than page size.\n");
    fprintf(stderr,
	    "Duplicate points at low end. (Run w/-g option to see.)\n\n");
    fprintf(LogFile,
	    "\"\nSpacing between points is less than page size.\"\n");
    fprintf(LogFile,
	    "\"Duplicate points at low end. (Run w/-g option to see.\"\n\n");
  }

  if (LineSize >= PageSize)
  {
    fprintf(stderr,"BuildTLBOnlyPerm() receives invalid parameters, see log\n");
    fprintf(LogFile,"\nBuildTLBOnlyPerm: invalid parameters.\n");
    fprintf(LogFile,"Line size %04d, page size %04d\n",LineSize,PageSize);
    fprintf(LogFile,"Test makes no sense in this context.\n");
    (void) exit(-1);
  }

  ColumnIndexSet = PACE_AllocMem( NColumns * sizeof(int) );
  RowIndexSet    = PACE_AllocMem( NRows * sizeof(int) );

  /* Generate the index sets we need & shuffle them */
  (void) GenerateLinearSet( ColumnIndexSet, NColumns, LineSize );
  (void) Shuffle( ColumnIndexSet, NColumns );

  (void) GenerateLinearSet( RowIndexSet, NRows, PageSize );
  (void) Shuffle( RowIndexSet, NRows );

  /* Allocate the Memory Array */
  Old = MemArray;
  MemArray = (void **) PACE_AllocMem(ArraySize*UnitSize);
  if (Old != (void **) NULL)
    PACE_FreeMem(Old); 

  /* Assemble the permutation */
  LastElt = -1;
  j = 0;
  for (i=0;i<NRows;i++)
  {
    ThisElt = RowIndexSet[i] + ColumnIndexSet[j];
    if (LastElt != -1)
      MemArray[ThisElt] = &MemArray[LastElt];  
    if (Debug>1)
      fprintf(LogFile,"M[%s] <- %s.\t\t(%d,%d)\t%d + %d\n",
	      PrintNum(ThisElt),PrintNum(LastElt),j,i,
	      RowIndexSet[i],ColumnIndexSet[j]);
    LastElt = ThisElt;

    j++;
    if (j >= NColumns)
      j = 0;
  }
  /* and, finally, MemArray[0][0] <- MemArray[NRows-1][NCols-1] */
  StartingPoint = RowIndexSet[0] + ColumnIndexSet[0];
  MemArray[StartingPoint] = &MemArray[LastElt];

  /* Check the permutation */
  if (Debug>1)
    fprintf(LogFile,"M[%s] <- %s. ** starting point **\n",
	    PrintNum(StartingPoint),PrintNum(LastElt));

  /* verify permutation */
  p = MemArray[StartingPoint];
  i = 0;
  count = NRows;
  while (p != &MemArray[StartingPoint])
  {
    p = *p;
    i++;
    if (i > count)
    {
      fprintf(stderr,"Cycle did not return to starting point.\n");
      fprintf(stderr,"Cycle length is %s of %s.\n",
	      PrintNum(i),PrintNum(count));
      break;
    }
  }
  if ((Debug) && (i == count-1))
    fprintf(LogFile,"\"Maximal length permutation.\"\n");
  if (i < count-1)
    fprintf(LogFile,"\n\"Short permutation.\"\n");

  PACE_FreeMem(ColumnIndexSet);
  ColumnIndexSet = (int *) NULL;

  PACE_FreeMem(RowIndexSet);
  RowIndexSet = (int *) NULL;
}

/* TLBTest 
 *
 * Conducts a specialized test to look for a TLB
 *
 * PARAMETERS:  ArraySize, PageSize, LineSize, NA
 *
 * RETURNS:     An elapsed time, in microseconds, as a double
 *
 */

double TLBTest( int ArraySize, /* footprint for test, in WORDS            */
	        int PageSize,  
	        int LineSize,  
		struct AccessCount NA         /* number of iterations     */
	       )
{
  int i;
  double result;
  
  if (HeartBeat > 1)
    fprintf(stderr,"Trial @ %s b: ",PrintNum(ArraySize*UnitSize));

  /* Initialize MemArray */
   BuildTLBOnlyPerm(ArraySize,PageSize,LineSize); 

  /* Run the test */
   result = TimePermPtr(MemArray, StartingPoint, NA);

   PACE_FreeMem(MemArray);
   MemArray = (void **) NULL;

  if (HeartBeat > 1)
    fprintf(stderr,"%s usec\n",PrintDNum(result));

  return result;
}

/* TLBTrial
 *
 * PARAMETERS: PageSize, UB, LineSize
 *
 * Runs TLBTest() on from PageSize to UB, recording its results in 
 * the LogFile.
 *
 */

void TLBTrial ( int UB )
{
  int i, j, k, Count;
  double Trial;

  int    AllSizes[MAX_TESTS_PER_RUN], ExtractSizes[MAX_TESTS_PER_RUN];
  int    Counters[MAX_TESTS_PER_RUN], NotDone;
  double AllTimes[MAX_TESTS_PER_RUN], ExtractTimes[MAX_TESTS_PER_RUN];
  struct IntList *p, *q;

  NotYetComplained = 1;
  GlobalZero = 0;

  /* The TLB test & permutation only makes sense at integral multiples of
   * the PageSize; rather than write yet another generator over in Lib,
   * we generate the full set of eighth step sizes into ExtractSizes
   * and then eliminate sizes that do not make sense.
   */
  
  Count = GenerateEighthStepSet(ExtractSizes,MAX_TESTS_PER_RUN, PageSize, UB);

  j = 0;
  for (i=0; i< Count; i++)
  {
    if (PageSize * (ExtractSizes[i]/PageSize) == ExtractSizes[i])
    {
      AllSizes[j] = ExtractSizes[i];
      Counters[j] = TRIALS_FOR_MIN; /* set the counter for this entry */
      j++;
    }
  }
  Count = j;

  NAccesses.outer = 1;
  NAccesses.inner = 1000000;

  fprintf(LogFile,"\nTLB Test of %s points between %s b to %s b.\n",
	  PrintNum(Count), PrintNum(AllSizes[0]*UnitSize),
	  PrintNum(AllSizes[Count-1]*UnitSize));
  fprintf(LogFile,"Page Size %s b, Line Size of %s b.\n",
	  PrintNum(PageSize*UnitSize),PrintNum(L1LineSize*UnitSize));
  fprintf(LogFile,"( %s ; %s ) accesses.\n",
	  PrintNum(NAccesses.outer),PrintNum(NAccesses.inner));

  /* run the trial until every point has a "good" time */
  NotDone = 1;
  i = 1;
  k = 0;
  while(NotDone)
  {
    if (HeartBeat)
      fprintf(stderr,"Starting TLB Test series %2d.",i);

    NotDone = 0;
    for (j=0; j<Count;j++)
    { 
      if (Counters[j])
      {
	Trial = TLBTest(AllSizes[j],PageSize,L1LineSize,NAccesses);
	k++;
	if (i == 1)
	  AllTimes[j] = Trial;
	else if (Trial < AllTimes[j])
	{
	  AllTimes[j] = Trial;
	  Counters[j] = TRIALS_FOR_MIN;
	}
	else
	{
	  Counters[j]--;
	}
	if (Counters[j] > 0)
	  NotDone = 1;
      }
    }
    if (HeartBeat)
      fprintf(stderr," Tested %d points.\n",k);
    k = 0; i++;
  }
  fprintf(stderr,"\n");

  /* Write the data to the Log File */
  fprintf(LogFile,"\nSize\tTime\n");
  for (j=0; j<Count; j++)
    fprintf(LogFile,"%s\t%s\n",
	    PrintNum(AllSizes[j]*UnitSize),PrintDNum(AllTimes[j]));

  if (MemArray != (void**) NULL)
  {
    PACE_FreeMem(MemArray);
    MemArray = (void**) NULL;
  }

  /* PreCondition the data */
  if (PreCon)
  {
    (void) PreCondition(AllSizes, AllTimes, Count, 0.05 /* Epsilon */ );
    fprintf(LogFile,"\n\nPreconditioned Data\nSize\tTime\n");
    for (j=0; j<Count; j++)
      fprintf(LogFile,"%s\t%s\n",
	      PrintNum(AllSizes[j]*UnitSize),PrintDNum(AllTimes[j]));

  }
  else /* convert to cycles */
  {
    fprintf(LogFile,"\nRounding with AddCost %f.\n",AddCostInNSecs);
    for (i=0;i<Count;i++)
    {
      AllTimes[i] = AllTimes[i] * 1000.0 / (double) NAccesses.inner;
      fprintf(LogFile,"%12s\t1. %f\t",PrintNum(AllSizes[i]*UnitSize),AllTimes[i]);
      AllTimes[i] = AllTimes[i] / (double) NAccesses.outer;
      fprintf(LogFile,"2. %f\t3. %f",AllTimes[i],AllTimes[i]/AddCostInNSecs);
      AllTimes[i] = round(AllTimes[i] / AddCostInNSecs);
      fprintf(LogFile,"4. %f\n",AllTimes[i]);
    }
    fprintf(LogFile,"\n\nRounded Data\nSize\tCycles\n");
    for (i=0; i<0; i++)
      fprintf(LogFile,"%s\t%s\n",
	      PrintNum(AllSizes[i]*UnitSize),PrintDNum(AllTimes[i]));

  }

  /* Simple data analysis */
  /* First, find L1 TLB using the "SharpRiseTest" */

  i =  SharpRiseTest(AllSizes, AllTimes, Count, 0.10); /* preserve i */
  fprintf(stderr, "\nFound L1 TLB at %s b (%d Pages). <SharpRiseTest>\n",
	  PrintNum(i*UnitSize),i/PageSize);
  fprintf(LogFile,"\nFound L1 TLB at %s b. (%d Pages) <SharpRiseTest>\n",
	  PrintNum(i*UnitSize),i/PageSize);
  WriteResult("TLBF01S", i*UnitSize );

  /* report other L1 TLB Parameters */
  fprintf(LogFile,"Page size is %s. (from sysconf())\n",
	  PrintNum(PageSize*UnitSize));
  WriteResult("TLBFPS",PageSize*UnitSize);

  j = AssocTrial( i, L1LineSize, PageSize );
  WriteResult("TLBF01A",j);

  /* Now, examine the rest of the data with the inflection test */
  /* First, extract the remaining data */
  for (j=0; j<Count; j++)
  {
    if (AllSizes[j] == i)
      break;
  }

  k = 0;
  for (i=j;i<Count;i++)
  {
    ExtractSizes[k]   = AllSizes[i];
    ExtractTimes[k++] = AllTimes[i];
  }

  i = 2;
  p = LogInflectionTest( ExtractSizes, ExtractTimes, k, ConfirmTLB );
  while (p != (struct IntList *) NULL)
  {
    j = p->Val; 

    /* Test for false positives */
    if (IsItATLB( j, AllSizes, AllTimes, Count, PageSize, L1LineSize ) == 0)
      fprintf(stderr,"Point at %s b is a false positive.\n",
	      PrintNum(j*UnitSize));
    else
    {
      fprintf(stderr, "\nFound L%d TLB @ %s b (%d Pages). <LogInflectionTest>\n",
	      i,PrintNum(j*UnitSize),j/PageSize);
      fprintf(LogFile,"\nFound L%d TLB @ %s b (%d Pages). <LogInflectionTest>\n",
	      i,PrintNum(j*UnitSize),p->Val/PageSize);
      sprintf(buf,"TLBF%02dS",i);
      WriteResult(buf,j*UnitSize);
      j = AssocTrial( j, L1LineSize, PageSize );
      sprintf(buf,"TLBF%02dA",i);
      WriteResult(buf,j);
      i++;
    }
    p = p->Next; 
  }
}

/* Confirmation Routine, passed to LogInflectionTest() above;
 */

#define SIGNIFICANT 0.02

static int ConfirmTLB( int suspect )
{
  int    i, j, k, Count, NotDone, limit;
  int    Sizes[3], Counters[3];
  double Times[3], DT[3], Slope[3], Trial;

  if (HeartBeat>1)
    fprintf(stderr,"-> Trying to confirm TLB @ %s .\n",
	    PrintNum(suspect*UnitSize));

  i = suspect / PageSize;         /* number of pages in suspect size */
  limit = 96 * 1024 / (UnitSize*PageSize); /* don't want to reach past 96 KW  */


  /* build a short vector of test points, linearly spaced */
  if (i < 64)
  {
    Sizes[0] = suspect - 4 * PageSize;
    Sizes[1] = suspect;
    Sizes[2] = suspect + 4 * PageSize;
  }
  else 
  {
    i = Min(i * 3 / 8,limit);
    Sizes[0] = suspect - i * PageSize;  
    Sizes[1] = suspect;
    Sizes[2] = suspect + i * PageSize;
  }

  Counters[0] = TRIALS_FOR_MIN;
  Counters[1] = TRIALS_FOR_MIN;
  Counters[2] = TRIALS_FOR_MIN;

  Count = 3;
  NotDone = 1;
  i = 0;
  k = 0;
  while(NotDone)
  {
    NotDone = 0;
    k++;
    i++;

    for (j=0; j<Count;j++)
    { 
      if (Counters[j])
      {
	Trial = TLBTest(Sizes[j],PageSize,L1LineSize,NAccesses);
	if (i == 1)
	  Times[j] = Trial;
	else if (Trial < Times[j])
	{
	  Times[j] = Trial;
	  Counters[j] = TRIALS_FOR_MIN;
	}
	else
	  Counters[j]--;

	if (Counters[j])
	  NotDone = 1;
      }
    }
  }
  
  fprintf(LogFile,"\n\"  Confirming TLB Suspect @ %s b (%d rounds).\"\n",
	  PrintNum(suspect*UnitSize),k);

  fprintf(LogFile,"\"%12s\tTime\tDT\tSlope\n","Size\"");
  fprintf(LogFile,"%12s\t%s\n",
	  PrintNum(Sizes[0]*UnitSize),PrintDNum(Times[0]));

  for (i=1; i<Count; i++)
  {
    DT[i] = Times[i] - Times[i-1];
    Slope[i] = DT[i] / (double) (Sizes[i] - Sizes[i-1]);

    fprintf(LogFile,"%12s\t%s\t%.0f\t%f\n",
	    PrintNum(Sizes[i]*UnitSize),PrintDNum(Times[i]),DT[i],Slope[i]);
  }

  fprintf(LogFile,"\"Rise after suspect is %.2f pct of suspect time.\"\n",
	  100.0*(Times[2]-Times[1])/Times[1]);
  fprintf(stderr,"Rise after %s is %.2f pct of suspect time.\n",
	  PrintNum(Sizes[1]*UnitSize),100.0*(Times[2]-Times[1])/Times[1]);

  Verbose = 1;

  /* Break this out into specific cases. Trying to get a general
   * and concise formulat for the entire test proved to be 
   * confusing and hard to debug.
   */
  if (DT[2] < SIGNIFICANT * Times[1])
  {
    if (Verbose)
      fprintf(LogFile,"\"Change after suspect is insignificant (%.2f pct)\"\n",
	      100.0*DT[2]/Times[1]);
    return 0;
  }
  if (Slope[2] < 0)  /* decline in slope @ suspect point => false */
  {
    if (Verbose)
      fprintf(LogFile,"--> Negative slope after suspect point.\n");
    return 0;
  }
    
  if (Slope[2] < Slope[1])
  {
    if (Verbose)
      fprintf(LogFile,"--> Slope after suspect < before suspect.\n");
    return 0;
  }

  if (Slope[1] < 0.0 && Slope[2] > 0.0)
  {
    if (Verbose)
      fprintf(LogFile,"--> negative slope before suspect, positive after.\n");
  }

  if (Slope[2] > (10.0 * Slope[1]))
  {
    if (Verbose)
      fprintf(LogFile,"--> Order of magnitude change in slope @ suspect.\n");
    return 1;
  }  
  else if (Verbose)
    fprintf(LogFile,"\"--> Change in slope is only %.2f pct.\n",
	    100 * (Slope[2] - Slope[1]) / Slope[1]);

  return 0;   /* no reason to confirm */
}

