//KYR NOTE: This is the NEW, patched (via Alexandre Duret-Lutz's at.patch file) 

#include "automaton_tools.h"
#include "tgbaalgos/sccfilter.hh"

namespace mm {

/**
 * A simple visitor class that counts the number of states in the
 * automaton. Used in automaton_tools::count_states() below.
 */
class sba_counter_visitor : public spot::tgba_reachable_iterator_breadth_first {
public:
  
  sba_counter_visitor(const spot::tgba_sba_proxy* a,
		      int* state_count_container,
		      int* trans_count_container) :
    tgba_reachable_iterator_breadth_first(a),
    accept_all_(-1), 
    states_count(state_count_container),
    trans_count(trans_count_container)
  {
  }
  
  
  void
  start()
  {
    init_ = automata_->get_init_state();
  }

  void
  end()
  {
    if (accept_all_ != -1) {
      // Add one more state from where we accept anything
      (*states_count) = (*states_count) + 1;
    }
      delete init_;
  }

  bool
  state_is_accepting(const spot::state *s)
  {
    return
      dynamic_cast<const spot::tgba_sba_proxy*>(automata_)->state_is_accepting(s);
  }

  
  void
  process_state(const spot::state* s, int n, spot::tgba_succ_iterator*)
  {
    (*states_count) = (*states_count) + 1;
    
    spot::tgba_succ_iterator* it = automata_->succ_iter(s);
    it->first();
    if (! it->done())
      {
	spot::state* current = it->current_state();
	if (state_is_accepting(s)
	    && it->current_condition() == bddtrue
	    && s->compare(init_) != 0
	    && s->compare(current) == 0)
	  {
	    accept_all_ = n;
	  }
	delete current;
      }
    delete it;
  }

  void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si)
  {
    if (in != accept_all_)
      {
	(*trans_count) = (*trans_count) + 1;
      }
  }

private:
  int accept_all_;
  spot::state* init_;
  int* states_count;
  int* trans_count;
  
}; //Class




//////////////////////////////////////////
//
//  class automaton_tools
//
////////////////////////////////
const spot::tgba_sba_proxy*
automaton_tools::ltl2tgba(std::string ltl) {
  
  start_timer();
  bool debug_spot = 0;
  
  // Code fragments borrowed from SPOT's ltl2tgba.cc
  spot::ltl::environment& env(spot::ltl::default_environment::instance()); //look for env somewhere in SPOT's code
  dict = new spot::bdd_dict(); //SPOT:tgba/bdddict.cc
  
  spot::ltl::parse_error_list pel;
  spot::ltl::formula* f = spot::ltl::parse(ltl.c_str(), pel, env, debug_spot); //KYR: this looks like a call to SPOT's Bison parser for LTL in ltlparse/ltlparse.cc
  spot::ltl::format_parse_errors(std::cerr, ltl.c_str(), pel);
  
  //This is where the original code used to call prune_scc() (described in the paper)
  //Now we have another way to delete the states in empty(A)
  // which are the set of start states of the nondeterministic 
  // Buchi automaton that cannot lead to accepting runs
  spot::tgba* af = spot::ltl_to_tgba_fm(f, dict);
  f->destroy();
  
  reduced = spot::scc_filter(af, false);
  delete af;
  
  degeneralized = new spot::tgba_sba_proxy(reduced);

  // The degeneralized automaton is just a proxy, performing
  // degeneralization on the fly.  The following line duplicates the
  // automaton explicitly so we can account for the time it takes to
  // degeneralize.
  delete tgba_dupexp_dfs(degeneralized);

  display_and_reset_timer("SPOT");

#ifdef AUTOMATON_STATS
  int sba_states = 0;
  int sba_transitions = 0;
  automaton_tools::count_states_and_trans(degeneralized,
					  &sba_states, &sba_transitions);
  std::cout << "Number of states in SBA = " << sba_states << std::endl;
  std::cout << "Number of transitions in SBA = " << sba_transitions << std::endl;
#endif
  
  return degeneralized;
} //end ltl2tgba




void
automaton_tools::cleanup() {
  
  delete degeneralized;
//   std::cout << "Deleted the degeneralized " << std::endl;
  delete reduced;
  

  
  assert( spot::ltl::atomic_prop::instance_count() == 0 );
  assert( spot::ltl::unop::instance_count() == 0 );
  assert( spot::ltl::binop::instance_count() == 0 );
  assert( spot::ltl::multop::instance_count() == 0 );
  //std::cout << "About to delete the dictionary " << std::endl;
  ///delete dict;
}


void
automaton_tools::count_states_and_trans(const spot::tgba_sba_proxy* g, int* st, int* tr) {
  
  sba_counter_visitor cntr(g, st, tr);
  cntr.run();
}


void 
automaton_tools::generate_monitor_registration_obsolete(const sset_t clocks_set, 
					       std::ostream &os,
					       const std::string obs_name) {
  
  for (sset_t::iterator it = clocks_set.begin();
       it != clocks_set.end();
       ++it)
    {
      std::string clk = *it;
      if (strcasecmp(clk.c_str(), "DEFAULT_CLOCK") == 0) {
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_UPDATE_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_UPDATE_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_DELTA_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_INIT_DELTA_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_CYCLE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_CYCLE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_EVALUATION_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_EVALUATION_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_UPDATE_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_UPDATE_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_DELTA_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_TIMED_NOTIFY_PHASE_BEGIN);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_TIMED_NOTIFY_PHASE_END);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_METHOD_SUSPEND);" << std::endl;
	os << "    " << obs_name << "->register_monitor(this, MON_THREAD_SUSPEND);" << std::endl;
      }
  
      else {
	os << "    " << obs_name << "->register_monitor(this, " << clk << ");" << std::endl;
      }
    }
}


/**
 * Here we assume that @formula is quote-escaped, so that we can
 * insert it directly in the std::cout stream without messing up the
 * quotes in the file.
 */
void
automaton_tools::generate_mon_helpers_obsolete(std::ostream& os, const std::string formula) {
  os << "  void property_failed() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_FAIL_IMMEDIATELY" << std::endl;
  os << "    SC_REPORT_WARNING(\"Property failed\", \"Monitor for property " << formula << " has reached an error state\");" << std::endl;
  os << "    std::cout << \"Property failed after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    status = MON_FAIL;" << std::endl;    
  os << "  }" << std::endl << std::endl;    
  
  os << "  void property_satisfied() {" << std::endl;
  os << "#ifdef MONITOR_REPORT_PASS_IMMEDIATELY" << std::endl;
  os << "    std::cout << \"The property \" << to_string() << \"HOLDS\" << std::endl;" << std::endl;
  os << "    std::cout << \"Property cannot fail after \" << num_steps << \" steps\" << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    status = MON_PASS;" << std::endl;
  os << "  }" << std::endl << std::endl;
  
  os << "  const char* to_string() const {" << std::endl;
  os << "    return \""<< formula  << "\";" << std::endl;
  os << "  }" << std::endl << std::endl;
  
  os << "  const mon_status_t get_status() const {" << std::endl;
  os << "#ifdef PRINT_NUM_STEPS_IN_STATUS_CALL" << std::endl;
  os << "    std::cout << to_string() << \" made \" << num_steps << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    return status;" << std::endl;
  os << "  }" << std::endl << std::endl;
}





/**
 * This function generates the code for the local observer.
 */
//NEED TO MODIFY: OUTPUTS C++ code! 
//outputs header for the local observer file (header_os)
//local_observer is a class in the output C++ code: this function creates the constructor
//ouputs a monitor status report function to the standard output file (os)
//I don't think I need this function for my purposes; I think I can simply comment out the call to it
void
automaton_tools::generate_local_observer(std::ostream& os,
					 std::ostream& header_os,
					 global_params* gp) {

  header_os << std::endl << std::endl;
  header_os << "/**" << std::endl;
  header_os << " * This object is derived from mon_observer defined in the SystemC" << std::endl;
  header_os << " * modified kernel. Thus, we can pass it as an argument to the classes" << std::endl;
  header_os << " * implementing the monitors." << std::endl;
  header_os << " */" << std::endl;  
  header_os << "class local_observer : public sc_core::mon_observer {" << std::endl << std::endl;
  header_os << " public:" << std::endl << std::endl;
  header_os << "  // The constructor" << std::endl;
  header_os << "  local_observer(unsigned int verbosity, unsigned int num_monitors";

  // Determine what objects need to be passed to the observer
  std::stringstream observer_objects;
  for(ssmap_t::const_iterator it = gp->all_objects.begin();
      it != gp->all_objects.end();
      ++it)
    {
      observer_objects << ", " << it->second << " " << it->first;
    }

  header_os << observer_objects.str() << ");"
	    << std::endl << std::endl;

  // Now the implementation
  os << std::endl << std::endl;
  os << "// The constructor" << std::endl;
  os << "local_observer::local_observer(unsigned int verbosity, unsigned int num_monitors";
  os << observer_objects.str() << ") : sc_core::mon_observer(verbosity) {"
     << std::endl << std::endl;

  // Instantiate the monitors
  for(mpset_t::iterator it = gp->all_monitors.begin();
      it != gp->all_monitors.end();
      ++it)
    {
      monitor_params* mp = *it;

      std::string id;
      mp->get_id_str(&id);
      std::string this_monitor_name = gp->get_mon_name() + id;

      std::string this_monitor_inst = this_monitor_name + "_inst";

      // Allow multiple instances of the monitor, for testing purposes
      os << "  for (int i = 0; i < num_monitors; i++) {" << std::endl;
      
      os << "    " << this_monitor_name << "* " << this_monitor_inst
	 << " = new " << this_monitor_name << "(this";

      //Determine if this monitor needs any objects passed
      for(ssmap_t::const_iterator sit = mp->get_vartypes()->begin();
	  sit != mp->get_vartypes()->end();
	  ++sit)
	{
	  os << ", " << sit->first;
	}
      
      os << ");" << std::endl;
      
      os << "    mons.push_back(" << this_monitor_inst << ");" << std::endl;
      os << "  } // for loop" << std::endl;
    }
  os << "} // constructor" << std::endl << std::endl << std::endl;
  // Constructor ends here.

  header_os << "  void report_status(std::ostream& os);" << std::endl;
  
  os << "/**" << std::endl;
  os << " * Report the status of each monitor. Output to @os" << std::endl;
  os << " */" << std::endl;
  os << "void" << std::endl;
  os << "local_observer::report_status(std::ostream& os) {" << std::endl;
  os << "  mon_status_t status = MON_FAIL;" << std::endl;
  os << "  os << \"Monitor Status Report\" << std::endl;" << std::endl << std::endl;
  os << "  for (sc_core::vect_mon_t::const_iterator it = mons.begin();" << std::endl;
  os << "       it != mons.end();" << std::endl;
  os << "       it++)" << std::endl;
  os << "    {" << std::endl;
  os << "      status = (*it)->get_status();" << std::endl;
  os << "      os << \"Monitor \\t\" << (*it)->to_string() << std::endl;" << std::endl;
  os << "      os << \"Status: \\t\" << ((status == MON_FAIL)? \"FAILED\" : (status == MON_PASS) ? \"HOLDS\" : \"UNDETERMINED (DID NOT FAIL)\") << std::endl << std::endl;" << std::endl << std::endl;
  os << "    }" << std::endl;
  os << "} // report_status" << std::endl;

  
  header_os << "}; // class local_observer" << std::endl;
  
}


/**
 * Generates an LBT parser that the monitor can use to parse the input automaton.
 */
void
automaton_tools::generate_lbt_parser(std::ostream& os) {

  os << std::endl << std::endl;
  os << "#include <string>" << std::endl;
  os << "#include <iostream>" << std::endl;
  os << "#include <fstream>" << std::endl;
  os << "" << std::endl;
  os << "" << std::endl;
  os << "class lbt_parser {" << std::endl;
  os << "" << std::endl;
  os << "public:" << std::endl;
  os << "" << std::endl;
  os << "  /**" << std::endl;
  os << "   * Reads an integer from the file associated with this parser, and" << std::endl;
  os << "   * returns it." << std::endl;
  os << "   */" << std::endl;
  os << "  int" << std::endl;
  os << "  parse_int() {" << std::endl;
  os << "    std::string str;" << std::endl;
  os << "    *input_stream >> str;" << std::endl;
  os << "    return atoi(str.c_str());" << std::endl;
  os << "  }" << std::endl;
  os << "" << std::endl;
  os << "" << std::endl;
  os << "  /**" << std::endl;
  os << "   * Returns true if the file stream @in_strm has any integers left." << std::endl;
  os << "   */" << std::endl;
  os << "  bool" << std::endl;
  os << "  has_int() {" << std::endl;
  os << "" << std::endl;
  os << "    // Save where we are in the file in case we need to roll back" << std::endl;
  os << "    std::streampos current = input_stream->tellg();" << std::endl;
  os << "" << std::endl;
  os << "    std::string str;" << std::endl;
  os << "    *input_stream >> str;" << std::endl;
  os << "" << std::endl;
  os << "    if (str.length() == 0) {" << std::endl;
  os << "      return false;" << std::endl;
  os << "    }" << std::endl;
  os << "" << std::endl;
  os << "    // Undo and return" << std::endl;
  os << "    input_stream->seekg(current);" << std::endl;
  os << "    return true;" << std::endl;
  os << "  }" << std::endl;
  os << "" << std::endl;
  os << "" << std::endl;
  os << "  /**" << std::endl;
  os << "   * Parses a transition from the input file and stores it in the" << std::endl;
  os << "   * lookup table. Returns false if there are no more transitions for" << std::endl;
  os << "   * the current state, otherwise returns true." << std::endl;
  os << "   */" << std::endl;
  os << "  bool" << std::endl;
  os << "  parse_transition() {" << std::endl;
  os << "" << std::endl;
  os << "    int destination = parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    if (destination == -1)" << std::endl;
  os << "      return false;" << std::endl;
  os << "" << std::endl;
  os << "    int guard = parse_int();" << std::endl;
  os << "    transition_table[current_state][guard] = destination;" << std::endl;
  os << "" << std::endl;
  os << "#ifdef DEBUG_LBT_PARSER" << std::endl;
  os << "    std::cout << \"Transition \" << current_state << \" on \" << guard << \" --> \" << destination << std::endl;" << std::endl;
  os << "#endif" << std::endl;
  os << "    return true;" << std::endl;
  os << "  }" << std::endl;
  os << "    " << std::endl;
  os << "  " << std::endl;
  os << "" << std::endl;
  os << "  /**" << std::endl;
  os << "   * Parses one state and its associated transitions." << std::endl;
  os << "   */" << std::endl;
  os << "  void" << std::endl;
  os << "  parse_state() {" << std::endl;
  os << "    current_state = parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    // Is it initial?" << std::endl;
  os << "    parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    // Is it accepting?" << std::endl;
  os << "    int acceptance = parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    // If it is accepting, we need to consume the remaining -1 before" << std::endl;
  os << "    // we start parsing the transitions" << std::endl;
  os << "    if (acceptance == 0) {" << std::endl;
  os << "      parse_int();" << std::endl;
  os << "    }" << std::endl;
  os << "" << std::endl;
  os << "    // Loop until parse_transitions() returns false" << std::endl;
  os << "    while(parse_transition());" << std::endl;
  os << "  }" << std::endl;
  os << "  " << std::endl;
  os << "" << std::endl;
  os << "  /**" << std::endl;
  os << "   * Reads the automaton from the file indicated by @source, and" << std::endl;
  os << "   * returns a transition lookup table corresponding to the" << std::endl;
  os << "   * transitions of the automaton." << std::endl;
  os << "   */" << std::endl;
  os << "  int**" << std::endl;
  os << "  parse_to_table(const char* source, unsigned int alphabet_size) {" << std::endl;
  os << "" << std::endl;
  os << "    input_stream = new std::ifstream(source, std::ios::in);" << std::endl;
  os << "    " << std::endl;
  os << "    if (! input_stream ) {" << std::endl;
  os << "      std::cerr << \"Unable to pen \" << source << \" for input.\" << std::endl;" << std::endl;
  os << "      exit (1);" << std::endl;
  os << "    }" << std::endl;
  os << "" << std::endl;
  os << "    unsigned int automaton_num_states = parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    // Consume the number of acceptance sets from the input file, but" << std::endl;
  os << "    // we do not need to store them." << std::endl;
  os << "    parse_int();" << std::endl;
  os << "" << std::endl;
  os << "    transition_table = (int**) malloc(automaton_num_states * sizeof(int*));" << std::endl;
  os << "" << std::endl;
  os << "    for (unsigned i = 0; i < automaton_num_states; i++) {" << std::endl;
  os << "      transition_table[i] = (int*) malloc(alphabet_size * sizeof(int));" << std::endl;
  os << "" << std::endl;
  os << "      for (unsigned j = 0; j < alphabet_size; j++) {" << std::endl;
  os << "	transition_table[i][j] = -1;" << std::endl;
  os << "      }" << std::endl;
  os << "    }" << std::endl;
  os << "    " << std::endl;
  os << "    while (has_int()) {" << std::endl;
  os << "      parse_state();" << std::endl;
  os << "    }" << std::endl;
  os << "" << std::endl;
  os << "#ifdef DEBUG_LBT_PARSER" << std::endl;
  os << "    for (int i = 0; i < automaton_num_states; i++) {" << std::endl;
  os << "      for (int j = 0; j < alphabet_size; j++) {" << std::endl;
  os << "	std::cout <<  i << \", \" << j << \" --> \" << transition_table[i][j] << std::endl;" << std::endl;
  os << "      }" << std::endl;
  os << "    }" << std::endl;
  os << "#endif" << std::endl;
  os << "    " << std::endl;
  os << "    input_stream->close();" << std::endl;
  os << "    delete input_stream;" << std::endl;
  os << "    return transition_table;" << std::endl;
  os << "  }" << std::endl;
  os << "" << std::endl;
  os << "" << std::endl;
  os << "protected:" << std::endl;
  os << "" << std::endl;
  os << "  // Transition lookup table" << std::endl;
  os << "  int** transition_table;" << std::endl;
  os << "" << std::endl;
  os << "  // Source file from which we read the automaton" << std::endl;
  os << "  std::ifstream* input_stream;" << std::endl;
  os << "" << std::endl;
  os << "  // The state for which we are generating transitions" << std::endl;
  os << "  int current_state;  " << std::endl;
  os << "};" << std::endl;
  os << std::endl << std::endl;
}


} // namespace mm
