#ifndef BOOL_GLOBAL_H
#define BOOL_GLOBAL_H

#include <string>
#include <iostream>
#include <map>


#define AND_STRING    std::string(" and ")
#define OR_STRING     std::string(" or ")
#define NOT_STRING    std::string(" not ")
#define LPAREN_STRING std::string(" [ ")
#define RPAREN_STRING std::string(" ] ")
#define TRUE_STRING   std::string(" true ")
#define FALSE_STRING  std::string(" false ")


typedef enum {
  MAX_PREC   = 9,
  OR_PREC    = 5,
  AND_PREC   = 4,
  NOT_PREC   = 1
} op_prec_t;


// String to Boolean map
typedef std::map<std::string, bool> sbmap_t;


// Forward declaration
class bool_operator;



/***************************************************
 *
 *                    Bool Formula
 *
 ***************************************************/
class bool_formula {
  
 public:
  
  bool_formula(bool_operator* o = 0, bool_formula* f1 = 0, bool_formula* f2 = 0);
  ~bool_formula();
  std::string to_string(op_prec_t precedence = MAX_PREC);
  bool eval(sbmap_t* truth_assignment );
  static int unfreed_formulas(){ return number_of_formulas; }
  
  
 protected:
  bool_formula* left_formula;
  bool_formula* right_formula;
  bool_operator* op;
  static int number_of_formulas;
};


/***************************************************
 *
 *                    Bool Operator
 *
 ***************************************************/
class bool_operator {

 public:
  bool_operator();
  virtual ~bool_operator();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec) = 0;
  virtual std::string to_string() = 0;
  virtual op_prec_t get_precedence() = 0;
  virtual bool eval(bool_formula*, bool_formula*, sbmap_t*) = 0;
  static int unfreed_operators() {return number_of_operators; }
  
 protected:
  static int number_of_operators;
  
}; //class




class op_and : public bool_operator {
 public:
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);
  virtual std::string to_string() { return AND_STRING; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
};


class op_or : public bool_operator {
 public:
  // Constructor
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);
  virtual std::string to_string() { return OR_STRING; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
};


class op_not : public bool_operator {
 public:
  // Constructor
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);
  virtual std::string to_string() { return NOT_STRING; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
};



class op_atom : public bool_operator {
  
 public:
  op_atom(std::string* a);
  virtual ~op_atom();
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);
  virtual std::string to_string() { return "atom:" + *atom; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
  
 private: 
  std::string* atom;
};


class op_true : public bool_operator {
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);
  virtual std::string to_string() { return TRUE_STRING; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
};


class op_false : public bool_operator {
  virtual op_prec_t get_precedence();
  virtual std::string to_string(bool_formula* f1, bool_formula* f2, op_prec_t prec);  
  virtual std::string to_string() { return FALSE_STRING; };
  virtual bool eval(bool_formula* f1, bool_formula* f2, sbmap_t* truth_assignment);
};

#endif
