#include "conversions.h"
#include "ir_visitors/lbt_format.h"


/**
 * This (virtual) class implements a visitor to the automaton.It
 * handles all of the visitor function except for the transition
 * processing, which is implemented in the subclasses.
 */
class sba2ir_visitor : public spot::tgba_reachable_iterator_breadth_first {
public:
  
  sba2ir_visitor(const spot::tgba_sba_proxy* a, states_set_t* s) :
    tgba_reachable_iterator_breadth_first(a),
    accept_all_(-1),
    states(s)
  {
    accept_all_label = -999;
  }
  
  
  void
  start()
  {
    init_ = automata_->get_init_state();
  }

  void
  end()
  {
    delete init_;
    if (accept_all_ != -1) {
#ifdef DEBUG_CONVERSIONS
      std::cout << __FILE__ << ": sba2ir_visitor" << "::" << __func__ << "(): accept_all holds" << std::endl;
#endif
      // We have encountered a reachable state with a self loop on @true
      int state_id = accept_all_label;
      bool init = false;
      bool accepting = true;
      bool acc_all = true;
      state_t* aa_state = new state_t(state_id, init, accepting, acc_all);
      states->insert(aa_state);

      // We need to add outgoing transitions to this state!
      current_state = aa_state;
    }
  }

  bool
  state_is_accepting(const spot::state *s)
  {
    return
      dynamic_cast<const spot::tgba_sba_proxy*>(automata_)->state_is_accepting(s);
  }


 
  int get_state_label(const spot::state* s, int n)
  {
#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << ": " << __func__ << "(): the integer is " << n << std::endl;
#endif    
    int label;
    if (s->compare(init_) == 0)
      label = 0;
    else
      {
	int ns = n;

	if (state_is_accepting(s))
	  {
	    spot::tgba_succ_iterator* it = automata_->succ_iter(s);
	    it->first();
	    if (it->done())
	      label = ns;
	    else
	      {
		spot::state* current = it->current_state();
		if (it->current_condition() != bddtrue
		    || s->compare(current) != 0)
		  label = ns;
		else {
		  //label = "-999"; // accept_all
		  label = accept_all_label;
		}
		delete current;
	      }
	    delete it;
	  }
	else
	  label = ns;
      }
    return label;
  }


  void
  process_state(const spot::state* s, int n, spot::tgba_succ_iterator*)
  {
    
    spot::tgba_succ_iterator* it = automata_->succ_iter(s);
    it->first();
    
    if (it->done()) {
#ifdef DEBUG_CONVERSIONS
      std::cout << __FILE__ << ": " << __func__
		<< ": Adding a state without outgoing transitions " <<std::endl;
#endif
      int state_id = get_state_label(s, n);
      bool init = (s->compare(init_) == 0);
      current_state = new state_t(state_id, init, state_is_accepting(s));
      states->insert(current_state);
    }
    
    else {
      spot::state* current = it->current_state();
      if (state_is_accepting(s)
	  && it->current_condition() == bddtrue
	  && s->compare(init_) != 0
	  && s->compare(current) == 0) {
	
	accept_all_ = n;
	
#ifdef DEBUG_CONVERSIONS
	std::cout << __FILE__ << ": " << __func__
		  << ": Accept_all state found. Not adding to the list yet." << accept_all_ <<std::endl;
#endif

      } // if accept_all_ is found
      else
	{
	  int state_id = get_state_label(s, n);
	  bool init = (s->compare(init_) == 0);
	  current_state = new state_t(state_id, init, state_is_accepting(s));
	  states->insert(current_state);
#ifdef DEBUG_CONVERSIONS
	  std::cout << __FILE__ << ": " << __func__
		    << ": Added state with id " << state_id <<std::endl;
#endif
	}
      delete current;
    }

    delete it;
  }

  virtual void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si) = 0;
    

protected:
  int accept_all_;
  spot::state* init_;
  int accept_all_label;
  state_t* current_state;
  states_set_t* states;
  
}; //Class sba2ir_visitor



/**
 * This visitor produces an IR with Boolean functions on the
 * transitions.
 */
class sba2bool_ir_visitor : public sba2ir_visitor {
public:
  
  sba2bool_ir_visitor(const spot::tgba_sba_proxy* a, states_set_t* s) :
    sba2ir_visitor(a, s) {}
  

  void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si)
  {
    if (in != accept_all_)
      {
	const spot::ltl::formula* f = bdd_to_formula(si->current_condition(),
						     automata_->get_dict());
	
	std::string guard = to_systemc_string_paren_with_quotes(f);
	
	//KYR deprecated: destroy(f);
	f->destroy(); //KYR: replacement
	spot::state* current = si->current_state();
	int destination = get_state_label(current, out);
	
	transition_t* transition = new transition_t(guard, destination);
	current_state->add_outgoing_transition(transition);
#ifdef DEBUG_CONVERSIONS
	std::cout << __FILE__ << ": " << __func__
		  << ": Added transition to " << destination <<std::endl;
#endif	
	delete current;
      }
  }

  void end() {
    sba2ir_visitor::end();
    
    if (accept_all_ != -1) {
#ifdef DEBUG_CONVERSIONS      
      std::cout << __FILE__ << ": sba2bool_ir_visitor::" << __func__
		<< ": Dealing with an accept_all state!" << std::endl;
#endif
      assert(current_state->is_accept_all());

      // Self-loops on "true"
      int destination = current_state->get_state_id();
      std::string guard = "1";
      
      transition_t* transition = new transition_t(guard, destination);
      current_state->add_outgoing_transition(transition);
#ifdef DEBUG_CONVERSIONS
      std::cout << __FILE__ << ": " << __func__
		<< ": Added transition to " << destination <<std::endl;
#endif	
    }
  }


protected:
    
}; //Class sba2bool_ir_visitor






/**
 * Visits the SBA @a and constructs a set of states corresponding to
 * the states in @a. The transitions are encoded into integers using
 * the BDD uniqueness.
 */
class sba2symbolic_ir_visitor : public sba2ir_visitor {
  
public:
  // Constructor
  sba2symbolic_ir_visitor(const spot::tgba_sba_proxy* a,
			  states_set_t* s,
			  DICTIONARY_T* d):
    sba2ir_visitor(a,s),
    dictionary(d) {}

  
  void add_transitions(state_t* source_state,
		       bdd transition_bdd,
		       std::string guard,
		       int destination) {
    
    int transition_bdd_index = -1;
	
    // Have we seen this BDD yet?
    for (unsigned int i = 0; i < current_bdds.size(); i++) {
      if (transition_bdd == current_bdds[i]) {
	transition_bdd_index = i;
      }
    }
    
    if (transition_bdd_index == -1) {
      // New BDD
      current_bdds.push_back(transition_bdd);
      transition_bdd_index = current_bdds.size() - 1;
      (*dictionary)[transition_bdd_index] = guard;
    }

    // Convert the bdd_index to a string
    std::stringstream ss;
    ss << transition_bdd_index;
    guard = ss.str();


    transition_t* transition = new transition_t(guard, destination);
    source_state->add_outgoing_transition(transition);
#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << ": " << __func__ << ": Added transition to "
	      << destination << " on guard " << guard << std::endl;
#endif	
  }

  
  void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si)
  {
    if (in != accept_all_)
      {
	const spot::ltl::formula* f = bdd_to_formula(si->current_condition(),
						     automata_->get_dict());
	
	std::string guard = to_systemc_string_paren_with_quotes(f);
	//KYR deprecated: destroy(f);
	f->destroy(); //KYR: replacement

	bdd transition_bdd = si->current_condition();
	spot::state* current = si->current_state();
	int destination = get_state_label(current, out);

	
	add_transitions(current_state, transition_bdd, guard, destination);
	delete current;
      }
  }



  
  void end() {
    sba2ir_visitor::end();
    
    if (accept_all_ != -1) {
#ifdef DEBUG_CONVERSIONS      
      std::cout << __FILE__ << ": sba2symbolic_ir_visitor::" << __func__
		<< ": Dealing with an accept_all state!" << std::endl;
      
#endif
      assert(current_state->is_accept_all());

      // Self-loops on "true"
      int destination = current_state->get_state_id();
      std::string guard = "1";
      add_transitions(current_state, bddtrue, guard, destination);
    }
  }

  


protected:
  DICTIONARY_T* dictionary;
  std::vector<bdd> current_bdds;
  
  
}; //class sba2symbolic_ir_visitor





/**
 * Visits the SBA @a and constructs a set of states corresponding to
 * the states in @a. The transitions are encoded into integers using
 * the truth assignments that make the transition true.
  */
class sba2assign_ir_visitor : public sba2ir_visitor {
  
public:
  // Constructor
  sba2assign_ir_visitor(const spot::tgba_sba_proxy* a,
			  states_set_t* s,
			  v_sbmap_t* d):
    sba2ir_visitor(a,s),
    dictionary(d) {}
  

  void add_transitions(state_t* source_state, std::string guard, int destination) {
#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << "::" << __func__ << "(): calling trans2letters("
	      << guard << ", " << "var dictionary, var &letters)" << std::endl;
#endif
    std::set<int> letters;
    mm::alphabetize::trans2letters(&guard, dictionary, &letters);
    
    // Add a new transition for each integer in @letters
    for (std::set<int>::const_iterator cit = letters.begin();
	 cit != letters.end();
	 cit++) 
      {
	int letter = *cit;
	std::stringstream ss;
	ss << letter;
	std::string int_string = ss.str();
	
	transition_t* transition = new transition_t(int_string, destination);
	source_state->add_outgoing_transition(transition);
      }
	
#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << ": sba2assign_ir_visitor::" << __func__
	      << ": Added transition to " << destination << " on guard " << guard << std::endl;
#endif	
  }
  
  void
  process_link(const spot::state*, int in, const spot::state*, int out,
	       const spot::tgba_succ_iterator* si)
  {

#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << ": sba2assign_ir_visitor::" << __func__ << "(): "
	      << "Checking if I am a transition from accept_all state" << std::endl;
#endif
    if (in != accept_all_)
      {
	const spot::ltl::formula* f = bdd_to_formula(si->current_condition(),
						     automata_->get_dict());
	
	std::string guard = to_systemc_string_paren_with_quotes(f);
	//KYR deprecated: destroy(f);
	f->destroy(); //KYR: replacement
	
	spot::state* current = si->current_state();
	int destination = get_state_label(current, out);

	add_transitions(current_state, guard, destination);
#ifdef DEBUG_CONVERSIONS
    std::cout << __FILE__ << ": sba2assign_ir_visitor::" << __func__ << "(): "
	      << "Instatiated and added a transition to state " << destination
	      << " with guard " << guard << std::endl;
#endif	
	delete current;
      }
  }


  void end() {
    sba2ir_visitor::end();

    if (accept_all_ != -1) {
#ifdef DEBUG_CONVERSIONS      
      std::cout << __FILE__ << ": sba2assign_ir_visitor::" << __func__
		<< ": Dealing with an accept_all state!" << std::endl;
      
#endif
      assert(current_state->is_accept_all());

      // Self-loops on "true"
      int destination = current_state->get_state_id();
      std::string guard = "1";
      add_transitions(current_state, guard, destination);
    }
  }


protected:
  v_sbmap_t* dictionary;
  
}; //class sba2assign_ir_visitor














/**
 * Converts the State-Based Automaton @s into an IR automaton.
 * Transitions are stored as std::strings containing Boolean formulas.
 */
void
mm::conversions::sba2bool_ir(const spot::tgba_sba_proxy* s,
			     intermediate_rep* automaton) {
  sba2bool_ir_visitor v(s, automaton);
  v.run();
#ifdef DEBUG_CONVERSIONS
  std::cout << __FILE__ << ": " << __func__
	    << ": LBT version of the Intermediate Representation automaton" << std::endl;
  mm::ir_visitors::ir2lbt(automaton, std::cout);
#endif
  automaton->restore_invariants();
}



/**
 * Generates an SBA using SPOT. Then it creates the corresponding IR
 * automaton using the transition guards from SPOT.
 */
void
mm::conversions::mp2bool_ir(monitor_params* mp, intermediate_rep* result) {
  
  std::string ltl_string;
  mp->get_ltl(&ltl_string);

  // LTL -> SPOT -> IR
  automaton_tools at;
  const spot::tgba_sba_proxy* s = at.ltl2tgba(ltl_string);
  

#ifdef DEBUG_CONVERSIONS
  std::cout << __FILE__ << ": " << __func__ 
	    << "(): Neverclaim version of the SPOT automaton: " << std::endl;
  spot::never_claim_reachable(std::cout, s);
#endif
  
  sba2bool_ir(s, result);
}






/**
 * Converts the State-Based Automaton @s into an IR automaton using
 * the symbolic encoding of the transitions. The @dictionary stores
 * the translation indices from int to a Boolean formula--containing
 * std::string.
 */
void
mm::conversions::sba2symb_ir(const spot::tgba_sba_proxy* s,
			     intermediate_rep* automaton,
			     DICTIONARY_T* dictionary) {
  sba2symbolic_ir_visitor v(s, automaton, dictionary);
  v.run();
  automaton->restore_invariants();
  
#ifdef DEBUG_CONVERSIONS
  std::cout << __FILE__ << "::" << __func__ << ":" << std::endl
	    << "The dictionary follows" << std::endl;

  for (DICTIONARY_T::const_iterator it = dictionary->begin();
       it != dictionary->end();
       it ++)
    {
      int index = it->first;
      std::string guard = it->second;
      std::cout << index << " = " << guard << std::endl;
    }
  std::cout << ": LBT version of the Intermediate Representation automaton" << std::endl;
  mm::ir_visitors::ir2lbt(automaton, std::cout);
#endif
  


  
}





/**
 * Converts the State-Based Automaton @s into an IR automaton using
 * the assignment encoding of the transitions. The @dictionary stores
 * the assignment indices of all possible assignments to the atomic
 * propositions in the formula.
 */
void
mm::conversions::sba2assign_ir(const spot::tgba_sba_proxy* s,
			       intermediate_rep* automaton,
			       v_sbmap_t* dictionary) {
  
  sba2assign_ir_visitor v(s, automaton, dictionary);
  v.run();

  // At this point the IR automaton may have "accept_all" states,
  // which are flagged in the IR. However, if we are doing
  // minimization, this information will be lost. Thus, here we add
  // transitions on "true" from an "accept_all" state back to itself.
  
  
#ifdef DEBUG_CONVERSIONS
  std::cout << __FILE__ << ": " << __func__
	    << ": LBT version of the Intermediate Representation automaton" << std::endl;
  mm::ir_visitors::ir2lbt(automaton, std::cout);
#endif
  automaton->restore_invariants();
}




