#ifndef GLOBAL_PARAMS_H
#define GLOBAL_PARAMS_H

#include <iostream>
#include <cassert>
#include <fstream>
#include <string.h>

#include "types.h"
#include "monitor_params.h"
#include "utils.h"

typedef enum {
  FRONT_NONDET, 
  FRONT_DET_SWITCH,
  FRONT_DET_IFELSE,
  FRONT_DET_TABLE, //KYR
  FRONT_DET_EXPLICIT_TABLE,
  BACK_NONDET,
  BACK_ASS_ALPHA,
} encoding_t;

typedef enum {
  PARTIAL_REDUCTION,
  FULL_REDUCTION,
  NO_REDUCTION
} alpha_reduction_t;

typedef enum {
  ASSIGNMENTS,
  SYMBOLIC
} alphabetization_t;

typedef enum {
  BEFORE,
  AFTER
} befter_t;

typedef enum {
  ENTRY,
  EXIT,
  CALL,
  RETURN,
  DUMMY
} func_befter_t;

typedef struct FunctionLocation {
  std::string loc_name;
  std::string func_identifier;
  func_befter_t func_befter;
} func_loc_t;

typedef struct UserValue {
  std::string val_type;
  std::string val_name;
  std::string func_identifier;
  int param_index;
} user_val_t;

class global_params {
  
 public:

  // Constructor
  global_params(int argc, char** argv);
  
  void set_conf_file(char* fn) {  conf_file = std::string(fn); }
  void set_verbosity(unsigned int v) { verbosity = v; }
  unsigned int get_verbosity() { return verbosity; }
  void set_output_file(char* file_name) { output_file = std::string(file_name); }
  void set_header_output_file(char* file_name) {header_output_file = std::string(file_name);}
  void add_includefile(std::string fn) { includefiles.insert(fn); }
  void add_includefile(char* fn) { includefiles.insert(std::string(fn)); }
  sset_t get_includefiles() { return includefiles; }
  void includefiles_to_stream(std::ostream &os);
  void set_mon_name(std::string mn) {mon_name = mn;}
  const char* get_mon_name() {return mon_name.c_str();}
  void add_mp(monitor_params* mp) {all_monitors.insert(mp);}
  void add_vartype(std::string a, std::string b) {all_objects[a] = b;}
  const char* get_output_file() {return output_file.c_str(); }
  const char* get_header_output_file() {return header_output_file.c_str(); }  
  void to_stream(std::ostream &os);
  void set_print_spin(bool s) {print_spin = s; }
  bool get_print_spin() { return print_spin; }
  void set_encoding(const char* enc);
  void set_encoding(std::string enc);
  encoding_t get_encoding() { return encoding; }
  void set_alpha_reduction(char* ar);
  void set_alpha_reduction(std::string ar);
  alpha_reduction_t get_alpha_reduction() { return reduction_type; }
  void get_scratch_location(std::string* container) { *container = scratch; }
  void get_systemc_home(std::string* sysc ) { *sysc = systemc_home; }
  void get_brics_root(std::string* brics ) {brics->assign(brics_root); }
  void get_min_algo(std::string* algo) {algo->assign(min_algo); }
  bool get_minimize() { return minimize; }
  void set_alphabetization(const char* alpha);
  void set_alphabetization(std::string alpha);
  alphabetization_t get_alphabetization() { return alphabetization_method; }
  void add_location(std::string loc);
  void add_value(std::string val);
  std::vector<func_loc_t*>* get_func_locs() {return &func_locs;}
  std::vector<user_val_t*>* get_user_vals() {return &user_vals;}
  
  // A set of monitor_params objects. Each object contains the params
  // of an individual monitor
  mpset_t all_monitors;

  // A set of objects (and their types) that will need to be passed to
  // the local observer from the top level of the model
  ssmap_t all_objects;

  // A subset of the types from the set above, which have been defined
  // as a "user" types. We will need forward class declarations for
  // those.
  sset_t user_objects;

  void user_val_to_stream(user_val_t* uv, std::ostream& os = std::cout);
  
  // Pretty print of func_locs 
  void func_loc_to_stream(func_loc_t* loc, std::ostream& os = std::cout);
  

 protected:
  
  // Parse data from the configuration file
  void parse_config_file(unsigned int);

  // Parse data from the command line
  void parse_command_line(int, char**);

  // Print help and exit
  void help(char* prog_name);
  
  // Print the contents of the ssmap_t to the output stream os
  void ssmap_to_stream(ssmap_t my_map, std::ostream& os);
  
  // Print the contents of the mpset_t to the output stream os
  void mpset_to_stream(mpset_t my_set, std::ostream& os);

  // Extract and set the name of the output file
  void set_output_file(std::string name);

  // Extract and set the name of the output file
  void set_header_output_file(std::string name);

  // Pretty print the user values
  void user_vals_to_stream(std::ostream& os);
  
  // The location of the configuration file
  std::string conf_file;

  // How much talking we do. The lower, the quieter
  unsigned int verbosity;

  // The name to give to the class of generated monitors.
  std::string mon_name;

  // Files to be included in the monitor definition file
  sset_t includefiles;

  // Whether to print the SPIN version of the automaton
  bool print_spin;


  // The type of encoding for generating the automaton
  encoding_t encoding;

  // The file name where all monitors and other objects will go
  std::string output_file;

  // The header file for the monitors
  std::string header_output_file;

  // The method for reducing the alphabet of the automaton
  alpha_reduction_t reduction_type;

  // The "scratch" directory
  std::string scratch;

  // The location of the SystemC root directory
  std::string systemc_home;

  // The location of the Brics's automaton root directory
  std::string brics_root;

  // The minimization algorithm for Brics
  std::string min_algo;

  // If the automaton should be minimized prior to generating the montir
  bool minimize;

  // User-defined locations in the model code
  std::vector<func_loc_t*> func_locs;
 
  // User-defined values from the model code
  std::vector<user_val_t*> user_vals;

  

  const char* funcbefter2str(func_befter_t fb);
  void func_locs2stream(std::ostream& os = std::cout);

  // The method for alphabetizing the automaton
  alphabetization_t alphabetization_method;

  // Two maps for going between encoding type and the corresponding string
  struct classcomp {
    bool operator() (const char* lhs, const char* rhs) const
    {return strcmp(lhs, rhs) < 0;}
  };

  std::map<encoding_t, const char*> encoding2char;
  std::map<const char*, encoding_t, classcomp> char2encoding;

  // Initialize the dictionary maps
  void initialize_encoding_maps();

  // Sanity checks for the input parameters
  void sanity_checks();

  

}; // class

#endif
