#include "intermediate_rep.h"

// Initialize the static members
int 
transition_t::num_transitions = 0;

int
state_t::num_states = 0;



/**************************************************************
      **************************************************
           ****************************************
 
                     class transition_t

           ****************************************
      **************************************************
 **************************************************************/

/**
 * Constructor
 */
transition_t::transition_t(std::string g, int d, int s) {
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "Creating transition #" << num_transitions << " from "
	    << s << " on " << g << " to " << d << std::endl;
#endif
  source = s;
  guard = g;
  destination = d;
  num_transitions++;
}


/**
 * Pretty print
 */
std::string
transition_t::to_string() {
  char buff[1<<10];
  sprintf(buff, "source: %d, dest: %d, guard: %s", source, destination, guard.c_str());
  return std::string(buff);
}




/**************************************************************
      **************************************************
           ****************************************
 
                          class state_t

           ****************************************
      **************************************************
 **************************************************************/


/**
 * Constructor
 */
state_t::state_t(int id, bool acc) {
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "Creating state #" << num_states << ", id = "
	    << id << ", accepting = " << acc << std::endl;
#endif

  initialize(id, false, acc, false);
}

/**
 * Constructor
 */
state_t::state_t(int id, bool init, bool acc) {
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "Creating state #" << num_states << ", id = "
	    << id << ", initial = " << init << ", accepting = "
	    << acc << std::endl;
#endif

  initialize(id, init, acc, false);
}


/**
 * Constructor
 */
state_t::state_t(int id, bool init, bool acc, bool aa) {
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "Creating state #" << num_states << ", id = "
	    << id << ", initial = " << init << ", accepting = "
	    << acc << std::endl;
#endif

  // accept_all -> ! init
  assert(!acc || !init);

  // accept_all -> acc
  assert(!aa || acc);
  initialize(id, init, acc, aa);
}


void
state_t::initialize(int id, bool init, bool acc, bool aa) {
  state_id = id;
  initial = init;
  accepting = acc;
  accept_all = aa;
  num_states++;
}


/**
 * Destructor
 */
state_t::~state_t() {
  
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << __FILE__ << ": " << __func__
	    << ": Deleting the outgoing transitions " << std::endl;
#endif
  
  delete_transitions(outgoing);
  num_states --;
}


/**
 * Deletes all transitions in the set @trans_set
 */
void
state_t::delete_transitions(trans_set_t& trans_set) {
  
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << __FILE__ << ": " << __func__ << ": There are "
	    << trans_set.size() << " transitions in this set." << std::endl;
#endif
  for (trans_set_t::iterator it = trans_set.begin();
       it != trans_set.end();
       it ++)
    {
      transition_t* transition = *it;
      delete transition;
    }
}



void
state_t::add_outgoing_transition(transition_t* trans) {
  outgoing.insert(trans);
}


void
state_t::add_incoming_transition(transition_t* trans) {
  incoming.insert(trans);
}


/**
 * Ensures that every outgoing transition from this state is an
 * incoming transition for the destination state. Uses the map @istate
 * to find the state_t with required state_id's. Also ensures that
 * each transition has the proper source.
 */
void
state_t::propagate_outgoing(std::map<int, state_t*> istate) {
  for (trans_set_t::const_iterator it = outgoing.begin();
       it != outgoing.end();
       it++)
    {
      transition_t* trans = *it;
      int source = trans->get_source();
      assert (source == state_id || source == -1);
      if (source == -1) {
	trans->set_source(state_id);
      }

      istate[trans->get_dest()]->add_incoming_transition(trans);
    }
}


/**
 * Ensures that every outgoing transition from this state is an
 * incoming transition for the destination state. Uses the map @istate
 * to find the state_t with required state_id's. Also ensures that
 * each transition has the proper source.
 */
void
state_t::propagate_incoming(std::map<int, state_t*> istate) {
  for (trans_set_t::const_iterator it = incoming.begin();
       it != incoming.end();
       it++)
    {
      transition_t* trans = *it;
      int destination = trans->get_dest();
      assert (destination == state_id);
    
      istate[trans->get_source()]->add_outgoing_transition(trans);
    }
}



/**
 * Returns true if the state is deterministic. A state is
 * deterministic if there are no two outgoing transitions with the
 * same guards.
 */
bool
state_t::is_deterministic() {
  std::set<std::string> guards;
  std::pair< std::set<std::string>::iterator, bool > p;

  
  for (trans_set_t::const_iterator it = outgoing.begin();
       it != outgoing.end();
       it++)
    {
      transition_t* trans = *it;
      p = guards.insert(trans->get_guard());
      if (! p.second) {
	// Another element already existed in @guards
	return false;
      }
    }
  
  return true;
}



/**
 * Calls the visitor's state and transition functions
 */
void
state_t::accept(mm::ir_visitors::ir_visitor* visitor) {
  visitor->process_state(this);

  for (trans_set_t::const_iterator it = outgoing.begin();
       it != outgoing.end();
       it++) {
    transition_t* trans = *it;
    visitor->process_outgoing(trans);
  } //end for

  for (trans_set_t::const_iterator it = incoming.begin();
       it != incoming.end();
       it++) {
    transition_t* trans = *it;
    visitor->process_incoming(trans);
  } //end for
} //end accept


/**
 * Changes @state_id to @new_id and propagates the changes to the
 * transitions.
 */
void
state_t::rename_transitions(std::map<int, state_t*> istate) {
  for (trans_set_t::const_iterator it = outgoing.begin();
       it != outgoing.end();
       it++)
    {
      transition_t* trans = *it;
      trans->set_source(state_id);
      trans->set_dest(istate[trans->get_dest()]->get_state_id());
    }

  for (trans_set_t::const_iterator it = incoming.begin();
       it != incoming.end();
       it++)
    {
      transition_t* trans = *it;
      trans->set_dest(state_id);
      trans->reset_source(istate[trans->get_source()]->get_state_id());
    }
}





/**************************************************************
      **************************************************
           ****************************************
 
                    class intermediate_rep

           ****************************************
      **************************************************
 **************************************************************/


/**
 * Constructor
 */
intermediate_rep::intermediate_rep() {
  initial_state = 0;
  is_deterministic = false;
}


/**
 * Destructor
 */
intermediate_rep::~intermediate_rep() {
  for (intermediate_rep::const_iterator it = begin();
       it != end();
       it ++) 
    {
      state_t* state = *it;
#ifdef DEBUG_INTERMEDIATE_REP
      std::cout << __FILE__ << ": " << __func__
		<< ": Deleting state " << state->get_state_id() << std::endl;
#endif
      delete state;      
    }
}





/**
 * Read and return an integer from the stream @in_strm
 */
int
intermediate_rep::parse_int(std::ifstream& in_strm) {
  std::string str;
  in_strm >> str;

  return atoi(str.c_str());
}


/**
 * Returns true if the file stream @in_strm has any integers left.
 */
bool
intermediate_rep::has_int(std::ifstream& in_strm) {

  // Save where we are in the file in case we need to roll back
  std::streampos current = in_strm.tellg();

  std::string str;
  in_strm >> str;

  if (str.length() == 0) {
    return false;
  }

  // Undo and return
  in_strm.seekg(current);
  return true;
}




/**
 * Extracts a transition from the input stream @in_strm. Returns the
 * transition, or returns null in case there are no more transitions
 * for this state.
 */
transition_t*
intermediate_rep::parse_transition(std::ifstream& in_strm) {
 
  int destination = parse_int(in_strm);
  if (destination == -1) {
    return NULL;
  }
  
  std::string guard;
  getline(in_strm, guard);

  // Remove leading whitespace
  char const* delims = " \t\f\v\r\n";
  std::string::size_type notwhite = guard.find_first_not_of(delims);
  guard.erase(0,notwhite);

  // Remove trailing garbage
  notwhite = guard.find_last_not_of(delims);
  guard.erase(notwhite + 1);

#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "parse_transition(): destination = " << destination
	    << ", guard = " << guard << std::endl;
#endif
  
  return new transition_t(guard, destination);
}




/**
 * Parses a state from the the stream @in_strm and stores it in the
 * automaton
 */
void
intermediate_rep::parse_state(std::ifstream& in_strm) {
  
  int state_id = parse_int(in_strm);
  bool initial = parse_int(in_strm);
  int acc = parse_int(in_strm);
  state_t* state;

  // Expect non-generalized automata
  assert(acc == 0 || acc == -1);

  if (acc == -1) {
    state = new state_t(state_id, initial, false);
  }
  else { // acc == 0
    state = new state_t(state_id, initial, true);
    acc = parse_int(in_strm);
    assert(acc == -1);
  }
  
  insert(state);
#ifdef DEBUG_INTERMEDIATE_REP
  std::cout << "State created and inserted. Parsing a transition." << std::endl;
#endif

  transition_t* t = parse_transition(in_strm);
  while (t != NULL) {
    state->add_outgoing_transition(t);
    t = parse_transition(in_strm);
  }
}



/**
 * Parses an automaton in LBT format from the source file @source.
 */
void
intermediate_rep::parse_automaton(const char* source) {
  ///intermediate_rep* automaton = new intermediate_rep();
   
  std::ifstream in_strm( source, std::ios::in);

  if (! in_strm) {
    std::cerr << "ERROR: unable to read from source file" << source << std::endl;
    exit(1);
  }

  // Number of states; number of acceptance sets
  unsigned int automaton_num_states = parse_int(in_strm);
  int num_acc_sets = parse_int(in_strm);
     
  do {
    parse_state(in_strm);
  }while (has_int(in_strm));
  
  in_strm.close();

  assert(num_acc_sets == 1);
  assert(automaton_num_states == size());
  restore_invariants();
}



state_t*
intermediate_rep::get_initial_state() {
  if (! initial_state) {
    for (const_iterator it = begin();
	 it != end();
	 it ++) 
      {
	if ((*it)->is_initial()) {
	  initial_state = *it;
	  return initial_state;
	}
      }
  }
  else {
    assert(initial_state->is_initial());
    return initial_state;
  }

  // We can never reach this point
  assert(false);
  return initial_state;
}



/**
 * Sets the incoming transitions of each set so that each outgoing
 * transition in the automaton is an incoming transition for some
 * state. Then does the same for the incoming transitions.
 */
void
intermediate_rep::restore_transition_invariants() {

  // First build a map state_id -> state_t*. We do not use an array
  // here, because we do not know apriory the upper bound of the
  // state_id
  std::map<int, state_t*> istate;
  
  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      state_t* state = *it;
      istate[state->get_state_id()] = state;
    }

  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      (*it)->propagate_outgoing(istate);
    }

  // We should not do this in the same loop as the outgoing
  // transitions because we may have incomplete sets
  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      (*it)->propagate_incoming(istate);
    }
}



/**
 * Re-numbers the state_id's of the states so that each state_id is
 * within [0..(num_states - 1)].
 */
void
intermediate_rep::rename_states() {
  
  // First build a map of state_id -> state_t*. We do not use an array
  // here, because we do not know apriory the upper bound of the
  // state_id
  std::map<int, state_t*> istate;
  
  for (const_iterator it = begin();
       it != end();
       it ++)
    {
      state_t* state = *it;
      istate[state->get_state_id()] = state;
    }

  // First rename each state
  unsigned int new_id = 0;
  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      state_t* state = *it;
      state->set_state_id(new_id++);
    }
  // Max new_id before last ++ is (size() - 1)
  assert(new_id == size());

  // Then relabel the transitions
  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      state_t* state = *it;
      state->rename_transitions(istate);
    }
}


/**
 * Checks if the automaton is deterministic, and sets the appropriate
 * value of is_deterministic
 */
void
intermediate_rep::check_determinism() {
  is_deterministic = true;
  
  for (const_iterator it = begin();
       it != end();
       it ++) 
    {
      state_t* state = *it;
      is_deterministic = is_deterministic && state->is_deterministic();
    }
}




/**
 * Restores the following invariants: 0) Each state's state_id is
 * within [0..size()-1].  1) For each state, each outgoing transition
 * is an incoming transition in another state.  2) For each state,
 * each incoming transition is an outgoing transition from another
 * state.  3) The pointer to initial state is set correctly.  4) The
 * variable is_deterministic correctly describes the automaton.
 */
void
intermediate_rep::restore_invariants() {
  rename_states();
  restore_transition_invariants();
  get_initial_state();
  check_determinism();
}



/**
 * Checks if the automaton is deterministic, and sets the appropriate
 * value of is_deterministic
 */
//called by front_nondeterministic.cc:ir2fr_nondet()
void
intermediate_rep::accept(mm::ir_visitors::ir_visitor* visitor) {
  visitor->begin(this);
  
  for (const_iterator it = begin();
       it != end();
       it ++) {
      state_t* state = *it;
      state->accept(visitor); //intermediate_rep.cc
  } //end for
  visitor->end();
} //end accept


