#include "back_nondeterministic.h"

namespace mm {
  namespace ir_visitors {


/**
 * Constructor
 */ 
back_nondeterministic_visitor::back_nondeterministic_visitor(global_params* g,
							     monitor_params* m,
							     std::ostream& o,
							     std::ostream& h) :
  ir2monitor(g, m, o, h, true),
  semicolon_needed(false),
  accept_all_state_id(-1)
{
}

/**
 * Destructor
 */
back_nondeterministic_visitor::~back_nondeterministic_visitor() {
  // Do nothing
}


void
back_nondeterministic_visitor::begin(intermediate_rep* automaton) {
  
  num_states = automaton->size();
  monitor_header();
  monitor_constructor(automaton->get_initial_state()->get_state_id());
  step_header();
}

void
back_nondeterministic_visitor::process_state(state_t* state) {

  // See if the previous state had any transitions
  if (semicolon_needed) {
    os << ";" << std::endl;
  }
  
  os << "      next_state[" << state->get_state_id() << "] = ";
  semicolon_needed = false;
  
  if (state->is_accept_all()) {
    assert (accept_all_state_id == -1); // There can only be one
    accept_all_state_id = state->get_state_id();
  }

  if (state->get_num_incoming() == 0) {
    os << "0;" << std::endl;
  }
}

void
back_nondeterministic_visitor::process_outgoing(transition_t* trans) {
  // Chill out
}

void
back_nondeterministic_visitor::process_incoming(transition_t* trans) {
  std::string guard;
  trans->get_guard(&guard);

  if (semicolon_needed) {
    // This is not the first incoming transition
    os << " ||" << std::endl << "                      ";
  }
  os << "( current_state[" << trans->get_source() << "] && (";
  // Remove the quotes and print the guard to the stream
  remove_str(guard, "\"'", os);
  os << "))";
  semicolon_needed = true;
}

void
back_nondeterministic_visitor::end() {
  step_footer();
  observer_callbacks();
  helper_functions();
  start_declaring_monitor_class_variables();
  finish_declaring_monitor_class_variables();
}



/**
 * Generates the end of the code for the step() function.
 */
void
back_nondeterministic_visitor::step_footer() {
  // See if the previous state had any transitions
  if (semicolon_needed) {
    os << ";" << std::endl;
  }

  
  os << "    // Check if we are stuck... unless we already satisfied the property" << std::endl;
  os << "    bool not_stuck = (status == MON_PASS);" << std::endl;
  os << "    for (int i = 0; i < " << num_states << "; i++) {" << std::endl;
  os << "      not_stuck = not_stuck || next_state[i];" << std::endl;
  os << "    }" << std::endl << std::endl;
  os << "    if (! not_stuck) {" << std::endl;
  os << "      property_failed();" << std::endl;
  os << "    }" << std::endl;

  if (accept_all_state_id != -1) {
    // Do the "accept_all" short-circuit
    os << "    // From this state we accept anything, so we cannot get stuck" << std::endl;
    os << "    if( next_state[" << accept_all_state_id << "] ) {" << std::endl;
    os << "      assert(status != MON_FAIL);" << std::endl;
    os << "      property_satisfied();" << std::endl;
    os << "    }" << std::endl;
  }  
  os << "  } // if (status == MON_UNDETERMINED)" << std::endl;
  
  os << "#ifdef MONITOR_DEBUG" << std::endl;
  os << "  else {" << std::endl;
  os << "    std::cout << \"" << this_monitor_name << ": property has already been determined to \"" << std::endl; 
  os << "              << std::string((status == MON_PASS)? \"hold\" : \"fail\") << std::endl;" << std::endl;
  os << "  }" << std::endl;
  os << "#endif" << std::endl;    
  os << "} // step()" << std::endl;
}



void
ir2back_nondet(intermediate_rep* automaton,
	       global_params* global,
	       monitor_params* monitor,
	       std::ostream& output,
	       std::ostream& header_output) {
  back_nondeterministic_visitor visitor(global, monitor, output, header_output);
  automaton->accept(&visitor);
}

  } // namespace ir_visitors
} // namespace mm
