/**
 * Monitor master main.
 */

#include <iomanip>
#include <fstream>
#include <iostream>
#include <assert.h>


// Helpers
#include "global_params.h"
#include "monitor_params.h"
#include "types.h"
#include "utils.h"
#include "minimize.h"
#include "conversions.h"
#include "aspect.h"
#include "ir_visitors/monitor_generators.h"
#include "ir_visitors/ir_state_trans_counter.h"


void generate_monitors(global_params* gp,
		       std::ofstream& out_strm,
		       std::ofstream& out_strm_h) {

  //Print out the beginning headers of generated monitor code to the output file

  std::string capitalized; //the #ifdef version of the filename
  get_capitalized_name(gp->get_header_output_file(), & capitalized); //utils.cc:get_capitalized_name
  out_strm_h << "#ifndef " << capitalized << std::endl;              //#ifndef FILENAME
  out_strm_h << "#define " << capitalized << std::endl << std::endl; //#define FILENAME

  out_strm_h << "#include <systemc>" << std::endl << std::endl;
 
  out_strm_h << "#define DEBUG_USERVALS 0" << std::endl;
  out_strm_h << "#define DEBUG_USERLOCS 0" << std::endl;

  //Prints user's includes to the monitor's cpp file
  out_strm << "// Files to be included at the user's request (if any)" << std::endl;
  //format and print all includefiles to this cpp file
  gp->includefiles_to_stream(out_strm); //global_params.cc
  out_strm << std::endl << std::endl;

  std::string naked_name; //the plain file name, stripped of the path
  //return the file name without the leading path:
  get_naked_file_name(gp->get_header_output_file(), & naked_name); //utils.cc
  //include the .h file in the associated .cpp file
  out_strm << "#include \"" << naked_name << "\"" << std::endl << std::endl;
  
  //declare each user object as a class
  for (sset_t::const_iterator it = gp->user_objects.begin();
       it != gp->user_objects.end();
       it++) {
    out_strm_h << "class " << *it << "; // User object. Forward declaration" << std::endl;
  }

  // we close the #ifndef at the end of main()
  
  //Loop through all monitors
  for(mpset_t::const_iterator it = gp->all_monitors.begin();
      it != gp->all_monitors.end();
      ++it) { 
    //for each monitor...
    
    std::cout << "Starting to generate a monitor" << std::endl;
    monitor_params* mp = *it;
    encoding_t enc = gp->get_encoding(); //global_params.h: encoding_t get_encoding() { return encoding; }
    alpha_reduction_t alpha_reduction = gp->get_alpha_reduction(); 
    intermediate_rep* ir = new intermediate_rep(); //intermediate_rep.cc
    
    start_timer(); //utils.cc

    //ALPHABET MINIMIZATION STEP
    
    if (alpha_reduction != NO_REDUCTION) {
      mm::augment::augment_in_place(gp, mp); //augment.cc
    } //end if
    
    display_and_reset_timer("Alphabet reduction"); //utils.cc
    
    //STATE MINIMIZATION STEP

    if (gp->get_minimize()) {
      minimize(mp, gp, ir); //minimize.cc
    } //end if
    else {
      mm::conversions::mp2bool_ir(mp, ir); //conversions.cc
    } //end else
    
    display_and_reset_timer("LTL to (possibly minimized) IR time (including SPOT)");
    
    //optionally count states and transitions
#ifdef AUTOMATON_STATS
    int num_states = 0;
    int num_transitions = 0;
    mm::ir_visitors::count_ir_state_trans(ir, &num_states, &num_transitions);
    printf("Number of IR states: %d\n", num_states);
    printf("Number of IR transitions: %d\n", num_transitions);
#endif
    
    //MONITOR ENCODING STEP

    //call the associated function in ir_visitors/*:ir_visitors::ir2fr_* to add the automaton encoding
    switch (enc) {
    case FRONT_NONDET: //called front_nondet in paper
      //
      mm::ir_visitors::ir2fr_nondet(ir, gp, mp, out_strm, out_strm_h); //front_nondeterministic.cc
      break;
      
    case FRONT_DET_SWITCH: //called front_det_switch in paper
      mm::ir_visitors::ir2fr_det_ass_alpha_switch(ir, gp, mp, out_strm, out_strm_h); //front_det_assign_alphabetized_switch.cc
      break;

    case FRONT_DET_TABLE: //called front_det_table in (journal) paper
      mm::ir_visitors::ir2fr_det_ass_alpha_table(ir, gp, mp, out_strm, out_strm_h); //front_det_assign_alphabetized_table.cc
      break;

    case FRONT_DET_EXPLICIT_TABLE:
      mm::ir_visitors::ir2fr_det_ass_alpha_explicit_table(ir, gp, mp, out_strm, out_strm_h); 
      break;
      
    case FRONT_DET_IFELSE: //called front_det_ifelse in paper
      mm::ir_visitors::ir2fr_det_ass_alpha_ifelse(ir, gp, mp, out_strm, out_strm_h); //front_det_assign_alphabetized_ifelse.cc
      break;
      
    case BACK_NONDET: //called back_nondet in paper
      mm::ir_visitors::ir2back_nondet(ir, gp, mp, out_strm, out_strm_h); //back_nondeterministic.cc
      break;
      
    case BACK_ASS_ALPHA: //called back_det in paper?
      mm::ir_visitors::ir2back_ass_alpha(ir, gp, mp, out_strm, out_strm_h); //back_assign_alphabetized.cc
      break;
      
    default:
      std::cerr << "Unknown monitor encoding" << std::endl;
      exit(1);
    } //end switch

    //record the final time: the time to generate the monitor
    display_and_reset_timer("Monitor generating");
    
    delete ir;
    assert (state_t::unfreed_states() == 0);
    assert (transition_t::unfreed_transitions() == 0);
    
    std::cout << "Finished generating this monitor" << std::endl;
  } // for loop over all mp's 
} // generate_monitors()


//This is the main main function of the monitor master code.
//Outputs end of monitor file
int main(int argc, char** argv) {
#ifdef TIMING
  std::cout << "Clock ticks per second: " << sysconf(_SC_CLK_TCK) << std::endl;
#endif

  //parse command line inputs and set configuration
  global_params* gp = new global_params(argc, argv); //calls global_params.cc:global_params::global_params(int argc, char** argv)
  
  //print the values of all configuration parameters to cout
  gp->to_stream(std::cout); //global_params.cc:global_params::to_stream

  //open two output files, check for errors, and assign the file handles
  std::ofstream& out_strm = *open_for_output(gp->get_output_file()); //utils.cc:open_for_output
  //global_params.h:get_output_file returns the filename as a string
  std::ofstream& out_strm_h = *open_for_output(gp->get_header_output_file()); //utils.cc:open_for_output
  //global_params.h:get_output_file returns the filename as a string
  
  //the vast majority of the program is triggered by this next line
  generate_monitors(gp, out_strm, out_strm_h); //main.cc

  std::cout << "Done generating the monitor classes!" << std::endl;
  
  //Generate the local observer at the end 
  mm::automaton_tools::generate_local_observer(out_strm, out_strm_h, gp); //automaton_tools.cc

  //Generate an aspect file
  generate_aspect_file(gp); //aspect.cc

  //Generate parser class
  if (gp->get_encoding() == FRONT_DET_TABLE)
    mm::automaton_tools::generate_lbt_parser(out_strm_h);

  out_strm_h << std::endl << "#endif" << std::endl; //output end of monitor file
  close_file(&out_strm);  //utils.cc: close a file previously opened with open_for_output()
  close_file(&out_strm_h);//utils.cc: close a file previously opened with open_for_output()
  return 0;
} // main


